"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const Observable_1 = require("rxjs/Observable");
const exception_1 = require("../exception/exception");
const virtual_1 = require("../tree/virtual");
/**
 * When a rule or source returns an invalid value.
 */
class InvalidRuleResultException extends exception_1.BaseException {
    constructor(value) {
        let v = 'Unknown Type';
        if (value === undefined) {
            v = 'undefined';
        }
        else if (value === null) {
            v = 'null';
        }
        else if (typeof value == 'function') {
            v = `Function()`;
        }
        else if (typeof value != 'object') {
            v = `${typeof value}(${JSON.stringify(value)})`;
        }
        else {
            if (Object.getPrototypeOf(value) == Object) {
                v = `Object(${JSON.stringify(value)})`;
            }
            else if (value.constructor) {
                v = `Instance of class ${value.constructor.name}`;
            }
            else {
                v = 'Unknown Object';
            }
        }
        super(`Invalid rule or source result: ${v}.`);
    }
}
exports.InvalidRuleResultException = InvalidRuleResultException;
function callSource(source, context) {
    const result = source(context);
    if (result instanceof virtual_1.VirtualTree) {
        return Observable_1.Observable.of(result);
    }
    else if (result instanceof Observable_1.Observable) {
        return result;
    }
    else {
        throw new InvalidRuleResultException(result);
    }
}
exports.callSource = callSource;
function callRule(rule, input, context) {
    return input.mergeMap(i => {
        const result = rule(i, context);
        if (result instanceof virtual_1.VirtualTree) {
            return Observable_1.Observable.of(result);
        }
        else if (result instanceof Observable_1.Observable) {
            return result;
        }
        else {
            throw new InvalidRuleResultException(result);
        }
    });
}
exports.callRule = callRule;
//# sourceMappingURL=data:application/json;base64,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