"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const Observable_1 = require("rxjs/Observable");
require("rxjs/add/observable/concat");
require("rxjs/add/observable/empty");
require("rxjs/add/observable/merge");
require("rxjs/add/observable/of");
require("rxjs/add/operator/do");
require("rxjs/add/operator/map");
require("rxjs/add/operator/reduce");
const exception_1 = require("../exception/exception");
const update_buffer_1 = require("../utility/update-buffer");
const sink_1 = require("./sink");
class VirtualFileSystemSink extends sink_1.SimpleSinkBase {
    constructor(_host, _force = false) {
        super();
        this._host = _host;
        this._force = _force;
        this._filesToDelete = new Set();
        this._filesToRename = new Set();
        this._filesToCreate = new Map();
        this._filesToUpdate = new Map();
    }
    _validateCreateAction(action) {
        return this._force ? Observable_1.Observable.empty() : super._validateCreateAction(action);
    }
    _readFile(p) {
        const maybeCreate = this._filesToCreate.get(p);
        if (maybeCreate) {
            return Observable_1.Observable.of(maybeCreate);
        }
        const maybeUpdate = this._filesToUpdate.get(p);
        if (maybeUpdate) {
            return Observable_1.Observable.of(maybeUpdate);
        }
        throw new exception_1.FileDoesNotExistException(p);
    }
    _validateFileExists(p) {
        if (this._filesToCreate.has(p) || this._filesToUpdate.has(p)) {
            return Observable_1.Observable.of(true);
        }
        else if (this._filesToDelete.has(p)) {
            return Observable_1.Observable.of(false);
        }
        else {
            return this._host.exists(p);
        }
    }
    _overwriteFile(path, content) {
        this._filesToUpdate.set(path, new update_buffer_1.UpdateBuffer(content));
        return Observable_1.Observable.empty();
    }
    _createFile(path, content) {
        this._filesToCreate.set(path, new update_buffer_1.UpdateBuffer(content));
        return Observable_1.Observable.empty();
    }
    _renameFile(from, to) {
        this._filesToRename.add([from, to]);
        return this._readFile(from)
            .do(buffer => this._filesToCreate.set(to, buffer))
            .do(() => this._filesToDelete.add(from))
            .map(() => { });
    }
    _deleteFile(path) {
        if (this._filesToCreate.has(path)) {
            this._filesToCreate.delete(path);
            this._filesToUpdate.delete(path);
        }
        else {
            this._filesToDelete.add(path);
        }
        return Observable_1.Observable.empty();
    }
    _done() {
        // Really commit everything to the actual filesystem.
        return Observable_1.Observable.concat(Observable_1.Observable.from([...this._filesToDelete.values()])
            .concatMap(path => this._host.delete(path)), Observable_1.Observable.from([...this._filesToCreate.entries()])
            .concatMap(([path, buffer]) => this._host.write(path, buffer.generate())), Observable_1.Observable.from([...this._filesToRename.entries()])
            .concatMap(([_, [path, to]]) => this._host.rename(path, to)), Observable_1.Observable.from([...this._filesToUpdate.entries()])
            .concatMap(([path, buffer]) => this._host.write(path, buffer.generate()))).reduce(() => { });
    }
}
exports.VirtualFileSystemSink = VirtualFileSystemSink;
//# sourceMappingURL=data:application/json;base64,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