"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const exception_1 = require("../exception/exception");
const path_1 = require("../utility/path");
const action_1 = require("./action");
const entry_1 = require("./entry");
const interface_1 = require("./interface");
const recorder_1 = require("./recorder");
/**
 * The root class of most trees.
 */
class VirtualTree {
    constructor() {
        this._root = new Map();
        this._actions = new action_1.ActionList();
        this._cacheMap = new Map();
    }
    /**
     * Normalize the path. Made available to subclasses to overload.
     * @param path The path to normalize.
     * @returns {string} A path that is resolved and normalized.
     */
    _normalizePath(path) {
        return path_1.normalizePath(path);
    }
    /**
     * A list of file names contained by this Tree.
     * @returns {[string]} File paths.
     */
    get files() {
        return [...new Set([...this._root.keys(), ...this._cacheMap.keys()]).values()];
    }
    get root() {
        return new Map(this._root);
    }
    get staging() {
        return new Map(this._cacheMap);
    }
    get(path) {
        const normalizedPath = this._normalizePath(path);
        return this._cacheMap.get(normalizedPath) || this._root.get(normalizedPath) || null;
    }
    has(path) {
        return this.get(path) != null;
    }
    set(entry) {
        return this._cacheMap.set(entry.path, entry);
    }
    exists(path) {
        return this.has(path);
    }
    read(path) {
        const entry = this.get(path);
        return entry ? entry.content : null;
    }
    beginUpdate(path) {
        const entry = this.get(path);
        if (!entry) {
            throw new exception_1.FileDoesNotExistException(path);
        }
        return new recorder_1.UpdateRecorderBase(entry);
    }
    commitUpdate(record) {
        if (record instanceof recorder_1.UpdateRecorderBase) {
            const path = record.path;
            const entry = this.get(path);
            if (!entry) {
                throw new exception_1.ContentHasMutatedException(path);
            }
            else {
                const newContent = record.apply(entry.content);
                this.overwrite(path, newContent);
            }
        }
        else {
            throw new exception_1.InvalidUpdateRecordException();
        }
    }
    overwrite(path, content) {
        const normalizedTo = this._normalizePath(path);
        if (typeof content == 'string') {
            content = new Buffer(content, 'utf-8');
        }
        const maybeEntry = this.get(normalizedTo);
        if (maybeEntry && maybeEntry.content.equals(content)) {
            return;
        }
        this._overwrite(normalizedTo, content);
    }
    create(path, content) {
        const normalizedTo = this._normalizePath(path);
        if (typeof content == 'string') {
            content = new Buffer(content);
        }
        this._create(normalizedTo, content);
    }
    rename(path, to) {
        const normalizedPath = this._normalizePath(path);
        const normalizedTo = this._normalizePath(to);
        if (normalizedPath === normalizedTo) {
            // Nothing to do.
            return;
        }
        this._rename(normalizedPath, normalizedTo);
    }
    delete(path) {
        this._delete(this._normalizePath(path));
    }
    _overwrite(path, content, action) {
        if (!this.has(path)) {
            throw new exception_1.FileDoesNotExistException(path);
        }
        // Update the action buffer.
        if (action) {
            this._actions.push(action);
        }
        else {
            this._actions.overwrite(path, content);
        }
        this.set(new entry_1.SimpleFileEntry(path, content));
    }
    _create(path, content, action) {
        if (this._cacheMap.has(path)) {
            throw new exception_1.FileAlreadyExistException(path);
        }
        if (action) {
            this._actions.push(action);
        }
        else {
            this._actions.create(path, content);
        }
        this.set(new entry_1.SimpleFileEntry(path, content));
    }
    _rename(path, to, action, force = false) {
        const entry = this.get(path);
        if (!entry) {
            throw new exception_1.FileDoesNotExistException(path);
        }
        if (this._cacheMap.has(to) && !force) {
            throw new exception_1.FileAlreadyExistException(to);
        }
        if (action) {
            this._actions.push(action);
        }
        else {
            this._actions.rename(path, to);
        }
        this.set(new entry_1.SimpleFileEntry(to, entry.content));
        this._cacheMap.delete(path);
    }
    _delete(path, action) {
        if (!this.has(path)) {
            throw new exception_1.FileDoesNotExistException(path);
        }
        if (action) {
            this._actions.push(action);
        }
        else {
            this._actions.delete(path);
        }
        this._cacheMap.delete(path);
    }
    apply(action, strategy) {
        if (this._actions.has(action)) {
            return;
        }
        switch (action.kind) {
            case 'o':
                // Update the action buffer.
                this._overwrite(action.path, action.content, action);
                break;
            case 'c':
                if (this._cacheMap.has(action.path)) {
                    switch (strategy) {
                        case interface_1.MergeStrategy.Error: throw new exception_1.MergeConflictException(action.path);
                        case interface_1.MergeStrategy.Overwrite:
                            this._overwrite(action.path, action.content, action);
                            break;
                    }
                }
                else {
                    this._create(action.path, action.content, action);
                }
                break;
            case 'r':
                const force = (strategy & interface_1.MergeStrategy.AllowOverwriteConflict) != 0;
                this._rename(action.path, action.to, action, force);
                break;
            case 'd':
                this._delete(action.path, action);
                break;
            default: throw new action_1.UnknownActionException(action);
        }
    }
    // Returns an ordered list of Action to get this host.
    get actions() {
        return [...this._actions];
    }
    /**
     * Allow subclasses to copy to a tree their own properties.
     * @return {Tree}
     * @private
     */
    _copyTo(tree) {
        tree._root = new Map(this._root);
        this._actions.forEach(action => tree._actions.push(action));
        [...this._cacheMap.entries()].forEach(([path, entry]) => {
            tree._cacheMap.set(path, entry);
        });
    }
    branch() {
        const newTree = new VirtualTree();
        this._copyTo(newTree);
        return newTree;
    }
    // Creates a new host from 2 hosts.
    merge(other, strategy = interface_1.MergeStrategy.Default) {
        other.actions.forEach(action => this.apply(action, strategy));
    }
    optimize() {
        // This destroys the history. Hope you know what you're doing.
        this._actions.optimize();
    }
    static branch(tree) {
        return tree.branch();
    }
    static merge(tree, other, strategy = interface_1.MergeStrategy.Default) {
        const newTree = tree.branch();
        newTree.merge(other, strategy);
        return newTree;
    }
    static optimize(tree) {
        const newTree = tree.branch();
        newTree.optimize();
        return newTree;
    }
}
exports.VirtualTree = VirtualTree;
//# sourceMappingURL=data:application/json;base64,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