"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const core_1 = require("@angular-devkit/core");
const fs_1 = require("fs");
/**
 * Read a file and returns its content. This supports different file encoding.
 */
function readFile(fileName) {
    if (!fs_1.existsSync(fileName)) {
        return null;
    }
    const buffer = fs_1.readFileSync(fileName);
    let len = buffer.length;
    if (len >= 2 && buffer[0] === 0xFE && buffer[1] === 0xFF) {
        // Big endian UTF-16 byte order mark detected. Since big endian is not supported by node.js,
        // flip all byte pairs and treat as little endian.
        len &= ~1;
        for (let i = 0; i < len; i += 2) {
            const temp = buffer[i];
            buffer[i] = buffer[i + 1];
            buffer[i + 1] = temp;
        }
        return buffer.toString('utf16le', 2);
    }
    if (len >= 2 && buffer[0] === 0xFF && buffer[1] === 0xFE) {
        // Little endian UTF-16 byte order mark detected
        return buffer.toString('utf16le', 2);
    }
    if (len >= 3 && buffer[0] === 0xEF && buffer[1] === 0xBB && buffer[2] === 0xBF) {
        // UTF-8 byte order mark detected
        return buffer.toString('utf8', 3);
    }
    // Default is UTF-8 with no byte order mark
    return buffer.toString('utf8');
}
exports.readFile = readFile;
function readJsonFile(path) {
    return core_1.parseJson(readFile(path) || '{}', core_1.JsonParseMode.Loose);
}
exports.readJsonFile = readJsonFile;
//# sourceMappingURL=data:application/json;base64,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