"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const core_1 = require("@angular-devkit/core");
class UnknownTaskDependencyException extends core_1.BaseException {
    constructor(id) {
        super(`Unknown task dependency [ID: ${id.id}].`);
    }
}
exports.UnknownTaskDependencyException = UnknownTaskDependencyException;
class TaskScheduler {
    constructor(_context) {
        this._context = _context;
        this._queue = new core_1.PriorityQueue((x, y) => x.priority - y.priority);
        this._taskIds = new Map();
    }
    _calculatePriority(dependencies) {
        if (dependencies.size === 0) {
            return 0;
        }
        const prio = [...dependencies].reduce((prio, task) => prio + task.priority, 1);
        return prio;
    }
    _mapDependencies(dependencies) {
        if (!dependencies) {
            return new Set();
        }
        const tasks = dependencies.map(dep => {
            const task = this._taskIds.get(dep);
            if (!task) {
                throw new UnknownTaskDependencyException(dep);
            }
            return task;
        });
        return new Set(tasks);
    }
    schedule(taskConfiguration) {
        const dependencies = this._mapDependencies(taskConfiguration.dependencies);
        const priority = this._calculatePriority(dependencies);
        const task = {
            id: TaskScheduler._taskIdCounter++,
            priority,
            configuration: taskConfiguration,
            context: this._context,
        };
        this._queue.push(task);
        const id = { id: task.id };
        this._taskIds.set(id, task);
        return id;
    }
    finalize() {
        const tasks = this._queue.toArray();
        this._queue.clear();
        this._taskIds.clear();
        return tasks;
    }
}
TaskScheduler._taskIdCounter = 1;
exports.TaskScheduler = TaskScheduler;
//# sourceMappingURL=data:application/json;base64,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