"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const core_1 = require("@angular-devkit/core");
class UnknownActionException extends core_1.BaseException {
    constructor(action) { super(`Unknown action: "${action.kind}".`); }
}
exports.UnknownActionException = UnknownActionException;
let _id = 1;
class ActionList {
    constructor() {
        this._actions = [];
    }
    _action(action) {
        this._actions.push(Object.assign({
            id: _id++,
            parent: this._actions[this._actions.length - 1] || 0,
        }, action));
    }
    create(path, content) {
        this._action({ kind: 'c', path, content });
    }
    overwrite(path, content) {
        this._action({ kind: 'o', path, content });
    }
    rename(path, to) {
        this._action({ kind: 'r', path, to });
    }
    delete(path) {
        this._action({ kind: 'd', path });
    }
    optimize() {
        const toCreate = new Map();
        const toRename = new Map();
        const toOverwrite = new Map();
        const toDelete = new Set();
        for (const action of this._actions) {
            switch (action.kind) {
                case 'c':
                    toCreate.set(action.path, action.content);
                    break;
                case 'o':
                    if (toCreate.has(action.path)) {
                        toCreate.set(action.path, action.content);
                    }
                    else {
                        toOverwrite.set(action.path, action.content);
                    }
                    break;
                case 'd':
                    toDelete.add(action.path);
                    break;
                case 'r':
                    const maybeCreate = toCreate.get(action.path);
                    const maybeOverwrite = toOverwrite.get(action.path);
                    if (maybeCreate) {
                        toCreate.delete(action.path);
                        toCreate.set(action.to, maybeCreate);
                    }
                    if (maybeOverwrite) {
                        toOverwrite.delete(action.path);
                        toOverwrite.set(action.to, maybeOverwrite);
                    }
                    let maybeRename = undefined;
                    for (const [from, to] of toRename.entries()) {
                        if (to == action.path) {
                            maybeRename = from;
                            break;
                        }
                    }
                    if (maybeRename) {
                        toRename.set(maybeRename, action.to);
                    }
                    if (!maybeCreate && !maybeOverwrite && !maybeRename) {
                        toRename.set(action.path, action.to);
                    }
                    break;
            }
        }
        this._actions = [];
        toDelete.forEach(x => {
            this.delete(x);
        });
        toRename.forEach((to, from) => {
            this.rename(from, to);
        });
        toCreate.forEach((content, path) => {
            this.create(path, content);
        });
        toOverwrite.forEach((content, path) => {
            this.overwrite(path, content);
        });
    }
    push(action) { this._actions.push(action); }
    get(i) { return this._actions[i]; }
    has(action) {
        for (let i = 0; i < this._actions.length; i++) {
            const a = this._actions[i];
            if (a.id == action.id) {
                return true;
            }
            if (a.id > action.id) {
                return false;
            }
        }
        return false;
    }
    find(predicate) {
        return this._actions.find(predicate) || null;
    }
    forEach(fn, thisArg) {
        this._actions.forEach(fn, thisArg);
    }
    get length() { return this._actions.length; }
    [Symbol.iterator]() { return this._actions[Symbol.iterator](); }
}
exports.ActionList = ActionList;
function isContentAction(action) {
    return action.kind == 'c' || action.kind == 'o';
}
exports.isContentAction = isContentAction;
function isAction(action) {
    const kind = action && action.kind;
    return action !== null
        && typeof action.id == 'number'
        && typeof action.path == 'string'
        && (kind == 'c' || kind == 'o' || kind == 'r' || kind == 'd');
}
exports.isAction = isAction;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYWN0aW9uLmpzIiwic291cmNlUm9vdCI6Ii4vIiwic291cmNlcyI6WyJwYWNrYWdlcy9hbmd1bGFyX2RldmtpdC9zY2hlbWF0aWNzL3NyYy90cmVlL2FjdGlvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBOzs7Ozs7R0FNRztBQUNILCtDQUEyRDtBQUczRCw0QkFBb0MsU0FBUSxvQkFBYTtJQUN2RCxZQUFZLE1BQWMsSUFBSSxLQUFLLENBQUMsb0JBQW9CLE1BQU0sQ0FBQyxJQUFJLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztDQUM1RTtBQUZELHdEQUVDO0FBZ0JELElBQUksR0FBRyxHQUFHLENBQUMsQ0FBQztBQUVaO0lBQUE7UUFDVSxhQUFRLEdBQWEsRUFBRSxDQUFDO0lBdUhsQyxDQUFDO0lBckhXLE9BQU8sQ0FBQyxNQUF1QjtRQUN2QyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDO1lBQy9CLEVBQUUsRUFBRSxHQUFHLEVBQUU7WUFDVCxNQUFNLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsSUFBSSxDQUFDO1NBQ3JELEVBQUUsTUFBTSxDQUFXLENBQUMsQ0FBQztJQUN4QixDQUFDO0lBRUQsTUFBTSxDQUFDLElBQVUsRUFBRSxPQUFlO1FBQ2hDLElBQUksQ0FBQyxPQUFPLENBQUMsRUFBRSxJQUFJLEVBQUUsR0FBRyxFQUFFLElBQUksRUFBRSxPQUFPLEVBQUUsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFDRCxTQUFTLENBQUMsSUFBVSxFQUFFLE9BQWU7UUFDbkMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxFQUFFLElBQUksRUFBRSxHQUFHLEVBQUUsSUFBSSxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUNELE1BQU0sQ0FBQyxJQUFVLEVBQUUsRUFBUTtRQUN6QixJQUFJLENBQUMsT0FBTyxDQUFDLEVBQUUsSUFBSSxFQUFFLEdBQUcsRUFBRSxJQUFJLEVBQUUsRUFBRSxFQUFFLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBQ0QsTUFBTSxDQUFDLElBQVU7UUFDZixJQUFJLENBQUMsT0FBTyxDQUFDLEVBQUUsSUFBSSxFQUFFLEdBQUcsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO0lBQ3BDLENBQUM7SUFHRCxRQUFRO1FBQ04sTUFBTSxRQUFRLEdBQUcsSUFBSSxHQUFHLEVBQWdCLENBQUM7UUFDekMsTUFBTSxRQUFRLEdBQUcsSUFBSSxHQUFHLEVBQWMsQ0FBQztRQUN2QyxNQUFNLFdBQVcsR0FBRyxJQUFJLEdBQUcsRUFBZ0IsQ0FBQztRQUM1QyxNQUFNLFFBQVEsR0FBRyxJQUFJLEdBQUcsRUFBUSxDQUFDO1FBRWpDLEdBQUcsQ0FBQyxDQUFDLE1BQU0sTUFBTSxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO1lBQ25DLE1BQU0sQ0FBQyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO2dCQUNwQixLQUFLLEdBQUc7b0JBQ04sUUFBUSxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQztvQkFDMUMsS0FBSyxDQUFDO2dCQUVSLEtBQUssR0FBRztvQkFDTixFQUFFLENBQUMsQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7d0JBQzlCLFFBQVEsQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUM7b0JBQzVDLENBQUM7b0JBQUMsSUFBSSxDQUFDLENBQUM7d0JBQ04sV0FBVyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQztvQkFDL0MsQ0FBQztvQkFDRCxLQUFLLENBQUM7Z0JBRVIsS0FBSyxHQUFHO29CQUNOLFFBQVEsQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDO29CQUMxQixLQUFLLENBQUM7Z0JBRVIsS0FBSyxHQUFHO29CQUNOLE1BQU0sV0FBVyxHQUFHLFFBQVEsQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDO29CQUM5QyxNQUFNLGNBQWMsR0FBRyxXQUFXLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztvQkFDcEQsRUFBRSxDQUFDLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQzt3QkFDaEIsUUFBUSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUM7d0JBQzdCLFFBQVEsQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLEVBQUUsRUFBRSxXQUFXLENBQUMsQ0FBQztvQkFDdkMsQ0FBQztvQkFDRCxFQUFFLENBQUMsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDO3dCQUNuQixXQUFXLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQzt3QkFDaEMsV0FBVyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsRUFBRSxFQUFFLGNBQWMsQ0FBQyxDQUFDO29CQUM3QyxDQUFDO29CQUVELElBQUksV0FBVyxHQUFxQixTQUFTLENBQUM7b0JBQzlDLEdBQUcsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUFDLElBQUksUUFBUSxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUMsQ0FBQzt3QkFDNUMsRUFBRSxDQUFDLENBQUMsRUFBRSxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDOzRCQUN0QixXQUFXLEdBQUcsSUFBSSxDQUFDOzRCQUNuQixLQUFLLENBQUM7d0JBQ1IsQ0FBQztvQkFDSCxDQUFDO29CQUVELEVBQUUsQ0FBQyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUM7d0JBQ2hCLFFBQVEsQ0FBQyxHQUFHLENBQUMsV0FBVyxFQUFFLE1BQU0sQ0FBQyxFQUFFLENBQUMsQ0FBQztvQkFDdkMsQ0FBQztvQkFFRCxFQUFFLENBQUMsQ0FBQyxDQUFDLFdBQVcsSUFBSSxDQUFDLGNBQWMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUM7d0JBQ3BELFFBQVEsQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUM7b0JBQ3ZDLENBQUM7b0JBQ0QsS0FBSyxDQUFDO1lBQ1YsQ0FBQztRQUNILENBQUM7UUFFRCxJQUFJLENBQUMsUUFBUSxHQUFHLEVBQUUsQ0FBQztRQUNuQixRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQ25CLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDakIsQ0FBQyxDQUFDLENBQUM7UUFFSCxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsRUFBRSxFQUFFLElBQUksRUFBRSxFQUFFO1lBQzVCLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ3hCLENBQUMsQ0FBQyxDQUFDO1FBRUgsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLE9BQU8sRUFBRSxJQUFJLEVBQUUsRUFBRTtZQUNqQyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztRQUM3QixDQUFDLENBQUMsQ0FBQztRQUVILFdBQVcsQ0FBQyxPQUFPLENBQUMsQ0FBQyxPQUFPLEVBQUUsSUFBSSxFQUFFLEVBQUU7WUFDcEMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDaEMsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQsSUFBSSxDQUFDLE1BQWMsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDcEQsR0FBRyxDQUFDLENBQVMsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDM0MsR0FBRyxDQUFDLE1BQWM7UUFDaEIsR0FBRyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDO1lBQzlDLE1BQU0sQ0FBQyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDM0IsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsSUFBSSxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztnQkFDdEIsTUFBTSxDQUFDLElBQUksQ0FBQztZQUNkLENBQUM7WUFDRCxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxHQUFHLE1BQU0sQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO2dCQUNyQixNQUFNLENBQUMsS0FBSyxDQUFDO1lBQ2YsQ0FBQztRQUNILENBQUM7UUFFRCxNQUFNLENBQUMsS0FBSyxDQUFDO0lBQ2YsQ0FBQztJQUNELElBQUksQ0FBQyxTQUFxQztRQUN4QyxNQUFNLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksSUFBSSxDQUFDO0lBQy9DLENBQUM7SUFDRCxPQUFPLENBQUMsRUFBMkQsRUFBRSxPQUFZO1FBQy9FLElBQUksQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBQ0QsSUFBSSxNQUFNLEtBQUssTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztJQUM3QyxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxNQUFNLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7Q0FDakU7QUF4SEQsZ0NBd0hDO0FBR0QseUJBQWdDLE1BQWM7SUFDNUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxJQUFJLElBQUksR0FBRyxJQUFJLE1BQU0sQ0FBQyxJQUFJLElBQUksR0FBRyxDQUFDO0FBQ2xELENBQUM7QUFGRCwwQ0FFQztBQUdELGtCQUF5QixNQUFXO0lBQ2xDLE1BQU0sSUFBSSxHQUFHLE1BQU0sSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDO0lBRW5DLE1BQU0sQ0FBQyxNQUFNLEtBQUssSUFBSTtXQUNmLE9BQU8sTUFBTSxDQUFDLEVBQUUsSUFBSSxRQUFRO1dBQzVCLE9BQU8sTUFBTSxDQUFDLElBQUksSUFBSSxRQUFRO1dBQzlCLENBQUMsSUFBSSxJQUFJLEdBQUcsSUFBSSxJQUFJLElBQUksR0FBRyxJQUFJLElBQUksSUFBSSxHQUFHLElBQUksSUFBSSxJQUFJLEdBQUcsQ0FBQyxDQUFDO0FBQ3BFLENBQUM7QUFQRCw0QkFPQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQGxpY2Vuc2VcbiAqIENvcHlyaWdodCBHb29nbGUgSW5jLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqIFVzZSBvZiB0aGlzIHNvdXJjZSBjb2RlIGlzIGdvdmVybmVkIGJ5IGFuIE1JVC1zdHlsZSBsaWNlbnNlIHRoYXQgY2FuIGJlXG4gKiBmb3VuZCBpbiB0aGUgTElDRU5TRSBmaWxlIGF0IGh0dHBzOi8vYW5ndWxhci5pby9saWNlbnNlXG4gKi9cbmltcG9ydCB7IEJhc2VFeGNlcHRpb24sIFBhdGggfSBmcm9tICdAYW5ndWxhci1kZXZraXQvY29yZSc7XG5cblxuZXhwb3J0IGNsYXNzIFVua25vd25BY3Rpb25FeGNlcHRpb24gZXh0ZW5kcyBCYXNlRXhjZXB0aW9uIHtcbiAgY29uc3RydWN0b3IoYWN0aW9uOiBBY3Rpb24pIHsgc3VwZXIoYFVua25vd24gYWN0aW9uOiBcIiR7YWN0aW9uLmtpbmR9XCIuYCk7IH1cbn1cblxuXG5leHBvcnQgdHlwZSBBY3Rpb24gPSBDcmVhdGVGaWxlQWN0aW9uXG4gICAgICAgICAgICAgICAgICAgfCBPdmVyd3JpdGVGaWxlQWN0aW9uXG4gICAgICAgICAgICAgICAgICAgfCBSZW5hbWVGaWxlQWN0aW9uXG4gICAgICAgICAgICAgICAgICAgfCBEZWxldGVGaWxlQWN0aW9uO1xuXG5cbmV4cG9ydCBpbnRlcmZhY2UgQWN0aW9uQmFzZSB7XG4gIHJlYWRvbmx5IGlkOiBudW1iZXI7XG4gIHJlYWRvbmx5IHBhcmVudDogbnVtYmVyO1xuICByZWFkb25seSBwYXRoOiBQYXRoO1xufVxuXG5cbmxldCBfaWQgPSAxO1xuXG5leHBvcnQgY2xhc3MgQWN0aW9uTGlzdCBpbXBsZW1lbnRzIEl0ZXJhYmxlPEFjdGlvbj4ge1xuICBwcml2YXRlIF9hY3Rpb25zOiBBY3Rpb25bXSA9IFtdO1xuXG4gIHByb3RlY3RlZCBfYWN0aW9uKGFjdGlvbjogUGFydGlhbDxBY3Rpb24+KSB7XG4gICAgdGhpcy5fYWN0aW9ucy5wdXNoKE9iamVjdC5hc3NpZ24oe1xuICAgICAgaWQ6IF9pZCsrLFxuICAgICAgcGFyZW50OiB0aGlzLl9hY3Rpb25zW3RoaXMuX2FjdGlvbnMubGVuZ3RoIC0gMV0gfHwgMCxcbiAgICB9LCBhY3Rpb24pIGFzIEFjdGlvbik7XG4gIH1cblxuICBjcmVhdGUocGF0aDogUGF0aCwgY29udGVudDogQnVmZmVyKSB7XG4gICAgdGhpcy5fYWN0aW9uKHsga2luZDogJ2MnLCBwYXRoLCBjb250ZW50IH0pO1xuICB9XG4gIG92ZXJ3cml0ZShwYXRoOiBQYXRoLCBjb250ZW50OiBCdWZmZXIpIHtcbiAgICB0aGlzLl9hY3Rpb24oeyBraW5kOiAnbycsIHBhdGgsIGNvbnRlbnQgfSk7XG4gIH1cbiAgcmVuYW1lKHBhdGg6IFBhdGgsIHRvOiBQYXRoKSB7XG4gICAgdGhpcy5fYWN0aW9uKHsga2luZDogJ3InLCBwYXRoLCB0byB9KTtcbiAgfVxuICBkZWxldGUocGF0aDogUGF0aCkge1xuICAgIHRoaXMuX2FjdGlvbih7IGtpbmQ6ICdkJywgcGF0aCB9KTtcbiAgfVxuXG5cbiAgb3B0aW1pemUoKSB7XG4gICAgY29uc3QgdG9DcmVhdGUgPSBuZXcgTWFwPFBhdGgsIEJ1ZmZlcj4oKTtcbiAgICBjb25zdCB0b1JlbmFtZSA9IG5ldyBNYXA8UGF0aCwgUGF0aD4oKTtcbiAgICBjb25zdCB0b092ZXJ3cml0ZSA9IG5ldyBNYXA8UGF0aCwgQnVmZmVyPigpO1xuICAgIGNvbnN0IHRvRGVsZXRlID0gbmV3IFNldDxQYXRoPigpO1xuXG4gICAgZm9yIChjb25zdCBhY3Rpb24gb2YgdGhpcy5fYWN0aW9ucykge1xuICAgICAgc3dpdGNoIChhY3Rpb24ua2luZCkge1xuICAgICAgICBjYXNlICdjJzpcbiAgICAgICAgICB0b0NyZWF0ZS5zZXQoYWN0aW9uLnBhdGgsIGFjdGlvbi5jb250ZW50KTtcbiAgICAgICAgICBicmVhaztcblxuICAgICAgICBjYXNlICdvJzpcbiAgICAgICAgICBpZiAodG9DcmVhdGUuaGFzKGFjdGlvbi5wYXRoKSkge1xuICAgICAgICAgICAgdG9DcmVhdGUuc2V0KGFjdGlvbi5wYXRoLCBhY3Rpb24uY29udGVudCk7XG4gICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIHRvT3ZlcndyaXRlLnNldChhY3Rpb24ucGF0aCwgYWN0aW9uLmNvbnRlbnQpO1xuICAgICAgICAgIH1cbiAgICAgICAgICBicmVhaztcblxuICAgICAgICBjYXNlICdkJzpcbiAgICAgICAgICB0b0RlbGV0ZS5hZGQoYWN0aW9uLnBhdGgpO1xuICAgICAgICAgIGJyZWFrO1xuXG4gICAgICAgIGNhc2UgJ3InOlxuICAgICAgICAgIGNvbnN0IG1heWJlQ3JlYXRlID0gdG9DcmVhdGUuZ2V0KGFjdGlvbi5wYXRoKTtcbiAgICAgICAgICBjb25zdCBtYXliZU92ZXJ3cml0ZSA9IHRvT3ZlcndyaXRlLmdldChhY3Rpb24ucGF0aCk7XG4gICAgICAgICAgaWYgKG1heWJlQ3JlYXRlKSB7XG4gICAgICAgICAgICB0b0NyZWF0ZS5kZWxldGUoYWN0aW9uLnBhdGgpO1xuICAgICAgICAgICAgdG9DcmVhdGUuc2V0KGFjdGlvbi50bywgbWF5YmVDcmVhdGUpO1xuICAgICAgICAgIH1cbiAgICAgICAgICBpZiAobWF5YmVPdmVyd3JpdGUpIHtcbiAgICAgICAgICAgIHRvT3ZlcndyaXRlLmRlbGV0ZShhY3Rpb24ucGF0aCk7XG4gICAgICAgICAgICB0b092ZXJ3cml0ZS5zZXQoYWN0aW9uLnRvLCBtYXliZU92ZXJ3cml0ZSk7XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgbGV0IG1heWJlUmVuYW1lOiBQYXRoIHwgdW5kZWZpbmVkID0gdW5kZWZpbmVkO1xuICAgICAgICAgIGZvciAoY29uc3QgW2Zyb20sIHRvXSBvZiB0b1JlbmFtZS5lbnRyaWVzKCkpIHtcbiAgICAgICAgICAgIGlmICh0byA9PSBhY3Rpb24ucGF0aCkge1xuICAgICAgICAgICAgICBtYXliZVJlbmFtZSA9IGZyb207XG4gICAgICAgICAgICAgIGJyZWFrO1xuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cblxuICAgICAgICAgIGlmIChtYXliZVJlbmFtZSkge1xuICAgICAgICAgICAgdG9SZW5hbWUuc2V0KG1heWJlUmVuYW1lLCBhY3Rpb24udG8pO1xuICAgICAgICAgIH1cblxuICAgICAgICAgIGlmICghbWF5YmVDcmVhdGUgJiYgIW1heWJlT3ZlcndyaXRlICYmICFtYXliZVJlbmFtZSkge1xuICAgICAgICAgICAgdG9SZW5hbWUuc2V0KGFjdGlvbi5wYXRoLCBhY3Rpb24udG8pO1xuICAgICAgICAgIH1cbiAgICAgICAgICBicmVhaztcbiAgICAgIH1cbiAgICB9XG5cbiAgICB0aGlzLl9hY3Rpb25zID0gW107XG4gICAgdG9EZWxldGUuZm9yRWFjaCh4ID0+IHtcbiAgICAgIHRoaXMuZGVsZXRlKHgpO1xuICAgIH0pO1xuXG4gICAgdG9SZW5hbWUuZm9yRWFjaCgodG8sIGZyb20pID0+IHtcbiAgICAgIHRoaXMucmVuYW1lKGZyb20sIHRvKTtcbiAgICB9KTtcblxuICAgIHRvQ3JlYXRlLmZvckVhY2goKGNvbnRlbnQsIHBhdGgpID0+IHtcbiAgICAgIHRoaXMuY3JlYXRlKHBhdGgsIGNvbnRlbnQpO1xuICAgIH0pO1xuXG4gICAgdG9PdmVyd3JpdGUuZm9yRWFjaCgoY29udGVudCwgcGF0aCkgPT4ge1xuICAgICAgdGhpcy5vdmVyd3JpdGUocGF0aCwgY29udGVudCk7XG4gICAgfSk7XG4gIH1cblxuICBwdXNoKGFjdGlvbjogQWN0aW9uKSB7IHRoaXMuX2FjdGlvbnMucHVzaChhY3Rpb24pOyB9XG4gIGdldChpOiBudW1iZXIpIHsgcmV0dXJuIHRoaXMuX2FjdGlvbnNbaV07IH1cbiAgaGFzKGFjdGlvbjogQWN0aW9uKSB7XG4gICAgZm9yIChsZXQgaSA9IDA7IGkgPCB0aGlzLl9hY3Rpb25zLmxlbmd0aDsgaSsrKSB7XG4gICAgICBjb25zdCBhID0gdGhpcy5fYWN0aW9uc1tpXTtcbiAgICAgIGlmIChhLmlkID09IGFjdGlvbi5pZCkge1xuICAgICAgICByZXR1cm4gdHJ1ZTtcbiAgICAgIH1cbiAgICAgIGlmIChhLmlkID4gYWN0aW9uLmlkKSB7XG4gICAgICAgIHJldHVybiBmYWxzZTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICByZXR1cm4gZmFsc2U7XG4gIH1cbiAgZmluZChwcmVkaWNhdGU6ICh2YWx1ZTogQWN0aW9uKSA9PiBib29sZWFuKTogQWN0aW9uIHwgbnVsbCB7XG4gICAgcmV0dXJuIHRoaXMuX2FjdGlvbnMuZmluZChwcmVkaWNhdGUpIHx8IG51bGw7XG4gIH1cbiAgZm9yRWFjaChmbjogKHZhbHVlOiBBY3Rpb24sIGluZGV4OiBudW1iZXIsIGFycmF5OiBBY3Rpb25bXSkgPT4gdm9pZCwgdGhpc0FyZz86IHt9KSB7XG4gICAgdGhpcy5fYWN0aW9ucy5mb3JFYWNoKGZuLCB0aGlzQXJnKTtcbiAgfVxuICBnZXQgbGVuZ3RoKCkgeyByZXR1cm4gdGhpcy5fYWN0aW9ucy5sZW5ndGg7IH1cbiAgW1N5bWJvbC5pdGVyYXRvcl0oKSB7IHJldHVybiB0aGlzLl9hY3Rpb25zW1N5bWJvbC5pdGVyYXRvcl0oKTsgfVxufVxuXG5cbmV4cG9ydCBmdW5jdGlvbiBpc0NvbnRlbnRBY3Rpb24oYWN0aW9uOiBBY3Rpb24pOiBhY3Rpb24gaXMgQ3JlYXRlRmlsZUFjdGlvbiB8IE92ZXJ3cml0ZUZpbGVBY3Rpb24ge1xuICByZXR1cm4gYWN0aW9uLmtpbmQgPT0gJ2MnIHx8IGFjdGlvbi5raW5kID09ICdvJztcbn1cblxuXG5leHBvcnQgZnVuY3Rpb24gaXNBY3Rpb24oYWN0aW9uOiBhbnkpOiBhY3Rpb24gaXMgQWN0aW9uIHsgIC8vIHRzbGludDpkaXNhYmxlLWxpbmU6bm8tYW55XG4gIGNvbnN0IGtpbmQgPSBhY3Rpb24gJiYgYWN0aW9uLmtpbmQ7XG5cbiAgcmV0dXJuIGFjdGlvbiAhPT0gbnVsbFxuICAgICAgJiYgdHlwZW9mIGFjdGlvbi5pZCA9PSAnbnVtYmVyJ1xuICAgICAgJiYgdHlwZW9mIGFjdGlvbi5wYXRoID09ICdzdHJpbmcnXG4gICAgICAmJiAoa2luZCA9PSAnYycgfHwga2luZCA9PSAnbycgfHwga2luZCA9PSAncicgfHwga2luZCA9PSAnZCcpO1xufVxuXG5cbi8vIENyZWF0ZSBhIGZpbGUuIElmIHRoZSBmaWxlIGFscmVhZHkgZXhpc3RzIHRoZW4gdGhpcyBpcyBhbiBlcnJvci5cbmV4cG9ydCBpbnRlcmZhY2UgQ3JlYXRlRmlsZUFjdGlvbiBleHRlbmRzIEFjdGlvbkJhc2Uge1xuICByZWFkb25seSBraW5kOiAnYyc7XG4gIHJlYWRvbmx5IGNvbnRlbnQ6IEJ1ZmZlcjtcbn1cblxuLy8gT3ZlcndyaXRlIGEgZmlsZS4gSWYgdGhlIGZpbGUgZG9lcyBub3QgYWxyZWFkeSBleGlzdCwgdGhpcyBpcyBhbiBlcnJvci5cbmV4cG9ydCBpbnRlcmZhY2UgT3ZlcndyaXRlRmlsZUFjdGlvbiBleHRlbmRzIEFjdGlvbkJhc2Uge1xuICByZWFkb25seSBraW5kOiAnbyc7XG4gIHJlYWRvbmx5IGNvbnRlbnQ6IEJ1ZmZlcjtcbn1cblxuLy8gTW92ZSBhIGZpbGUgZnJvbSBvbmUgcGF0aCB0byBhbm90aGVyLiBJZiB0aGUgc291cmNlIGZpbGVzIGRvZXMgbm90IGV4aXN0LCB0aGlzIGlzIGFuIGVycm9yLlxuLy8gSWYgdGhlIHRhcmdldCBwYXRoIGFscmVhZHkgZXhpc3RzLCB0aGlzIGlzIGFuIGVycm9yLlxuZXhwb3J0IGludGVyZmFjZSBSZW5hbWVGaWxlQWN0aW9uIGV4dGVuZHMgQWN0aW9uQmFzZSB7XG4gIHJlYWRvbmx5IGtpbmQ6ICdyJztcbiAgcmVhZG9ubHkgdG86IFBhdGg7XG59XG5cbi8vIERlbGV0ZSBhIGZpbGUuIElmIHRoZSBmaWxlIGRvZXMgbm90IGV4aXN0LCB0aGlzIGlzIGFuIGVycm9yLlxuZXhwb3J0IGludGVyZmFjZSBEZWxldGVGaWxlQWN0aW9uIGV4dGVuZHMgQWN0aW9uQmFzZSB7XG4gIHJlYWRvbmx5IGtpbmQ6ICdkJztcbn1cbiJdfQ==