"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const core_1 = require("@angular-devkit/core");
const entry_1 = require("./entry");
const virtual_1 = require("./virtual");
class FileSystemDirEntry extends virtual_1.VirtualDirEntry {
    constructor(_host, tree, path = core_1.normalize('/')) {
        super(tree, path);
        this._host = _host;
    }
    _createDir(name) {
        return new FileSystemDirEntry(this._host, this._tree, core_1.join(this._path, name));
    }
    get parent() {
        return this._path == '/' ? null : this._tree.getDir(core_1.dirname(this._path));
    }
    get subdirs() {
        const result = new Set();
        try {
            this._host.list(this._path)
                .filter(name => this._host.isDirectory(core_1.join(this._path, name)))
                .forEach(name => result.add(core_1.fragment(name)));
        }
        catch (e) {
            if (e.code != 'ENOENT' && e.code != 'ENOTDIR') {
                throw e;
            }
        }
        for (const path of this._tree.staging.keys()) {
            if (path.startsWith(this._path) && core_1.dirname(path) != this._path) {
                result.add(core_1.basename(path));
            }
        }
        return [...result];
    }
    get subfiles() {
        const result = new Set();
        try {
            this._host.list(this._path)
                .filter(name => !this._host.isDirectory(core_1.join(this._path, name)))
                .forEach(name => result.add(core_1.fragment(name)));
        }
        catch (e) {
            if (e.code != 'ENOENT' && e.code != 'ENOTDIR') {
                throw e;
            }
        }
        for (const path of this._tree.staging.keys()) {
            if (path.startsWith(this._path) && core_1.dirname(path) == this._path) {
                result.add(core_1.basename(path));
            }
        }
        return [...result];
    }
    file(name) {
        return this._tree.get(core_1.join(this._path, name));
    }
}
exports.FileSystemDirEntry = FileSystemDirEntry;
class FileSystemTree extends virtual_1.VirtualTree {
    constructor(host) {
        super();
        this._initialized = false;
        this._host = new core_1.virtualFs.SyncDelegateHost(host);
        this._root = new FileSystemDirEntry(this._host, this);
    }
    get tree() {
        const host = this._host;
        if (!this._initialized) {
            this._initialized = true;
            this._recursiveFileList().forEach(path => {
                this._tree.set(path, new entry_1.LazyFileEntry(path, () => {
                    return new Buffer(host.read(path));
                }));
            });
        }
        return this._tree;
    }
    get(path) {
        const normalizedPath = this._normalizePath(path);
        let entry = this._cacheMap.get(normalizedPath) || this._tree.get(normalizedPath) || null;
        if (entry == null && !this._initialized) {
            const systemPath = normalizedPath;
            const fileExists = this._host.exists(systemPath) && !this._host.isDirectory(systemPath);
            if (fileExists) {
                const host = this._host;
                entry = new entry_1.LazyFileEntry(normalizedPath, () => new Buffer(host.read(systemPath)));
                this._tree.set(normalizedPath, entry);
            }
        }
        return entry;
    }
    branch() {
        const newTree = new FileSystemTree(this._host.delegate);
        this._copyTo(newTree);
        return newTree;
    }
    _copyTo(tree) {
        if (tree instanceof FileSystemTree) {
            const x = tree;
            x._tree = this._tree;
            x._initialized = this._initialized;
            this._actions.forEach(action => x._actions.push(action));
            [...this._cacheMap.entries()].forEach(([path, entry]) => {
                x._cacheMap.set(path, entry);
            });
        }
        else {
            super._copyTo(tree);
        }
    }
    _recursiveFileList() {
        const host = this._host;
        const list = [];
        function recurse(schematicPath) {
            for (const name of host.list(schematicPath)) {
                const normalizedPath = core_1.join(schematicPath, name);
                if (host.isDirectory(normalizedPath)) {
                    recurse(normalizedPath);
                }
                else {
                    list.push(normalizedPath);
                }
            }
        }
        recurse(core_1.normalize('/'));
        return list;
    }
}
exports.FileSystemTree = FileSystemTree;
class HostTree extends FileSystemTree {
}
exports.HostTree = HostTree;
class FileSystemCreateTree extends FileSystemTree {
    constructor(host) {
        super(host);
        this._recursiveFileList().forEach(path => {
            this.create(path, new Buffer(this._host.read(path)));
        });
        this._initialized = true;
    }
}
exports.FileSystemCreateTree = FileSystemCreateTree;
//# sourceMappingURL=data:application/json;base64,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