"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const core_1 = require("@angular-devkit/core");
const exception_1 = require("../exception/exception");
const action_1 = require("./action");
const entry_1 = require("./entry");
const interface_1 = require("./interface");
const recorder_1 = require("./recorder");
class VirtualDirEntry {
    constructor(_tree, _path = core_1.normalize('/')) {
        this._tree = _tree;
        this._path = _path;
        this._subdirs = new Map();
    }
    _createDir(name) {
        return new VirtualDirEntry(this._tree, core_1.join(this._path, name));
    }
    get parent() {
        return this._path == '/' ? null : this._tree.getDir(core_1.dirname(this._path));
    }
    get path() { return this._path; }
    get subdirs() {
        const directChildPartsCount = core_1.split(core_1.normalize(this._path)).length + 1;
        const directories = this._tree.files
            .filter(path => path.startsWith(this._path))
            .map(path => core_1.split(core_1.normalize(path)).slice(0, -1).slice(0, directChildPartsCount))
            .filter(parts => parts.length === directChildPartsCount)
            .map(parts => parts[parts.length - 1]);
        // make sure to have a unique set (directories contain multiple files so appear multiple times)
        return Array.from(new Set(directories));
    }
    get subfiles() {
        return this._tree.files
            .filter(path => core_1.dirname(path) === this._path)
            .map(path => core_1.basename(path));
    }
    dir(name) {
        let maybe = this._subdirs.get(name);
        if (!maybe) {
            this._subdirs.set(name, maybe = this._createDir(name));
        }
        return maybe;
    }
    file(name) {
        return this._tree.get(core_1.join(this._path, name));
    }
    visit(visitor) {
        function _recurse(entry) {
            entry.subfiles.forEach(path => {
                visitor(core_1.join(entry._path, path), entry.file(path));
            });
            entry.subdirs.forEach(path => {
                _recurse(entry.dir(path));
            });
        }
        try {
            _recurse(this);
        }
        catch (e) {
            if (e !== interface_1.FileVisitorCancelToken) {
                throw e;
            }
        }
    }
}
exports.VirtualDirEntry = VirtualDirEntry;
/**
 * The root class of most trees.
 */
class VirtualTree {
    constructor() {
        this._actions = new action_1.ActionList();
        this._cacheMap = new Map();
        this._root = new VirtualDirEntry(this);
        this._tree = new Map();
    }
    /**
     * Normalize the path. Made available to subclasses to overload.
     * @param path The path to normalize.
     * @returns {string} A path that is resolved and normalized.
     */
    _normalizePath(path) {
        return core_1.normalize('/' + path);
    }
    get tree() { return this._tree; }
    get staging() { return this._cacheMap; }
    [interface_1.TreeSymbol]() {
        return this;
    }
    /**
     * A list of file names contained by this Tree.
     * @returns {[string]} File paths.
     */
    get files() {
        return [...new Set([...this.tree.keys(), ...this._cacheMap.keys()]).values()];
    }
    get root() { return this._root; }
    get(path) {
        const normalizedPath = this._normalizePath(path);
        return this._cacheMap.get(normalizedPath) || this.tree.get(normalizedPath) || null;
    }
    has(path) {
        return this.get(path) != null;
    }
    set(entry) {
        return this._cacheMap.set(entry.path, entry);
    }
    exists(path) {
        return this.has(path);
    }
    read(path) {
        const entry = this.get(path);
        return entry ? entry.content : null;
    }
    getDir(path) {
        let dir = this.root;
        core_1.split(this._normalizePath(path)).slice(1).forEach(fragment => {
            dir = dir.dir(fragment);
        });
        return dir;
    }
    visit(visitor) {
        try {
            this.files.forEach(path => visitor(path, this.get(path)));
        }
        catch (e) {
            if (e !== interface_1.FileVisitorCancelToken) {
                throw e;
            }
        }
    }
    beginUpdate(path) {
        const entry = this.get(path);
        if (!entry) {
            throw new exception_1.FileDoesNotExistException(path);
        }
        return new recorder_1.UpdateRecorderBase(entry);
    }
    commitUpdate(record) {
        if (record instanceof recorder_1.UpdateRecorderBase) {
            const path = record.path;
            const entry = this.get(path);
            if (!entry) {
                throw new exception_1.ContentHasMutatedException(path);
            }
            else {
                const newContent = record.apply(entry.content);
                this.overwrite(path, newContent);
            }
        }
        else {
            throw new exception_1.InvalidUpdateRecordException();
        }
    }
    overwrite(path, content) {
        const normalizedTo = this._normalizePath(path);
        if (typeof content == 'string') {
            content = new Buffer(content, 'utf-8');
        }
        const maybeEntry = this.get(normalizedTo);
        if (maybeEntry && maybeEntry.content.equals(content)) {
            return;
        }
        this._overwrite(normalizedTo, content);
    }
    create(path, content) {
        const normalizedTo = this._normalizePath(path);
        if (typeof content == 'string') {
            content = new Buffer(content);
        }
        this._create(normalizedTo, content);
    }
    rename(path, to) {
        const normalizedPath = this._normalizePath(path);
        const normalizedTo = this._normalizePath(to);
        if (normalizedPath === normalizedTo) {
            // Nothing to do.
            return;
        }
        this._rename(normalizedPath, normalizedTo);
    }
    delete(path) {
        this._delete(this._normalizePath(path));
    }
    _overwrite(path, content, action) {
        if (!this.has(path)) {
            throw new exception_1.FileDoesNotExistException(path);
        }
        // Update the action buffer.
        if (action) {
            this._actions.push(action);
        }
        else {
            this._actions.overwrite(path, content);
        }
        this.set(new entry_1.SimpleFileEntry(path, content));
    }
    _create(path, content, action) {
        if (this._cacheMap.has(path)) {
            throw new exception_1.FileAlreadyExistException(path);
        }
        if (action) {
            this._actions.push(action);
        }
        else {
            this._actions.create(path, content);
        }
        this.set(new entry_1.SimpleFileEntry(path, content));
    }
    _rename(path, to, action, force = false) {
        const entry = this.get(path);
        if (!entry) {
            throw new exception_1.FileDoesNotExistException(path);
        }
        if (this._cacheMap.has(to) && !force) {
            throw new exception_1.FileAlreadyExistException(to);
        }
        if (action) {
            this._actions.push(action);
        }
        else {
            this._actions.rename(path, to);
        }
        this.set(new entry_1.SimpleFileEntry(to, entry.content));
        this._cacheMap.delete(path);
    }
    _delete(path, action) {
        if (!this.has(path)) {
            throw new exception_1.FileDoesNotExistException(path);
        }
        if (action) {
            this._actions.push(action);
        }
        else {
            this._actions.delete(path);
        }
        this._cacheMap.delete(path);
    }
    apply(action, strategy) {
        if (this._actions.has(action)) {
            return;
        }
        switch (action.kind) {
            case 'o':
                // Update the action buffer.
                this._overwrite(action.path, action.content, action);
                break;
            case 'c':
                if (this._cacheMap.has(action.path)) {
                    switch (strategy) {
                        case interface_1.MergeStrategy.Error: throw new exception_1.MergeConflictException(action.path);
                        case interface_1.MergeStrategy.Overwrite:
                            this._overwrite(action.path, action.content, action);
                            break;
                    }
                }
                else {
                    this._create(action.path, action.content, action);
                }
                break;
            case 'r':
                const force = (strategy & interface_1.MergeStrategy.AllowOverwriteConflict) != 0;
                this._rename(action.path, action.to, action, force);
                break;
            case 'd':
                this._delete(action.path, action);
                break;
            default: throw new action_1.UnknownActionException(action);
        }
    }
    // Returns an ordered list of Action to get this host.
    get actions() {
        return [...this._actions];
    }
    /**
     * Allow subclasses to copy to a tree their own properties.
     * @return {Tree}
     * @private
     */
    _copyTo(tree) {
        tree._tree = new Map(this.tree);
        this._actions.forEach(action => tree._actions.push(action));
        [...this._cacheMap.entries()].forEach(([path, entry]) => {
            tree._cacheMap.set(path, entry);
        });
    }
    branch() {
        const newTree = new VirtualTree();
        this._copyTo(newTree);
        return newTree;
    }
    // Creates a new host from 2 hosts.
    merge(other, strategy = interface_1.MergeStrategy.Default) {
        other.actions.forEach(action => this.apply(action, strategy));
    }
    optimize() {
        // This destroys the history. Hope you know what you're doing.
        this._actions.optimize();
    }
    static branch(tree) {
        return tree.branch();
    }
    static merge(tree, other, strategy = interface_1.MergeStrategy.Default) {
        const newTree = tree.branch();
        newTree.merge(other, strategy);
        return newTree;
    }
    static optimize(tree) {
        const newTree = tree.branch();
        newTree.optimize();
        return newTree;
    }
}
exports.VirtualTree = VirtualTree;
//# sourceMappingURL=data:application/json;base64,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