"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const node_1 = require("@angular-devkit/core/node");
const fs = require("fs");
const path = require("path");
const rxjs_1 = require("rxjs");
function _loadConfiguration(Configuration, options, root, file) {
    if (options.tslintConfig) {
        return Configuration.parseConfigFile(options.tslintConfig, root);
    }
    else if (options.tslintPath) {
        const tslintPath = path.join(root, options.tslintPath);
        return Configuration.findConfiguration(tslintPath, file && path.join(root, file)).results;
    }
    else {
        throw new Error('Executor must specify a tslint configuration.');
    }
}
function _getFileContent(file, options, program) {
    // The linter retrieves the SourceFile TS node directly if a program is used
    if (program) {
        const source = program.getSourceFile(file);
        if (!source) {
            const message = `File '${file}' is not part of the TypeScript project '${options.tsConfigPath}'.`;
            throw new Error(message);
        }
        return source.getFullText(source);
    }
    // NOTE: The tslint CLI checks for and excludes MPEG transport streams; this does not.
    try {
        // Strip BOM from file data.
        // https://stackoverflow.com/questions/24356713
        return fs.readFileSync(file, 'utf-8').replace(/^\uFEFF/, '');
    }
    catch (e) {
        throw new Error(`Could not read file '${file}'.`);
    }
}
function _listAllFiles(root) {
    const result = [];
    function _recurse(location) {
        const dir = fs.readdirSync(path.join(root, location));
        dir.forEach(name => {
            const loc = path.join(location, name);
            if (fs.statSync(path.join(root, loc)).isDirectory()) {
                _recurse(loc);
            }
            else {
                result.push(loc);
            }
        });
    }
    _recurse('');
    return result;
}
function default_1() {
    return (options, context) => {
        return new rxjs_1.Observable(obs => {
            const root = process.cwd();
            const tslint = require(node_1.resolve('tslint', {
                basedir: root,
                checkGlobal: true,
                checkLocal: true,
            }));
            const includes = (Array.isArray(options.includes)
                ? options.includes
                : (options.includes ? [options.includes] : []));
            const Linter = tslint.Linter;
            const Configuration = tslint.Configuration;
            let program = undefined;
            let filesToLint = [];
            if (options.tsConfigPath) {
                const tsConfigPath = path.join(process.cwd(), options.tsConfigPath);
                if (!fs.existsSync(tsConfigPath)) {
                    obs.error(new Error('Could not find tsconfig.'));
                    return;
                }
                program = Linter.createProgram(tsConfigPath);
                filesToLint = Linter.getFileNames(program);
            }
            if (includes.length > 0) {
                const allFilesRel = _listAllFiles(root);
                const pattern = '^('
                    + includes
                        .map(ex => '('
                        + ex.split(/[\/\\]/g).map(f => f
                            .replace(/[\-\[\]{}()+?.^$|]/g, '\\$&')
                            .replace(/^\*\*/g, '(.+?)?')
                            .replace(/\*/g, '[^/\\\\]*'))
                            .join('[\/\\\\]')
                        + ')')
                        .join('|')
                    + ')($|/|\\\\)';
                const re = new RegExp(pattern);
                filesToLint.push(...allFilesRel
                    .filter(x => re.test(x))
                    .map(x => path.join(root, x)));
            }
            const lintOptions = {
                fix: true,
                formatter: options.format || 'prose',
            };
            const linter = new Linter(lintOptions, program);
            const config = _loadConfiguration(Configuration, options, root);
            for (const file of filesToLint) {
                const content = _getFileContent(file, options, program);
                if (!content) {
                    continue;
                }
                linter.lint(file, content, config);
            }
            const result = linter.getResult();
            // Format and show the results.
            if (!options.silent) {
                const Formatter = tslint.findFormatter(options.format || 'prose');
                if (!Formatter) {
                    throw new Error(`Invalid lint format "${options.format}".`);
                }
                const formatter = new Formatter();
                const output = formatter.format(result.failures, result.fixes);
                if (output) {
                    context.logger.info(output);
                }
            }
            if (!options.ignoreErrors && result.errorCount > 0) {
                obs.error(new Error('Lint errors were found.'));
            }
            else {
                obs.next();
                obs.complete();
            }
        });
    };
}
exports.default = default_1;
//# sourceMappingURL=data:application/json;base64,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