"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const exception_1 = require("../exception/exception");
const update_buffer_1 = require("../utility/update-buffer");
class UpdateRecorderBase {
    constructor(entry) {
        this._original = new Buffer(entry.content);
        this._content = new update_buffer_1.UpdateBuffer(entry.content);
        this._path = entry.path;
    }
    static createFromFileEntry(entry) {
        const c0 = entry.content.readUInt8(0, true);
        const c1 = entry.content.readUInt8(1, true);
        const c2 = entry.content.readUInt8(2, true);
        // Check if we're BOM.
        if (c0 == 0xEF && c1 == 0xBB && c2 == 0xBF) {
            return new UpdateRecorderBom(entry);
        }
        else if (c0 === 0xFF && c1 == 0xFE) {
            return new UpdateRecorderBom(entry, 2);
        }
        else if (c0 === 0xFE && c1 == 0xFF) {
            return new UpdateRecorderBom(entry, 2);
        }
        return new UpdateRecorderBase(entry);
    }
    get path() { return this._path; }
    // These just record changes.
    insertLeft(index, content) {
        this._content.insertLeft(index, typeof content == 'string' ? new Buffer(content) : content);
        return this;
    }
    insertRight(index, content) {
        this._content.insertRight(index, typeof content == 'string' ? new Buffer(content) : content);
        return this;
    }
    remove(index, length) {
        this._content.remove(index, length);
        return this;
    }
    apply(content) {
        if (!content.equals(this._content.original)) {
            throw new exception_1.ContentHasMutatedException(this.path);
        }
        return this._content.generate();
    }
}
exports.UpdateRecorderBase = UpdateRecorderBase;
class UpdateRecorderBom extends UpdateRecorderBase {
    constructor(entry, _delta = 3) {
        super(entry);
        this._delta = _delta;
    }
    insertLeft(index, content) {
        return super.insertLeft(index + this._delta, content);
    }
    insertRight(index, content) {
        return super.insertRight(index + this._delta, content);
    }
    remove(index, length) {
        return super.remove(index + this._delta, length);
    }
}
exports.UpdateRecorderBom = UpdateRecorderBom;
//# sourceMappingURL=data:application/json;base64,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