"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const core_1 = require("@angular-devkit/core");
const exception_1 = require("../exception/exception");
const action_1 = require("./action");
const entry_1 = require("./entry");
const interface_1 = require("./interface");
const recorder_1 = require("./recorder");
class VirtualDirEntry {
    constructor(_tree, _path = core_1.normalize('/')) {
        this._tree = _tree;
        this._path = _path;
        this._subdirs = new Map();
    }
    _createDir(name) {
        return new VirtualDirEntry(this._tree, core_1.join(this._path, name));
    }
    get parent() {
        return this._path == '/' ? null : this._tree.getDir(core_1.dirname(this._path));
    }
    get path() { return this._path; }
    get subdirs() {
        const directChildPartsCount = core_1.split(core_1.normalize(this._path)).length + 1;
        const directories = this._tree.files
            .filter(path => path.startsWith(this._path))
            .map(path => core_1.split(core_1.normalize(path)).slice(0, -1).slice(0, directChildPartsCount))
            .filter(parts => parts.length === directChildPartsCount)
            .map(parts => parts[parts.length - 1]);
        // make sure to have a unique set (directories contain multiple files so appear multiple times)
        return Array.from(new Set(directories));
    }
    get subfiles() {
        return this._tree.files
            .filter(path => core_1.dirname(path) === this._path)
            .map(path => core_1.basename(path));
    }
    dir(name) {
        let maybe = this._subdirs.get(name);
        if (!maybe) {
            this._subdirs.set(name, maybe = this._createDir(name));
        }
        return maybe;
    }
    file(name) {
        return this._tree.get(core_1.join(this._path, name));
    }
    visit(visitor) {
        function _recurse(entry) {
            entry.subfiles.forEach(path => {
                visitor(core_1.join(entry._path, path), entry.file(path));
            });
            entry.subdirs.forEach(path => {
                _recurse(entry.dir(path));
            });
        }
        try {
            _recurse(this);
        }
        catch (e) {
            if (e !== interface_1.FileVisitorCancelToken) {
                throw e;
            }
        }
    }
}
exports.VirtualDirEntry = VirtualDirEntry;
/**
 * The root class of most trees.
 */
class VirtualTree {
    constructor() {
        this._actions = new action_1.ActionList();
        this._cacheMap = new Map();
        this._root = new VirtualDirEntry(this);
        this._tree = new Map();
    }
    /**
     * Normalize the path. Made available to subclasses to overload.
     * @param path The path to normalize.
     * @returns {string} A path that is resolved and normalized.
     */
    _normalizePath(path) {
        return core_1.normalize('/' + path);
    }
    get tree() { return this._tree; }
    get staging() { return this._cacheMap; }
    [interface_1.TreeSymbol]() {
        return this;
    }
    /**
     * A list of file names contained by this Tree.
     * @returns {[string]} File paths.
     */
    get files() {
        return [...new Set([...this.tree.keys(), ...this._cacheMap.keys()]).values()];
    }
    get root() { return this._root; }
    get(path) {
        const normalizedPath = this._normalizePath(path);
        return this._cacheMap.get(normalizedPath) || this.tree.get(normalizedPath) || null;
    }
    has(path) {
        return this.get(path) != null;
    }
    set(entry) {
        return this._cacheMap.set(entry.path, entry);
    }
    exists(path) {
        return this.has(path);
    }
    read(path) {
        const entry = this.get(path);
        return entry ? entry.content : null;
    }
    getDir(path) {
        let dir = this.root;
        core_1.split(this._normalizePath(path)).slice(1).forEach(fragment => {
            dir = dir.dir(fragment);
        });
        return dir;
    }
    visit(visitor) {
        try {
            this.files.forEach(path => visitor(path, this.get(path)));
        }
        catch (e) {
            if (e !== interface_1.FileVisitorCancelToken) {
                throw e;
            }
        }
    }
    beginUpdate(path) {
        const entry = this.get(path);
        if (!entry) {
            throw new exception_1.FileDoesNotExistException(path);
        }
        return recorder_1.UpdateRecorderBase.createFromFileEntry(entry);
    }
    commitUpdate(record) {
        if (record instanceof recorder_1.UpdateRecorderBase) {
            const path = record.path;
            const entry = this.get(path);
            if (!entry) {
                throw new exception_1.ContentHasMutatedException(path);
            }
            else {
                const newContent = record.apply(entry.content);
                this.overwrite(path, newContent);
            }
        }
        else {
            throw new exception_1.InvalidUpdateRecordException();
        }
    }
    overwrite(path, content) {
        const normalizedTo = this._normalizePath(path);
        if (typeof content == 'string') {
            content = new Buffer(content, 'utf-8');
        }
        const maybeEntry = this.get(normalizedTo);
        if (maybeEntry && maybeEntry.content.equals(content)) {
            return;
        }
        this._overwrite(normalizedTo, content);
    }
    create(path, content) {
        const normalizedTo = this._normalizePath(path);
        if (typeof content == 'string') {
            content = new Buffer(content);
        }
        this._create(normalizedTo, content);
    }
    rename(path, to) {
        const normalizedPath = this._normalizePath(path);
        const normalizedTo = this._normalizePath(to);
        if (normalizedPath === normalizedTo) {
            // Nothing to do.
            return;
        }
        this._rename(normalizedPath, normalizedTo);
    }
    delete(path) {
        this._delete(this._normalizePath(path));
    }
    _overwrite(path, content, action) {
        if (!this.has(path)) {
            throw new exception_1.FileDoesNotExistException(path);
        }
        // Update the action buffer.
        if (action) {
            this._actions.push(action);
        }
        else {
            this._actions.overwrite(path, content);
        }
        this.set(new entry_1.SimpleFileEntry(path, content));
    }
    _create(path, content, action) {
        if (this._cacheMap.has(path)) {
            throw new exception_1.FileAlreadyExistException(path);
        }
        if (action) {
            this._actions.push(action);
        }
        else {
            this._actions.create(path, content);
        }
        this.set(new entry_1.SimpleFileEntry(path, content));
    }
    _rename(path, to, action, force = false) {
        const entry = this.get(path);
        if (!entry) {
            throw new exception_1.FileDoesNotExistException(path);
        }
        if (this._cacheMap.has(to) && !force) {
            throw new exception_1.FileAlreadyExistException(to);
        }
        if (action) {
            this._actions.push(action);
        }
        else {
            this._actions.rename(path, to);
        }
        this.set(new entry_1.SimpleFileEntry(to, entry.content));
        this._cacheMap.delete(path);
    }
    _delete(path, action) {
        if (!this.has(path)) {
            throw new exception_1.FileDoesNotExistException(path);
        }
        if (action) {
            this._actions.push(action);
        }
        else {
            this._actions.delete(path);
        }
        this._cacheMap.delete(path);
    }
    apply(action, strategy) {
        if (this._actions.has(action)) {
            return;
        }
        switch (action.kind) {
            case 'o':
                // Update the action buffer.
                this._overwrite(action.path, action.content, action);
                break;
            case 'c':
                if (this._cacheMap.has(action.path)) {
                    switch (strategy) {
                        case interface_1.MergeStrategy.Error: throw new exception_1.MergeConflictException(action.path);
                        case interface_1.MergeStrategy.Overwrite:
                            this._overwrite(action.path, action.content, action);
                            break;
                    }
                }
                else {
                    this._create(action.path, action.content, action);
                }
                break;
            case 'r':
                const force = (strategy & interface_1.MergeStrategy.AllowOverwriteConflict) != 0;
                this._rename(action.path, action.to, action, force);
                break;
            case 'd':
                this._delete(action.path, action);
                break;
            default: throw new action_1.UnknownActionException(action);
        }
    }
    // Returns an ordered list of Action to get this host.
    get actions() {
        return [...this._actions];
    }
    /**
     * Allow subclasses to copy to a tree their own properties.
     * @return {Tree}
     * @private
     */
    _copyTo(tree) {
        tree._tree = new Map(this.tree);
        this._actions.forEach(action => tree._actions.push(action));
        [...this._cacheMap.entries()].forEach(([path, entry]) => {
            tree._cacheMap.set(path, entry);
        });
    }
    branch() {
        const newTree = new VirtualTree();
        this._copyTo(newTree);
        return newTree;
    }
    // Creates a new host from 2 hosts.
    merge(other, strategy = interface_1.MergeStrategy.Default) {
        other.actions.forEach(action => this.apply(action, strategy));
    }
    optimize() {
        // This destroys the history. Hope you know what you're doing.
        this._actions.optimize();
    }
    static branch(tree) {
        return tree.branch();
    }
    static merge(tree, other, strategy = interface_1.MergeStrategy.Default) {
        const newTree = tree.branch();
        newTree.merge(other, strategy);
        return newTree;
    }
    static optimize(tree) {
        const newTree = tree.branch();
        newTree.optimize();
        return newTree;
    }
}
exports.VirtualTree = VirtualTree;
//# sourceMappingURL=data:application/json;base64,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