"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
const interface_1 = require("../tree/interface");
const static_1 = require("../tree/static");
/**
 * Run a schematic from a separate collection.
 *
 * @param collectionName The name of the collection that contains the schematic to run.
 * @param schematicName The name of the schematic to run.
 * @param options The options to pass as input to the RuleFactory.
 */
function externalSchematic(collectionName, schematicName, options) {
    return (input, context) => {
        const collection = context.engine.createCollection(collectionName);
        const schematic = collection.createSchematic(schematicName);
        return schematic.call(options, rxjs_1.of(static_1.branch(input)), context);
    };
}
exports.externalSchematic = externalSchematic;
/**
 * Run a schematic from the same collection.
 *
 * @param schematicName The name of the schematic to run.
 * @param options The options to pass as input to the RuleFactory.
 */
function schematic(schematicName, options) {
    return (input, context) => {
        const collection = context.schematic.collection;
        const schematic = collection.createSchematic(schematicName, true);
        return schematic.call(options, rxjs_1.of(static_1.branch(input)), context).pipe(operators_1.last(), operators_1.map(x => {
            // We allow overwrite conflict here because they're the only merge conflict we particularly
            // don't want to deal with; the input tree might have an OVERWRITE which the sub
            input.merge(x, interface_1.MergeStrategy.AllowOverwriteConflict);
            return input;
        }));
    };
}
exports.schematic = schematic;
//# sourceMappingURL=data:application/json;base64,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