"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
const exception_1 = require("../exception/exception");
const action_1 = require("../tree/action");
const Noop = function () { };
class SimpleSinkBase {
    constructor() {
        this.preCommitAction = Noop;
        this.postCommitAction = Noop;
        this.preCommit = Noop;
        this.postCommit = Noop;
    }
    _fileAlreadyExistException(path) {
        throw new exception_1.FileAlreadyExistException(path);
    }
    _fileDoesNotExistException(path) {
        throw new exception_1.FileDoesNotExistException(path);
    }
    _validateOverwriteAction(action) {
        return this._validateFileExists(action.path)
            .pipe(operators_1.map(b => { if (!b) {
            this._fileDoesNotExistException(action.path);
        } }));
    }
    _validateCreateAction(action) {
        return this._validateFileExists(action.path)
            .pipe(operators_1.map(b => { if (b) {
            this._fileAlreadyExistException(action.path);
        } }));
    }
    _validateRenameAction(action) {
        return this._validateFileExists(action.path).pipe(operators_1.map(b => { if (!b) {
            this._fileDoesNotExistException(action.path);
        } }), operators_1.mergeMap(() => this._validateFileExists(action.to)), operators_1.map(b => { if (b) {
            this._fileAlreadyExistException(action.to);
        } }));
    }
    _validateDeleteAction(action) {
        return this._validateFileExists(action.path)
            .pipe(operators_1.map(b => { if (!b) {
            this._fileDoesNotExistException(action.path);
        } }));
    }
    validateSingleAction(action) {
        switch (action.kind) {
            case 'o': return this._validateOverwriteAction(action);
            case 'c': return this._validateCreateAction(action);
            case 'r': return this._validateRenameAction(action);
            case 'd': return this._validateDeleteAction(action);
            default: throw new action_1.UnknownActionException(action);
        }
    }
    commitSingleAction(action) {
        return rxjs_1.concat(this.validateSingleAction(action), new rxjs_1.Observable(observer => {
            let committed = null;
            switch (action.kind) {
                case 'o':
                    committed = this._overwriteFile(action.path, action.content);
                    break;
                case 'c':
                    committed = this._createFile(action.path, action.content);
                    break;
                case 'r':
                    committed = this._renameFile(action.path, action.to);
                    break;
                case 'd':
                    committed = this._deleteFile(action.path);
                    break;
            }
            if (committed) {
                committed.subscribe(observer);
            }
            else {
                observer.complete();
            }
        })).pipe(operators_1.ignoreElements());
    }
    commit(tree) {
        const actions = rxjs_1.from(tree.actions);
        return rxjs_1.concat((this.preCommit() || rxjs_1.of(null)), rxjs_1.defer(() => actions).pipe(operators_1.concatMap(action => {
            const maybeAction = this.preCommitAction(action);
            if (!maybeAction) {
                return rxjs_1.of(action);
            }
            else if (action_1.isAction(maybeAction)) {
                return rxjs_1.of(maybeAction);
            }
            else {
                return maybeAction;
            }
        }), operators_1.concatMap(action => {
            return rxjs_1.concat(this.commitSingleAction(action).pipe(operators_1.ignoreElements()), rxjs_1.of(action));
        }), operators_1.concatMap(action => this.postCommitAction(action) || rxjs_1.of(null))), rxjs_1.defer(() => this._done()), rxjs_1.defer(() => this.postCommit() || rxjs_1.of(null))).pipe(operators_1.ignoreElements());
    }
}
exports.SimpleSinkBase = SimpleSinkBase;
//# sourceMappingURL=data:application/json;base64,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