"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const core_1 = require("@angular-devkit/core");
const entry_1 = require("./entry");
const virtual_1 = require("./virtual");
class FileSystemDirEntry extends virtual_1.VirtualDirEntry {
    constructor(_host, tree, path = core_1.normalize('/')) {
        super(tree, path);
        this._host = _host;
    }
    _createDir(name) {
        return new FileSystemDirEntry(this._host, this._tree, core_1.join(this._path, name));
    }
    get parent() {
        return this._path == '/' ? null : this._tree.getDir(core_1.dirname(this._path));
    }
    get subdirs() {
        const result = new Set();
        try {
            this._host.list(this._path)
                .filter(name => this._host.isDirectory(core_1.join(this._path, name)))
                .forEach(name => result.add(core_1.fragment(name)));
        }
        catch (e) {
            if (e.code != 'ENOENT' && e.code != 'ENOTDIR') {
                throw e;
            }
        }
        for (const path of this._tree.staging.keys()) {
            if (path.startsWith(this._path) && core_1.dirname(path) != this._path) {
                result.add(core_1.basename(path));
            }
        }
        return [...result];
    }
    get subfiles() {
        const result = new Set();
        try {
            this._host.list(this._path)
                .filter(name => !this._host.isDirectory(core_1.join(this._path, name)))
                .forEach(name => result.add(core_1.fragment(name)));
        }
        catch (e) {
            if (e.code != 'ENOENT' && e.code != 'ENOTDIR') {
                throw e;
            }
        }
        for (const path of this._tree.staging.keys()) {
            if (path.startsWith(this._path) && core_1.dirname(path) == this._path) {
                result.add(core_1.basename(path));
            }
        }
        return [...result];
    }
    file(name) {
        return this._tree.get(core_1.join(this._path, name));
    }
}
exports.FileSystemDirEntry = FileSystemDirEntry;
class FileSystemTree extends virtual_1.VirtualTree {
    constructor(host) {
        super();
        this._initialized = false;
        this._host = new core_1.virtualFs.SyncDelegateHost(host);
        this._root = new FileSystemDirEntry(this._host, this);
    }
    get tree() {
        const host = this._host;
        if (!this._initialized) {
            this._initialized = true;
            this._recursiveFileList().forEach(path => {
                this._tree.set(path, new entry_1.LazyFileEntry(path, () => {
                    return Buffer.from(host.read(path));
                }));
            });
        }
        return this._tree;
    }
    get(path) {
        const normalizedPath = this._normalizePath(path);
        let entry = this._cacheMap.get(normalizedPath) || this._tree.get(normalizedPath) || null;
        if (entry == null && !this._initialized) {
            const systemPath = normalizedPath;
            const fileExists = this._host.exists(systemPath) && !this._host.isDirectory(systemPath);
            if (fileExists) {
                const host = this._host;
                entry = new entry_1.LazyFileEntry(normalizedPath, () => Buffer.from(host.read(systemPath)));
                this._tree.set(normalizedPath, entry);
            }
        }
        return entry;
    }
    branch() {
        const newTree = new FileSystemTree(this._host.delegate);
        this._copyTo(newTree);
        return newTree;
    }
    _copyTo(tree) {
        if (tree instanceof FileSystemTree) {
            const x = tree;
            x._tree = this._tree;
            x._initialized = this._initialized;
            this._actions.forEach(action => x._actions.push(action));
            [...this._cacheMap.entries()].forEach(([path, entry]) => {
                x._cacheMap.set(path, entry);
            });
        }
        else {
            super._copyTo(tree);
        }
    }
    _recursiveFileList() {
        const host = this._host;
        const list = [];
        function recurse(schematicPath) {
            for (const name of host.list(schematicPath)) {
                const normalizedPath = core_1.join(schematicPath, name);
                if (host.isDirectory(normalizedPath)) {
                    recurse(normalizedPath);
                }
                else {
                    list.push(normalizedPath);
                }
            }
        }
        recurse(core_1.normalize('/'));
        return list;
    }
}
exports.FileSystemTree = FileSystemTree;
class FileSystemCreateTree extends FileSystemTree {
    constructor(host) {
        super(host);
        this._recursiveFileList().forEach(path => {
            this.create(path, Buffer.from(this._host.read(path)));
        });
        this._initialized = true;
    }
}
exports.FileSystemCreateTree = FileSystemCreateTree;
//# sourceMappingURL=data:application/json;base64,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