"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const node_1 = require("@angular-devkit/core/node");
const fs = require("fs");
const path = require("path");
const rxjs_1 = require("rxjs");
function _loadConfiguration(Configuration, options, root, file) {
    if (options.tslintConfig) {
        return Configuration.parseConfigFile(options.tslintConfig, root);
    }
    else if (options.tslintPath) {
        const tslintPath = path.join(root, options.tslintPath);
        return Configuration.findConfiguration(tslintPath, file && path.join(root, file)).results;
    }
    else {
        throw new Error('Executor must specify a tslint configuration.');
    }
}
function _getFileContent(file, options, program) {
    // The linter retrieves the SourceFile TS node directly if a program is used
    if (program) {
        const source = program.getSourceFile(file);
        if (!source) {
            const message = `File '${file}' is not part of the TypeScript project '${options.tsConfigPath}'.`;
            throw new Error(message);
        }
        return source.getFullText(source);
    }
    // NOTE: The tslint CLI checks for and excludes MPEG transport streams; this does not.
    try {
        // Strip BOM from file data.
        // https://stackoverflow.com/questions/24356713
        return fs.readFileSync(file, 'utf-8').replace(/^\uFEFF/, '');
    }
    catch (_a) {
        throw new Error(`Could not read file '${file}'.`);
    }
}
function _listAllFiles(root) {
    const result = [];
    function _recurse(location) {
        const dir = fs.readdirSync(path.join(root, location));
        dir.forEach(name => {
            const loc = path.join(location, name);
            if (fs.statSync(path.join(root, loc)).isDirectory()) {
                _recurse(loc);
            }
            else {
                result.push(loc);
            }
        });
    }
    _recurse('');
    return result;
}
function default_1() {
    return (options, context) => {
        return new rxjs_1.Observable(obs => {
            const root = process.cwd();
            const tslint = require(node_1.resolve('tslint', {
                basedir: root,
                checkGlobal: true,
                checkLocal: true,
            }));
            const includes = (Array.isArray(options.includes)
                ? options.includes
                : (options.includes ? [options.includes] : []));
            const Linter = tslint.Linter;
            const Configuration = tslint.Configuration;
            let program = undefined;
            let filesToLint = [];
            if (options.tsConfigPath) {
                const tsConfigPath = path.join(process.cwd(), options.tsConfigPath);
                if (!fs.existsSync(tsConfigPath)) {
                    obs.error(new Error('Could not find tsconfig.'));
                    return;
                }
                program = Linter.createProgram(tsConfigPath);
                filesToLint = Linter.getFileNames(program);
            }
            if (includes.length > 0) {
                const allFilesRel = _listAllFiles(root);
                const pattern = '^('
                    + includes
                        .map(ex => '('
                        + ex.split(/[\/\\]/g).map(f => f
                            .replace(/[\-\[\]{}()+?.^$|]/g, '\\$&')
                            .replace(/^\*\*/g, '(.+?)?')
                            .replace(/\*/g, '[^/\\\\]*'))
                            .join('[\/\\\\]')
                        + ')')
                        .join('|')
                    + ')($|/|\\\\)';
                const re = new RegExp(pattern);
                filesToLint.push(...allFilesRel
                    .filter(x => re.test(x))
                    .map(x => path.join(root, x)));
            }
            const lintOptions = {
                fix: true,
                formatter: options.format || 'prose',
            };
            const linter = new Linter(lintOptions, program);
            const config = _loadConfiguration(Configuration, options, root);
            for (const file of filesToLint) {
                const content = _getFileContent(file, options, program);
                if (!content) {
                    continue;
                }
                linter.lint(file, content, config);
            }
            const result = linter.getResult();
            // Format and show the results.
            if (!options.silent) {
                const Formatter = tslint.findFormatter(options.format || 'prose');
                if (!Formatter) {
                    throw new Error(`Invalid lint format "${options.format}".`);
                }
                const formatter = new Formatter();
                const output = formatter.format(result.failures, result.fixes);
                if (output) {
                    context.logger.info(output);
                }
            }
            if (!options.ignoreErrors && result.errorCount > 0) {
                obs.error(new Error('Lint errors were found.'));
            }
            else {
                obs.next();
                obs.complete();
            }
        });
    };
}
exports.default = default_1;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZXhlY3V0b3IuanMiLCJzb3VyY2VSb290IjoiLi8iLCJzb3VyY2VzIjpbInBhY2thZ2VzL2FuZ3VsYXJfZGV2a2l0L3NjaGVtYXRpY3MvdGFza3MvdHNsaW50LWZpeC9leGVjdXRvci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBOzs7Ozs7R0FNRztBQUNILG9EQUFvRDtBQUNwRCx5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLCtCQUFrQztBQWNsQyw0QkFDRSxhQUE2QixFQUM3QixPQUE2QixFQUM3QixJQUFZLEVBQ1osSUFBYTtJQUViLEVBQUUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDO1FBQ3pCLE1BQU0sQ0FBQyxhQUFhLENBQUMsZUFBZSxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFDbkUsQ0FBQztJQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQztRQUM5QixNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUM7UUFFdkQsTUFBTSxDQUFDLGFBQWEsQ0FBQyxpQkFBaUIsQ0FBQyxVQUFVLEVBQUUsSUFBSSxJQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDO0lBQzVGLENBQUM7SUFBQyxJQUFJLENBQUMsQ0FBQztRQUNOLE1BQU0sSUFBSSxLQUFLLENBQUMsK0NBQStDLENBQUMsQ0FBQztJQUNuRSxDQUFDO0FBQ0gsQ0FBQztBQUdELHlCQUNFLElBQVksRUFDWixPQUE2QixFQUM3QixPQUFvQjtJQUVwQiw0RUFBNEU7SUFDNUUsRUFBRSxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztRQUNaLE1BQU0sTUFBTSxHQUFHLE9BQU8sQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDM0MsRUFBRSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDO1lBQ1osTUFBTSxPQUFPLEdBQ1QsU0FBUyxJQUFJLDRDQUE0QyxPQUFPLENBQUMsWUFBWSxJQUFJLENBQUM7WUFDdEYsTUFBTSxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUMzQixDQUFDO1FBRUQsTUFBTSxDQUFDLE1BQU0sQ0FBQyxXQUFXLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVELHNGQUFzRjtJQUN0RixJQUFJLENBQUM7UUFDSCw0QkFBNEI7UUFDNUIsK0NBQStDO1FBQy9DLE1BQU0sQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQyxPQUFPLENBQUMsU0FBUyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQy9ELENBQUM7SUFBQyxLQUFLLENBQUMsQ0FBQyxJQUFELENBQUM7UUFDUCxNQUFNLElBQUksS0FBSyxDQUFDLHdCQUF3QixJQUFJLElBQUksQ0FBQyxDQUFDO0lBQ3BELENBQUM7QUFDSCxDQUFDO0FBR0QsdUJBQXVCLElBQVk7SUFDakMsTUFBTSxNQUFNLEdBQWEsRUFBRSxDQUFDO0lBRTVCLGtCQUFrQixRQUFnQjtRQUNoQyxNQUFNLEdBQUcsR0FBRyxFQUFFLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLFFBQVEsQ0FBQyxDQUFDLENBQUM7UUFFdEQsR0FBRyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUNqQixNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsQ0FBQztZQUN0QyxFQUFFLENBQUMsQ0FBQyxFQUFFLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDO2dCQUNwRCxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUM7WUFDaEIsQ0FBQztZQUFDLElBQUksQ0FBQyxDQUFDO2dCQUNOLE1BQU0sQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7WUFDbkIsQ0FBQztRQUNILENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUNELFFBQVEsQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUViLE1BQU0sQ0FBQyxNQUFNLENBQUM7QUFDaEIsQ0FBQztBQUdEO0lBQ0UsTUFBTSxDQUFDLENBQUMsT0FBNkIsRUFBRSxPQUF5QixFQUFFLEVBQUU7UUFDbEUsTUFBTSxDQUFDLElBQUksaUJBQVUsQ0FBQyxHQUFHLENBQUMsRUFBRTtZQUMxQixNQUFNLElBQUksR0FBRyxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUM7WUFDM0IsTUFBTSxNQUFNLEdBQUcsT0FBTyxDQUFDLGNBQU8sQ0FBQyxRQUFRLEVBQUU7Z0JBQ3ZDLE9BQU8sRUFBRSxJQUFJO2dCQUNiLFdBQVcsRUFBRSxJQUFJO2dCQUNqQixVQUFVLEVBQUUsSUFBSTthQUNqQixDQUFDLENBQUMsQ0FBQztZQUNKLE1BQU0sUUFBUSxHQUFHLENBQ2YsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDO2dCQUM3QixDQUFDLENBQUMsT0FBTyxDQUFDLFFBQVE7Z0JBQ2xCLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FDakQsQ0FBQztZQUVGLE1BQU0sTUFBTSxHQUFHLE1BQU0sQ0FBQyxNQUFpQixDQUFDO1lBQ3hDLE1BQU0sYUFBYSxHQUFHLE1BQU0sQ0FBQyxhQUErQixDQUFDO1lBQzdELElBQUksT0FBTyxHQUEyQixTQUFTLENBQUM7WUFDaEQsSUFBSSxXQUFXLEdBQWEsRUFBRSxDQUFDO1lBRS9CLEVBQUUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDO2dCQUN6QixNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsRUFBRSxPQUFPLENBQUMsWUFBWSxDQUFDLENBQUM7Z0JBRXBFLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLFVBQVUsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUM7b0JBQ2pDLEdBQUcsQ0FBQyxLQUFLLENBQUMsSUFBSSxLQUFLLENBQUMsMEJBQTBCLENBQUMsQ0FBQyxDQUFDO29CQUVqRCxNQUFNLENBQUM7Z0JBQ1QsQ0FBQztnQkFDRCxPQUFPLEdBQUcsTUFBTSxDQUFDLGFBQWEsQ0FBQyxZQUFZLENBQUMsQ0FBQztnQkFDN0MsV0FBVyxHQUFHLE1BQU0sQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLENBQUM7WUFDN0MsQ0FBQztZQUVELEVBQUUsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDeEIsTUFBTSxXQUFXLEdBQUcsYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFDO2dCQUN4QyxNQUFNLE9BQU8sR0FBRyxJQUFJO3NCQUNmLFFBQXFCO3lCQUNyQixHQUFHLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxHQUFHOzBCQUNWLEVBQUUsQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQzs2QkFDN0IsT0FBTyxDQUFDLHFCQUFxQixFQUFFLE1BQU0sQ0FBQzs2QkFDdEMsT0FBTyxDQUFDLFFBQVEsRUFBRSxRQUFRLENBQUM7NkJBQzNCLE9BQU8sQ0FBQyxLQUFLLEVBQUUsV0FBVyxDQUFDLENBQUM7NkJBQzVCLElBQUksQ0FBQyxVQUFVLENBQUM7MEJBQ2pCLEdBQUcsQ0FBQzt5QkFDUCxJQUFJLENBQUMsR0FBRyxDQUFDO3NCQUNWLGFBQWEsQ0FBQztnQkFDbEIsTUFBTSxFQUFFLEdBQUcsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUM7Z0JBRS9CLFdBQVcsQ0FBQyxJQUFJLENBQUMsR0FBRyxXQUFXO3FCQUM1QixNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO3FCQUN2QixHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUM5QixDQUFDO1lBQ0osQ0FBQztZQUVELE1BQU0sV0FBVyxHQUFHO2dCQUNsQixHQUFHLEVBQUUsSUFBSTtnQkFDVCxTQUFTLEVBQUUsT0FBTyxDQUFDLE1BQU0sSUFBSSxPQUFPO2FBQ3JDLENBQUM7WUFFRixNQUFNLE1BQU0sR0FBRyxJQUFJLE1BQU0sQ0FBQyxXQUFXLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDaEQsTUFBTSxNQUFNLEdBQUcsa0JBQWtCLENBQUMsYUFBYSxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsQ0FBQztZQUVoRSxHQUFHLENBQUMsQ0FBQyxNQUFNLElBQUksSUFBSSxXQUFXLENBQUMsQ0FBQyxDQUFDO2dCQUMvQixNQUFNLE9BQU8sR0FBRyxlQUFlLENBQUMsSUFBSSxFQUFFLE9BQU8sRUFBRSxPQUFPLENBQUMsQ0FBQztnQkFFeEQsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO29CQUNiLFFBQVEsQ0FBQztnQkFDWCxDQUFDO2dCQUVELE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLE9BQU8sRUFBRSxNQUFNLENBQUMsQ0FBQztZQUNyQyxDQUFDO1lBRUQsTUFBTSxNQUFNLEdBQUcsTUFBTSxDQUFDLFNBQVMsRUFBRSxDQUFDO1lBRWxDLCtCQUErQjtZQUMvQixFQUFFLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDO2dCQUNwQixNQUFNLFNBQVMsR0FBRyxNQUFNLENBQUMsYUFBYSxDQUFDLE9BQU8sQ0FBQyxNQUFNLElBQUksT0FBTyxDQUFDLENBQUM7Z0JBQ2xFLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQztvQkFDZixNQUFNLElBQUksS0FBSyxDQUFDLHdCQUF3QixPQUFPLENBQUMsTUFBTSxJQUFJLENBQUMsQ0FBQztnQkFDOUQsQ0FBQztnQkFDRCxNQUFNLFNBQVMsR0FBRyxJQUFJLFNBQVMsRUFBRSxDQUFDO2dCQUVsQyxNQUFNLE1BQU0sR0FBRyxTQUFTLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxRQUFRLEVBQUUsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDO2dCQUMvRCxFQUFFLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDO29CQUNYLE9BQU8sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO2dCQUM5QixDQUFDO1lBQ0gsQ0FBQztZQUVELEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLFlBQVksSUFBSSxNQUFNLENBQUMsVUFBVSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ25ELEdBQUcsQ0FBQyxLQUFLLENBQUMsSUFBSSxLQUFLLENBQUMseUJBQXlCLENBQUMsQ0FBQyxDQUFDO1lBQ2xELENBQUM7WUFBQyxJQUFJLENBQUMsQ0FBQztnQkFDTixHQUFHLENBQUMsSUFBSSxFQUFFLENBQUM7Z0JBQ1gsR0FBRyxDQUFDLFFBQVEsRUFBRSxDQUFDO1lBQ2pCLENBQUM7UUFDSCxDQUFDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQztBQUNKLENBQUM7QUEvRkQsNEJBK0ZDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBAbGljZW5zZVxuICogQ29weXJpZ2h0IEdvb2dsZSBJbmMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogVXNlIG9mIHRoaXMgc291cmNlIGNvZGUgaXMgZ292ZXJuZWQgYnkgYW4gTUlULXN0eWxlIGxpY2Vuc2UgdGhhdCBjYW4gYmVcbiAqIGZvdW5kIGluIHRoZSBMSUNFTlNFIGZpbGUgYXQgaHR0cHM6Ly9hbmd1bGFyLmlvL2xpY2Vuc2VcbiAqL1xuaW1wb3J0IHsgcmVzb2x2ZSB9IGZyb20gJ0Bhbmd1bGFyLWRldmtpdC9jb3JlL25vZGUnO1xuaW1wb3J0ICogYXMgZnMgZnJvbSAnZnMnO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCB7IE9ic2VydmFibGUgfSBmcm9tICdyeGpzJztcbmltcG9ydCB7XG4gIENvbmZpZ3VyYXRpb24gYXMgQ29uZmlndXJhdGlvbk5TLFxuICBMaW50ZXIgYXMgTGludGVyTlMsXG59IGZyb20gJ3RzbGludCc7ICAvLyB0c2xpbnQ6ZGlzYWJsZS1saW5lOm5vLWltcGxpY2l0LWRlcGVuZGVuY2llc1xuaW1wb3J0ICogYXMgdHMgZnJvbSAndHlwZXNjcmlwdCc7ICAvLyB0c2xpbnQ6ZGlzYWJsZS1saW5lOm5vLWltcGxpY2l0LWRlcGVuZGVuY2llc1xuaW1wb3J0IHsgU2NoZW1hdGljQ29udGV4dCwgVGFza0V4ZWN1dG9yIH0gZnJvbSAnLi4vLi4vc3JjJztcbmltcG9ydCB7IFRzbGludEZpeFRhc2tPcHRpb25zIH0gZnJvbSAnLi9vcHRpb25zJztcblxuXG50eXBlIENvbmZpZ3VyYXRpb25UID0gdHlwZW9mIENvbmZpZ3VyYXRpb25OUztcbnR5cGUgTGludGVyVCA9IHR5cGVvZiBMaW50ZXJOUztcblxuXG5mdW5jdGlvbiBfbG9hZENvbmZpZ3VyYXRpb24oXG4gIENvbmZpZ3VyYXRpb246IENvbmZpZ3VyYXRpb25ULFxuICBvcHRpb25zOiBUc2xpbnRGaXhUYXNrT3B0aW9ucyxcbiAgcm9vdDogc3RyaW5nLFxuICBmaWxlPzogc3RyaW5nLFxuKSB7XG4gIGlmIChvcHRpb25zLnRzbGludENvbmZpZykge1xuICAgIHJldHVybiBDb25maWd1cmF0aW9uLnBhcnNlQ29uZmlnRmlsZShvcHRpb25zLnRzbGludENvbmZpZywgcm9vdCk7XG4gIH0gZWxzZSBpZiAob3B0aW9ucy50c2xpbnRQYXRoKSB7XG4gICAgY29uc3QgdHNsaW50UGF0aCA9IHBhdGguam9pbihyb290LCBvcHRpb25zLnRzbGludFBhdGgpO1xuXG4gICAgcmV0dXJuIENvbmZpZ3VyYXRpb24uZmluZENvbmZpZ3VyYXRpb24odHNsaW50UGF0aCwgZmlsZSAmJiBwYXRoLmpvaW4ocm9vdCwgZmlsZSkpLnJlc3VsdHM7XG4gIH0gZWxzZSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKCdFeGVjdXRvciBtdXN0IHNwZWNpZnkgYSB0c2xpbnQgY29uZmlndXJhdGlvbi4nKTtcbiAgfVxufVxuXG5cbmZ1bmN0aW9uIF9nZXRGaWxlQ29udGVudChcbiAgZmlsZTogc3RyaW5nLFxuICBvcHRpb25zOiBUc2xpbnRGaXhUYXNrT3B0aW9ucyxcbiAgcHJvZ3JhbT86IHRzLlByb2dyYW0sXG4pOiBzdHJpbmcgfCB1bmRlZmluZWQge1xuICAvLyBUaGUgbGludGVyIHJldHJpZXZlcyB0aGUgU291cmNlRmlsZSBUUyBub2RlIGRpcmVjdGx5IGlmIGEgcHJvZ3JhbSBpcyB1c2VkXG4gIGlmIChwcm9ncmFtKSB7XG4gICAgY29uc3Qgc291cmNlID0gcHJvZ3JhbS5nZXRTb3VyY2VGaWxlKGZpbGUpO1xuICAgIGlmICghc291cmNlKSB7XG4gICAgICBjb25zdCBtZXNzYWdlXG4gICAgICAgID0gYEZpbGUgJyR7ZmlsZX0nIGlzIG5vdCBwYXJ0IG9mIHRoZSBUeXBlU2NyaXB0IHByb2plY3QgJyR7b3B0aW9ucy50c0NvbmZpZ1BhdGh9Jy5gO1xuICAgICAgdGhyb3cgbmV3IEVycm9yKG1lc3NhZ2UpO1xuICAgIH1cblxuICAgIHJldHVybiBzb3VyY2UuZ2V0RnVsbFRleHQoc291cmNlKTtcbiAgfVxuXG4gIC8vIE5PVEU6IFRoZSB0c2xpbnQgQ0xJIGNoZWNrcyBmb3IgYW5kIGV4Y2x1ZGVzIE1QRUcgdHJhbnNwb3J0IHN0cmVhbXM7IHRoaXMgZG9lcyBub3QuXG4gIHRyeSB7XG4gICAgLy8gU3RyaXAgQk9NIGZyb20gZmlsZSBkYXRhLlxuICAgIC8vIGh0dHBzOi8vc3RhY2tvdmVyZmxvdy5jb20vcXVlc3Rpb25zLzI0MzU2NzEzXG4gICAgcmV0dXJuIGZzLnJlYWRGaWxlU3luYyhmaWxlLCAndXRmLTgnKS5yZXBsYWNlKC9eXFx1RkVGRi8sICcnKTtcbiAgfSBjYXRjaCB7XG4gICAgdGhyb3cgbmV3IEVycm9yKGBDb3VsZCBub3QgcmVhZCBmaWxlICcke2ZpbGV9Jy5gKTtcbiAgfVxufVxuXG5cbmZ1bmN0aW9uIF9saXN0QWxsRmlsZXMocm9vdDogc3RyaW5nKTogc3RyaW5nW10ge1xuICBjb25zdCByZXN1bHQ6IHN0cmluZ1tdID0gW107XG5cbiAgZnVuY3Rpb24gX3JlY3Vyc2UobG9jYXRpb246IHN0cmluZykge1xuICAgIGNvbnN0IGRpciA9IGZzLnJlYWRkaXJTeW5jKHBhdGguam9pbihyb290LCBsb2NhdGlvbikpO1xuXG4gICAgZGlyLmZvckVhY2gobmFtZSA9PiB7XG4gICAgICBjb25zdCBsb2MgPSBwYXRoLmpvaW4obG9jYXRpb24sIG5hbWUpO1xuICAgICAgaWYgKGZzLnN0YXRTeW5jKHBhdGguam9pbihyb290LCBsb2MpKS5pc0RpcmVjdG9yeSgpKSB7XG4gICAgICAgIF9yZWN1cnNlKGxvYyk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICByZXN1bHQucHVzaChsb2MpO1xuICAgICAgfVxuICAgIH0pO1xuICB9XG4gIF9yZWN1cnNlKCcnKTtcblxuICByZXR1cm4gcmVzdWx0O1xufVxuXG5cbmV4cG9ydCBkZWZhdWx0IGZ1bmN0aW9uKCk6IFRhc2tFeGVjdXRvcjxUc2xpbnRGaXhUYXNrT3B0aW9ucz4ge1xuICByZXR1cm4gKG9wdGlvbnM6IFRzbGludEZpeFRhc2tPcHRpb25zLCBjb250ZXh0OiBTY2hlbWF0aWNDb250ZXh0KSA9PiB7XG4gICAgcmV0dXJuIG5ldyBPYnNlcnZhYmxlKG9icyA9PiB7XG4gICAgICBjb25zdCByb290ID0gcHJvY2Vzcy5jd2QoKTtcbiAgICAgIGNvbnN0IHRzbGludCA9IHJlcXVpcmUocmVzb2x2ZSgndHNsaW50Jywge1xuICAgICAgICBiYXNlZGlyOiByb290LFxuICAgICAgICBjaGVja0dsb2JhbDogdHJ1ZSxcbiAgICAgICAgY2hlY2tMb2NhbDogdHJ1ZSxcbiAgICAgIH0pKTtcbiAgICAgIGNvbnN0IGluY2x1ZGVzID0gKFxuICAgICAgICBBcnJheS5pc0FycmF5KG9wdGlvbnMuaW5jbHVkZXMpXG4gICAgICAgICAgPyBvcHRpb25zLmluY2x1ZGVzXG4gICAgICAgICAgOiAob3B0aW9ucy5pbmNsdWRlcyA/IFtvcHRpb25zLmluY2x1ZGVzXSA6IFtdKVxuICAgICAgKTtcblxuICAgICAgY29uc3QgTGludGVyID0gdHNsaW50LkxpbnRlciBhcyBMaW50ZXJUO1xuICAgICAgY29uc3QgQ29uZmlndXJhdGlvbiA9IHRzbGludC5Db25maWd1cmF0aW9uIGFzIENvbmZpZ3VyYXRpb25UO1xuICAgICAgbGV0IHByb2dyYW06IHRzLlByb2dyYW0gfCB1bmRlZmluZWQgPSB1bmRlZmluZWQ7XG4gICAgICBsZXQgZmlsZXNUb0xpbnQ6IHN0cmluZ1tdID0gW107XG5cbiAgICAgIGlmIChvcHRpb25zLnRzQ29uZmlnUGF0aCkge1xuICAgICAgICBjb25zdCB0c0NvbmZpZ1BhdGggPSBwYXRoLmpvaW4ocHJvY2Vzcy5jd2QoKSwgb3B0aW9ucy50c0NvbmZpZ1BhdGgpO1xuXG4gICAgICAgIGlmICghZnMuZXhpc3RzU3luYyh0c0NvbmZpZ1BhdGgpKSB7XG4gICAgICAgICAgb2JzLmVycm9yKG5ldyBFcnJvcignQ291bGQgbm90IGZpbmQgdHNjb25maWcuJykpO1xuXG4gICAgICAgICAgcmV0dXJuO1xuICAgICAgICB9XG4gICAgICAgIHByb2dyYW0gPSBMaW50ZXIuY3JlYXRlUHJvZ3JhbSh0c0NvbmZpZ1BhdGgpO1xuICAgICAgICBmaWxlc1RvTGludCA9IExpbnRlci5nZXRGaWxlTmFtZXMocHJvZ3JhbSk7XG4gICAgICB9XG5cbiAgICAgIGlmIChpbmNsdWRlcy5sZW5ndGggPiAwKSB7XG4gICAgICAgIGNvbnN0IGFsbEZpbGVzUmVsID0gX2xpc3RBbGxGaWxlcyhyb290KTtcbiAgICAgICAgY29uc3QgcGF0dGVybiA9ICdeKCdcbiAgICAgICAgICArIChpbmNsdWRlcyBhcyBzdHJpbmdbXSlcbiAgICAgICAgICAgIC5tYXAoZXggPT4gJygnXG4gICAgICAgICAgICAgICsgZXguc3BsaXQoL1tcXC9cXFxcXS9nKS5tYXAoZiA9PiBmXG4gICAgICAgICAgICAgICAgLnJlcGxhY2UoL1tcXC1cXFtcXF17fSgpKz8uXiR8XS9nLCAnXFxcXCQmJylcbiAgICAgICAgICAgICAgICAucmVwbGFjZSgvXlxcKlxcKi9nLCAnKC4rPyk/JylcbiAgICAgICAgICAgICAgICAucmVwbGFjZSgvXFwqL2csICdbXi9cXFxcXFxcXF0qJykpXG4gICAgICAgICAgICAgICAgLmpvaW4oJ1tcXC9cXFxcXFxcXF0nKVxuICAgICAgICAgICAgICArICcpJylcbiAgICAgICAgICAgIC5qb2luKCd8JylcbiAgICAgICAgICArICcpKCR8L3xcXFxcXFxcXCknO1xuICAgICAgICBjb25zdCByZSA9IG5ldyBSZWdFeHAocGF0dGVybik7XG5cbiAgICAgICAgZmlsZXNUb0xpbnQucHVzaCguLi5hbGxGaWxlc1JlbFxuICAgICAgICAgIC5maWx0ZXIoeCA9PiByZS50ZXN0KHgpKVxuICAgICAgICAgIC5tYXAoeCA9PiBwYXRoLmpvaW4ocm9vdCwgeCkpLFxuICAgICAgICApO1xuICAgICAgfVxuXG4gICAgICBjb25zdCBsaW50T3B0aW9ucyA9IHtcbiAgICAgICAgZml4OiB0cnVlLFxuICAgICAgICBmb3JtYXR0ZXI6IG9wdGlvbnMuZm9ybWF0IHx8ICdwcm9zZScsXG4gICAgICB9O1xuXG4gICAgICBjb25zdCBsaW50ZXIgPSBuZXcgTGludGVyKGxpbnRPcHRpb25zLCBwcm9ncmFtKTtcbiAgICAgIGNvbnN0IGNvbmZpZyA9IF9sb2FkQ29uZmlndXJhdGlvbihDb25maWd1cmF0aW9uLCBvcHRpb25zLCByb290KTtcblxuICAgICAgZm9yIChjb25zdCBmaWxlIG9mIGZpbGVzVG9MaW50KSB7XG4gICAgICAgIGNvbnN0IGNvbnRlbnQgPSBfZ2V0RmlsZUNvbnRlbnQoZmlsZSwgb3B0aW9ucywgcHJvZ3JhbSk7XG5cbiAgICAgICAgaWYgKCFjb250ZW50KSB7XG4gICAgICAgICAgY29udGludWU7XG4gICAgICAgIH1cblxuICAgICAgICBsaW50ZXIubGludChmaWxlLCBjb250ZW50LCBjb25maWcpO1xuICAgICAgfVxuXG4gICAgICBjb25zdCByZXN1bHQgPSBsaW50ZXIuZ2V0UmVzdWx0KCk7XG5cbiAgICAgIC8vIEZvcm1hdCBhbmQgc2hvdyB0aGUgcmVzdWx0cy5cbiAgICAgIGlmICghb3B0aW9ucy5zaWxlbnQpIHtcbiAgICAgICAgY29uc3QgRm9ybWF0dGVyID0gdHNsaW50LmZpbmRGb3JtYXR0ZXIob3B0aW9ucy5mb3JtYXQgfHwgJ3Byb3NlJyk7XG4gICAgICAgIGlmICghRm9ybWF0dGVyKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBJbnZhbGlkIGxpbnQgZm9ybWF0IFwiJHtvcHRpb25zLmZvcm1hdH1cIi5gKTtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCBmb3JtYXR0ZXIgPSBuZXcgRm9ybWF0dGVyKCk7XG5cbiAgICAgICAgY29uc3Qgb3V0cHV0ID0gZm9ybWF0dGVyLmZvcm1hdChyZXN1bHQuZmFpbHVyZXMsIHJlc3VsdC5maXhlcyk7XG4gICAgICAgIGlmIChvdXRwdXQpIHtcbiAgICAgICAgICBjb250ZXh0LmxvZ2dlci5pbmZvKG91dHB1dCk7XG4gICAgICAgIH1cbiAgICAgIH1cblxuICAgICAgaWYgKCFvcHRpb25zLmlnbm9yZUVycm9ycyAmJiByZXN1bHQuZXJyb3JDb3VudCA+IDApIHtcbiAgICAgICAgb2JzLmVycm9yKG5ldyBFcnJvcignTGludCBlcnJvcnMgd2VyZSBmb3VuZC4nKSk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBvYnMubmV4dCgpO1xuICAgICAgICBvYnMuY29tcGxldGUoKTtcbiAgICAgIH1cbiAgICB9KTtcbiAgfTtcbn1cbiJdfQ==