"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const core_1 = require("@angular-devkit/core");
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
const src_1 = require("../src");
const call_1 = require("../src/rules/call");
const node_1 = require("../tasks/node");
const tools_1 = require("../tools");
class UnitTestTree extends src_1.DelegateTree {
    get files() {
        const result = [];
        this.visit(path => result.push(path));
        return result;
    }
    readContent(path) {
        const buffer = this.read(path);
        if (buffer === null) {
            return '';
        }
        return buffer.toString();
    }
}
exports.UnitTestTree = UnitTestTree;
class SchematicTestRunner {
    constructor(_collectionName, collectionPath) {
        this._collectionName = _collectionName;
        this._engineHost = new tools_1.NodeModulesTestEngineHost();
        this._engine = new src_1.SchematicEngine(this._engineHost);
        this._engineHost.registerCollection(_collectionName, collectionPath);
        this._logger = new core_1.logging.Logger('test');
        const registry = new core_1.schema.CoreSchemaRegistry(src_1.formats.standardFormats);
        this._engineHost.registerOptionsTransform(tools_1.validateOptionsWithSchema(registry));
        this._engineHost.registerTaskExecutor(node_1.BuiltinTaskExecutor.NodePackage);
        this._engineHost.registerTaskExecutor(node_1.BuiltinTaskExecutor.RepositoryInitializer);
        this._engineHost.registerTaskExecutor(node_1.BuiltinTaskExecutor.RunSchematic);
        this._engineHost.registerTaskExecutor(node_1.BuiltinTaskExecutor.TslintFix);
        this._collection = this._engine.createCollection(this._collectionName);
    }
    get engine() { return this._engine; }
    get logger() { return this._logger; }
    get tasks() { return [...this._engineHost.tasks]; }
    runSchematicAsync(schematicName, opts, tree) {
        const schematic = this._collection.createSchematic(schematicName, true);
        const host = rxjs_1.of(tree || new src_1.HostTree);
        this._engineHost.clearTasks();
        return schematic.call(opts || {}, host, { logger: this._logger })
            .pipe(operators_1.map(tree => new UnitTestTree(tree)));
    }
    runSchematic(schematicName, opts, tree) {
        const schematic = this._collection.createSchematic(schematicName, true);
        let result = null;
        let error;
        const host = rxjs_1.of(tree || new src_1.HostTree);
        this._engineHost.clearTasks();
        schematic.call(opts || {}, host, { logger: this._logger })
            .subscribe(t => result = new UnitTestTree(t), e => error = e);
        if (error) {
            throw error;
        }
        if (result === null) {
            throw new Error('Schematic is async, please use runSchematicAsync');
        }
        return result;
    }
    runExternalSchematicAsync(collectionName, schematicName, opts, tree) {
        const externalCollection = this._engine.createCollection(collectionName);
        const schematic = externalCollection.createSchematic(schematicName, true);
        const host = rxjs_1.of(tree || new src_1.HostTree);
        this._engineHost.clearTasks();
        return schematic.call(opts || {}, host, { logger: this._logger })
            .pipe(operators_1.map(tree => new UnitTestTree(tree)));
    }
    runExternalSchematic(collectionName, schematicName, opts, tree) {
        const externalCollection = this._engine.createCollection(collectionName);
        const schematic = externalCollection.createSchematic(schematicName, true);
        let result = null;
        const host = rxjs_1.of(tree || new src_1.HostTree);
        this._engineHost.clearTasks();
        schematic.call(opts || {}, host, { logger: this._logger })
            .subscribe(t => result = new UnitTestTree(t));
        if (result === null) {
            throw new Error('Schematic is async, please use runSchematicAsync');
        }
        return result;
    }
    callRule(rule, tree, parentContext) {
        const context = this._engine.createContext({}, parentContext);
        return call_1.callRule(rule, rxjs_1.of(tree), context);
    }
}
exports.SchematicTestRunner = SchematicTestRunner;
//# sourceMappingURL=data:application/json;base64,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