import { ChangeDetectorRef, Directive, ElementRef, EventEmitter, Injectable, Injector, Input, NgModule, Output, Pipe, Renderer2, forwardRef } from '@angular/core';
import { Subject } from 'rxjs/Subject';
import { Observable } from 'rxjs/Observable';
import 'rxjs/add/operator/toPromise';
import 'rxjs/add/observable/merge';
import { Http } from '@angular/http';
import 'rxjs/add/operator/map';
import { CurrencyPipe, DatePipe, DecimalPipe, PercentPipe } from '@angular/common';
import { NG_VALIDATORS } from '@angular/forms';

/**
 * Allows to get the dependencies at the module level or component.
 */
class InjectorRef {
    /**
     * @param {?} injector
     */
    constructor(injector) {
        this.injector = injector;
        InjectorRef.injector = this.injector;
    }
    /**
     * @template T
     * @param {?} token
     * @return {?}
     */
    static get(token) {
        return this.injector.get(token);
    }
}
InjectorRef.decorators = [
    { type: Injectable },
];
/**
 * @nocollapse
 */
InjectorRef.ctorParameters = () => [
    { type: Injector, },
];

class LocaleConfig {
    constructor() {
        this.languageCodes = [];
        this.storageIsDisabled = false;
        this.localStorage = false;
        this.sessionStorage = false;
    }
}
LocaleConfig.decorators = [
    { type: Injectable },
];
/**
 * @nocollapse
 */
LocaleConfig.ctorParameters = () => [];

class LocaleConfigAPI {
    /**
     * @param {?} configuration
     */
    constructor(configuration) {
        this.configuration = configuration;
    }
    /**
     * @param {?} languageCode
     * @param {?=} textDirection
     * @return {?}
     */
    addLanguage(languageCode, textDirection = "LTR") {
        this.configuration.languageCodes.push({ code: languageCode, direction: textDirection });
        return this;
    }
    /**
     * @param {?} languageCodes
     * @return {?}
     */
    addLanguages(languageCodes) {
        for (const /** @type {?} */ languageCode of languageCodes) {
            this.configuration.languageCodes.push({ code: languageCode, direction: "ltr" });
        }
        return this;
    }
    /**
     * @return {?}
     */
    disableStorage() {
        this.configuration.storageIsDisabled = true;
        return this;
    }
    /**
     * @param {?=} days
     * @return {?}
     */
    setCookieExpiration(days) {
        this.configuration.cookiesExpirationDays = days;
        return this;
    }
    /**
     * @return {?}
     */
    useLocalStorage() {
        this.configuration.localStorage = true;
        return this;
    }
    /**
     * @return {?}
     */
    useSessionStorage() {
        this.configuration.sessionStorage = true;
        return this;
    }
    /**
     * @param {?} languageCode
     * @return {?}
     */
    defineLanguage(languageCode) {
        this.configuration.languageCode = languageCode;
        return this;
    }
    /**
     * @param {?} languageCode
     * @param {?} countryCode
     * @param {?=} scriptCode
     * @param {?=} numberingSystem
     * @param {?=} calendar
     * @return {?}
     */
    defineDefaultLocale(languageCode, countryCode, scriptCode, numberingSystem, calendar) {
        this.configuration.languageCode = languageCode;
        this.configuration.countryCode = countryCode;
        this.configuration.scriptCode = scriptCode;
        this.configuration.numberingSystem = numberingSystem;
        this.configuration.calendar = calendar;
        return this;
    }
    /**
     * @param {?} currencyCode
     * @return {?}
     */
    defineCurrency(currencyCode) {
        this.configuration.currencyCode = currencyCode;
        return this;
    }
}

var __awaiter$1 = (undefined && undefined.__awaiter) || function (thisArg, _arguments, P, generator) {
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : new P(function (resolve) { resolve(result.value); }).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
/**
 * Implement this class-interface to create a custom storage for default locale & currency.
 * @abstract
 */
class LocaleStorage {
    /**
     * This method must contain the logic to read the storage.
    \@param name 'defaultLocale' or 'currency'
    \@return A promise with the value of the given name
     * @abstract
     * @param {?} name
     * @return {?}
     */
    read(name) { }
    /**
     * This method must contain the logic to write the storage.
    \@param name 'defaultLocale' or 'currency'
    \@param value The value for the given name
     * @abstract
     * @param {?} name
     * @param {?} value
     * @return {?}
     */
    write(name, value) { }
}
LocaleStorage.decorators = [
    { type: Injectable },
];
/**
 * @nocollapse
 */
LocaleStorage.ctorParameters = () => [];
class BrowserStorage {
    /**
     * @param {?} configuration
     */
    constructor(configuration) {
        this.configuration = configuration;
        this.hasCookie = typeof navigator !== "undefined" && navigator.cookieEnabled;
        this.hasStorage = typeof Storage !== "undefined";
    }
    /**
     * @param {?} name
     * @return {?}
     */
    read(name) {
        return __awaiter$1(this, void 0, void 0, function* () {
            let /** @type {?} */ value = null;
            if (!this.configuration.storageIsDisabled) {
                if (this.configuration.localStorage && this.hasStorage) {
                    value = this.getLocalStorage(name);
                }
                else if (this.configuration.sessionStorage && this.hasStorage) {
                    value = this.getSessionStorage(name);
                }
                else if (this.hasCookie) {
                    value = this.getCookie(name);
                }
            }
            return value;
        });
    }
    /**
     * @param {?} name
     * @param {?} value
     * @return {?}
     */
    write(name, value) {
        return __awaiter$1(this, void 0, void 0, function* () {
            if (!this.configuration.storageIsDisabled) {
                if (this.configuration.localStorage && this.hasStorage) {
                    this.setLocalStorage(name, value);
                }
                else if (this.configuration.sessionStorage && this.hasStorage) {
                    this.setSessionStorage(name, value);
                }
                else if (this.hasCookie) {
                    this.setCookie(name, value);
                }
            }
        });
    }
    /**
     * @param {?} name
     * @return {?}
     */
    getLocalStorage(name) {
        return localStorage.getItem(name);
    }
    /**
     * @param {?} name
     * @return {?}
     */
    getSessionStorage(name) {
        return sessionStorage.getItem(name);
    }
    /**
     * @param {?} name
     * @return {?}
     */
    getCookie(name) {
        let /** @type {?} */ result = null;
        if (typeof document !== "undefined") {
            result = new RegExp("(?:^|; )" + encodeURIComponent(name) + "=([^;]*)").exec(document.cookie);
        }
        return result ? result[1] : null;
    }
    /**
     * @param {?} name
     * @param {?} value
     * @return {?}
     */
    setLocalStorage(name, value) {
        localStorage.setItem(name, value);
    }
    /**
     * @param {?} name
     * @param {?} value
     * @return {?}
     */
    setSessionStorage(name, value) {
        sessionStorage.setItem(name, value);
    }
    /**
     * @param {?} name
     * @param {?} value
     * @return {?}
     */
    setCookie(name, value) {
        let /** @type {?} */ expires = "";
        if (this.configuration.cookiesExpirationDays != null) {
            const /** @type {?} */ expirationDate = new Date();
            expirationDate.setTime(expirationDate.getTime() +
                (this.configuration.cookiesExpirationDays * 24 * 60 * 60 * 1000));
            expires = "; expires=" + expirationDate.toUTCString();
        }
        if (typeof document !== "undefined") {
            document.cookie = name + "=" + value + expires + "; path=/";
        }
    }
}
BrowserStorage.decorators = [
    { type: Injectable },
];
/**
 * @nocollapse
 */
BrowserStorage.ctorParameters = () => [
    { type: LocaleConfig, },
];

class DefaultLocale {
    /**
     * @return {?}
     */
    get value() {
        return this._value;
    }
    /**
     * @param {?} defaultLocale
     * @return {?}
     */
    set value(defaultLocale) {
        this._value = defaultLocale;
        this.parseValue();
    }
    /**
     * @param {?} languageCode
     * @param {?=} countryCode
     * @param {?=} scriptCode
     * @param {?=} numberingSystem
     * @param {?=} calendar
     * @return {?}
     */
    build(languageCode, countryCode, scriptCode, numberingSystem, calendar) {
        this.languageCode = languageCode;
        this.scriptCode = scriptCode;
        this.countryCode = countryCode;
        this.numberingSystem = numberingSystem;
        this.calendar = calendar;
        const /** @type {?} */ value = [];
        value.push(languageCode);
        value.push(!!scriptCode ? "-" + scriptCode : "");
        value.push(!!countryCode ? "-" + countryCode : "");
        // Adds the 'u' (Unicode) extension.
        value.push((!!numberingSystem || !!calendar) ? "-u" : "");
        value.push(!!numberingSystem ? "-nu-" + numberingSystem : "");
        value.push(!!calendar ? "-ca-" + calendar : "");
        this._value = value.join("");
    }
    /**
     * @return {?}
     */
    parseValue() {
        if (!!this.value) {
            let /** @type {?} */ value = this.value;
            // Looks for the 'u' (Unicode) extension.
            const /** @type {?} */ index = value.search("-u");
            if (index != -1) {
                const /** @type {?} */ extensions = value.substring(index + 1).split("-");
                switch (extensions.length) {
                    case 3:
                        if (extensions[1] == "nu") {
                            this.numberingSystem = extensions[2];
                        }
                        else if (extensions[1] == "ca") {
                            this.calendar = extensions[2];
                        }
                        break;
                    default:
                        this.numberingSystem = extensions[2];
                        this.calendar = extensions[4];
                        break;
                }
                // Extracts the codes.
                value = value.substring(0, index);
            }
            const /** @type {?} */ codes = value.split("-");
            switch (codes.length) {
                case 1:
                    this.languageCode = codes[0];
                    break;
                case 2:
                    this.languageCode = codes[0];
                    this.countryCode = codes[1];
                    break;
                default:
                    this.languageCode = codes[0];
                    this.scriptCode = codes[1];
                    this.countryCode = codes[2];
                    break;
            }
        }
    }
}

var __awaiter = (undefined && undefined.__awaiter) || function (thisArg, _arguments, P, generator) {
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : new P(function (resolve) { resolve(result.value); }).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
class LocaleService {
    /**
     * @param {?} configuration
     * @param {?} storage
     */
    constructor(configuration, storage) {
        this.configuration = configuration;
        this.storage = storage;
        this.languageCodeChanged = new EventEmitter(true);
        this.defaultLocaleChanged = new EventEmitter(true);
        this.currencyCodeChanged = new EventEmitter(true);
        this.loadTranslation = new Subject();
        this.defaultLocale = new DefaultLocale();
    }
    /**
     * @return {?}
     */
    addConfiguration() {
        return new LocaleConfigAPI(this.configuration);
    }
    /**
     * @return {?}
     */
    getConfiguration() {
        return this.configuration;
    }
    /**
     * @return {?}
     */
    init() {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.initStorage();
            if (!!this.configuration.languageCode && !!this.configuration.countryCode) {
                this.initDefaultLocale();
            }
            else if (!!this.configuration.languageCode) {
                this.initLanguage();
            }
            if (!!this.configuration.currencyCode) {
                this.initCurrency();
            }
        });
    }
    /**
     * @return {?}
     */
    getAvailableLanguages() {
        return this.configuration.languageCodes.map((language) => language.code);
    }
    /**
     * @param {?=} languageCode
     * @return {?}
     */
    getLanguageDirection(languageCode = this.defaultLocale.languageCode) {
        const /** @type {?} */ matchedLanguages = this.matchLanguage(languageCode);
        return matchedLanguages[0].direction;
    }
    /**
     * @return {?}
     */
    getCurrentLanguage() {
        return this.defaultLocale.languageCode;
    }
    /**
     * @return {?}
     */
    getCurrentCountry() {
        if (!!this.defaultLocale.countryCode) {
            return this.defaultLocale.countryCode;
        }
        return "";
    }
    /**
     * @return {?}
     */
    getCurrentScript() {
        if (!!this.defaultLocale.scriptCode) {
            return this.defaultLocale.scriptCode;
        }
        return "";
    }
    /**
     * @return {?}
     */
    getCurrentLocale() {
        const /** @type {?} */ locale = !!this.defaultLocale.countryCode
            ? this.defaultLocale.languageCode + "-" + this.defaultLocale.countryCode
            : this.defaultLocale.languageCode;
        return locale;
    }
    /**
     * @return {?}
     */
    getCurrentNumberingSystem() {
        if (!!this.defaultLocale.numberingSystem) {
            return this.defaultLocale.numberingSystem;
        }
        return "";
    }
    /**
     * @return {?}
     */
    getCurrentCalendar() {
        if (!!this.defaultLocale.calendar) {
            return this.defaultLocale.calendar;
        }
        return "";
    }
    /**
     * @return {?}
     */
    getDefaultLocale() {
        return this.defaultLocale.value;
    }
    /**
     * @return {?}
     */
    getCurrentCurrency() {
        return this.currencyCode;
    }
    /**
     * @param {?} languageCode
     * @return {?}
     */
    setCurrentLanguage(languageCode) {
        if (this.defaultLocale.languageCode != languageCode) {
            this.defaultLocale.build(languageCode);
            this.storage.write("defaultLocale", this.defaultLocale.value);
            this.sendLanguageEvents();
            this.sendTranslationEvents();
        }
    }
    /**
     * @param {?} languageCode
     * @param {?} countryCode
     * @param {?=} scriptCode
     * @param {?=} numberingSystem
     * @param {?=} calendar
     * @return {?}
     */
    setDefaultLocale(languageCode, countryCode, scriptCode, numberingSystem, calendar) {
        if (this.defaultLocale.languageCode != languageCode ||
            this.defaultLocale.countryCode != countryCode ||
            this.defaultLocale.scriptCode != scriptCode ||
            this.defaultLocale.numberingSystem != numberingSystem ||
            this.defaultLocale.calendar != calendar) {
            this.defaultLocale.build(languageCode, countryCode, scriptCode, numberingSystem, calendar);
            this.storage.write("defaultLocale", this.defaultLocale.value);
            this.sendDefaultLocaleEvents();
            this.sendTranslationEvents();
        }
    }
    /**
     * @param {?} currencyCode
     * @return {?}
     */
    setCurrentCurrency(currencyCode) {
        if (this.currencyCode != currencyCode) {
            this.currencyCode = currencyCode;
            this.storage.write("currency", this.currencyCode);
            this.sendCurrencyEvents();
        }
    }
    /**
     * @return {?}
     */
    initStorage() {
        return __awaiter(this, void 0, void 0, function* () {
            // Tries to retrieve default locale & currency from the browser storage.
            if (!this.defaultLocale.value) {
                const /** @type {?} */ defaultLocale = yield this.storage.read("defaultLocale");
                if (!!defaultLocale) {
                    this.defaultLocale.value = defaultLocale;
                }
            }
            if (this.currencyCode == null) {
                const /** @type {?} */ currencyCode = yield this.storage.read("currency");
                if (!!currencyCode) {
                    this.currencyCode = currencyCode;
                }
            }
        });
    }
    /**
     * @return {?}
     */
    initLanguage() {
        if (!this.defaultLocale.languageCode) {
            const /** @type {?} */ browserLanguage = this.getBrowserLanguage();
            let /** @type {?} */ matchedLanguages = [];
            if (!!browserLanguage) {
                matchedLanguages = this.matchLanguage(browserLanguage);
            }
            if (!!browserLanguage && matchedLanguages.length > 0) {
                this.defaultLocale.build(browserLanguage);
            }
            else {
                this.defaultLocale.build(this.configuration.languageCode);
            }
            this.storage.write("defaultLocale", this.defaultLocale.value);
        }
        this.sendLanguageEvents();
    }
    /**
     * @return {?}
     */
    initDefaultLocale() {
        if (!this.defaultLocale.value) {
            this.defaultLocale.build(this.configuration.languageCode, this.configuration.countryCode, this.configuration.scriptCode, this.configuration.numberingSystem, this.configuration.calendar);
            this.storage.write("defaultLocale", this.defaultLocale.value);
        }
        this.sendDefaultLocaleEvents();
    }
    /**
     * @return {?}
     */
    initCurrency() {
        if (this.currencyCode == null) {
            this.currencyCode = this.configuration.currencyCode;
            this.storage.write("currency", this.currencyCode);
        }
        this.sendCurrencyEvents();
    }
    /**
     * @return {?}
     */
    getBrowserLanguage() {
        let /** @type {?} */ browserLanguage = null;
        if (typeof navigator !== "undefined" && navigator.language) {
            browserLanguage = navigator.language;
        }
        if (browserLanguage != null) {
            const /** @type {?} */ index = browserLanguage.indexOf("-");
            if (index != -1) {
                browserLanguage = browserLanguage.substring(0, index);
            }
        }
        return browserLanguage;
    }
    /**
     * @param {?} languageCode
     * @return {?}
     */
    matchLanguage(languageCode) {
        const /** @type {?} */ matchedLanguages = this.configuration.languageCodes.filter((language) => {
            return language.code == languageCode;
        });
        return matchedLanguages;
    }
    /**
     * @return {?}
     */
    sendLanguageEvents() {
        this.languageCodeChanged.emit(this.defaultLocale.languageCode);
    }
    /**
     * @return {?}
     */
    sendDefaultLocaleEvents() {
        this.defaultLocaleChanged.emit(this.defaultLocale.value);
    }
    /**
     * @return {?}
     */
    sendCurrencyEvents() {
        this.currencyCodeChanged.emit(this.currencyCode);
    }
    /**
     * @return {?}
     */
    sendTranslationEvents() {
        // This event is subscribed by TranslationService to load the translation data.
        this.loadTranslation.next();
    }
}
LocaleService.decorators = [
    { type: Injectable },
];
/**
 * @nocollapse
 */
LocaleService.ctorParameters = () => [
    { type: LocaleConfig, },
    { type: LocaleStorage, },
];
LocaleService.propDecorators = {
    'languageCodeChanged': [{ type: Output },],
    'defaultLocaleChanged': [{ type: Output },],
    'currencyCodeChanged': [{ type: Output },],
};

class TranslationConfig {
    constructor() {
        this.translationData = {};
        this.providers = [];
        this.localeAsLanguage = false;
        this.keySeparator = ".";
        this.i18nPlural = true;
    }
}
TranslationConfig.decorators = [
    { type: Injectable },
];
/**
 * @nocollapse
 */
TranslationConfig.ctorParameters = () => [];

/**
 * Provides the methods to check if Intl APIs are supported.
 */
class IntlAPI {
    /**
     * @return {?}
     */
    static hasDateTimeFormat() {
        return this.hasIntl && Intl.hasOwnProperty("DateTimeFormat");
    }
    /**
     * @return {?}
     */
    static hasNumberFormat() {
        return this.hasIntl && Intl.hasOwnProperty("NumberFormat");
    }
    /**
     * @return {?}
     */
    static hasCollator() {
        return this.hasIntl && Intl.hasOwnProperty("Collator");
    }
}
IntlAPI.hasIntl = Intl && typeof Intl === "object";

class TranslationConfigAPI {
    /**
     * @param {?} configuration
     */
    constructor(configuration) {
        this.configuration = configuration;
    }
    /**
     * @param {?} languageCode
     * @param {?} translation
     * @return {?}
     */
    addTranslation(languageCode, translation) {
        this.configuration.translationData[languageCode] =
            typeof this.configuration.translationData[languageCode] !== "undefined"
                ? Object.assign({}, this.configuration.translationData[languageCode], translation) : translation;
        return this;
    }
    /**
     * @param {?} prefix
     * @param {?=} dataFormat
     * @return {?}
     */
    addProvider(prefix, dataFormat = "json") {
        this.configuration.providers.push({ args: { type: "Static", prefix: prefix, dataFormat: dataFormat } });
        return this;
    }
    /**
     * @param {?} path
     * @param {?=} dataFormat
     * @return {?}
     */
    addWebAPIProvider(path, dataFormat = "json") {
        this.configuration.providers.push({ args: { type: "WebAPI", path: path, dataFormat: dataFormat } });
        return this;
    }
    /**
     * @param {?=} args
     * @return {?}
     */
    addCustomProvider(args) {
        this.configuration.providers.push({ args: args });
        return this;
    }
    /**
     * @return {?}
     */
    useLocaleAsLanguage() {
        this.configuration.localeAsLanguage = true;
        return this;
    }
    /**
     * @param {?} value
     * @return {?}
     */
    setMissingValue(value) {
        this.configuration.missingValue = value;
        return this;
    }
    /**
     * @param {?} key
     * @return {?}
     */
    setMissingKey(key) {
        this.configuration.missingKey = key;
        return this;
    }
    /**
     * @param {?} keySeparator
     * @return {?}
     */
    setComposedKeySeparator(keySeparator) {
        this.configuration.keySeparator = keySeparator;
        return this;
    }
    /**
     * @return {?}
     */
    disableI18nPlural() {
        this.configuration.i18nPlural = false;
        return this;
    }
}

/**
 * Implement this class-interface to create a custom provider for translation data.
 * @abstract
 */
class TranslationProvider {
    /**
     * This method must contain the logic of data access.
     * @abstract
     * @param {?} language The current language
     * @param {?} args The parameter of addCustomProvider method
     * @return {?} An observable of an object of translation data: {key: value}
     */
    getTranslation(language, args) { }
}
TranslationProvider.decorators = [
    { type: Injectable },
];
/**
 * @nocollapse
 */
TranslationProvider.ctorParameters = () => [];
class HttpTranslationProvider {
    /**
     * @param {?} http
     */
    constructor(http) {
        this.http = http;
    }
    /**
     * @param {?} language
     * @param {?} args
     * @return {?}
     */
    getTranslation(language, args) {
        let /** @type {?} */ url = "";
        switch (args.type) {
            case "WebAPI":
                url += args.path + language;
                break;
            default:
                url += args.prefix + language + "." + args.dataFormat;
        }
        return this.http.get(url)
            .map((res) => res.json());
    }
}
HttpTranslationProvider.decorators = [
    { type: Injectable },
];
/**
 * @nocollapse
 */
HttpTranslationProvider.ctorParameters = () => [
    { type: Http, },
];

/**
 * Implement this class-interface to create a custom handler for translated values.
 * @abstract
 */
class TranslationHandler {
    /**
     * This method must contain the logic to parse the translated value.
     * @abstract
     * @param {?} path The path of the key
     * @param {?} key The key that has been requested
     * @param {?} value The translated value
     * @param {?} args The parameters passed along with the key
     * @param {?} lang The current language
     * @return {?} The parsed value
     */
    parseValue(path, key, value, args, lang) { }
}
TranslationHandler.decorators = [
    { type: Injectable },
];
/**
 * @nocollapse
 */
TranslationHandler.ctorParameters = () => [];
class DefaultTranslationHandler {
    /**
     * @param {?} configuration
     */
    constructor(configuration) {
        this.configuration = configuration;
    }
    /**
     * @param {?} path
     * @param {?} key
     * @param {?} value
     * @param {?} args
     * @param {?} lang
     * @return {?}
     */
    parseValue(path, key, value, args, lang) {
        if (value == null) {
            return this.handleMissingValue(path);
        }
        else if (args) {
            return this.handleArgs(value, args);
        }
        return value;
    }
    /**
     * @param {?} path
     * @return {?}
     */
    handleMissingValue(path) {
        if (this.configuration.missingValue != null) {
            return this.configuration.missingValue;
        }
        // The same path is returned.
        return path;
    }
    /**
     * @param {?} value
     * @param {?} args
     * @return {?}
     */
    handleArgs(value, args) {
        const /** @type {?} */ TEMPLATE_REGEXP = /{{\s?([^{}\s]*)\s?}}/g;
        return value.replace(TEMPLATE_REGEXP, (substring, parsedKey) => {
            const /** @type {?} */ replacer = (args[parsedKey]);
            return typeof replacer !== "undefined" ? replacer : substring;
        });
    }
}
DefaultTranslationHandler.decorators = [
    { type: Injectable },
];
/**
 * @nocollapse
 */
DefaultTranslationHandler.ctorParameters = () => [
    { type: TranslationConfig, },
];

let LoadingMode = {};
LoadingMode.Direct = 0;
LoadingMode.Async = 1;
LoadingMode[LoadingMode.Direct] = "Direct";
LoadingMode[LoadingMode.Async] = "Async";
let ServiceState = {};
ServiceState.isReady = 0;
ServiceState.isLoading = 1;
ServiceState.isWaiting = 2;
ServiceState[ServiceState.isReady] = "isReady";
ServiceState[ServiceState.isLoading] = "isLoading";
ServiceState[ServiceState.isWaiting] = "isWaiting";

var __awaiter$2 = (undefined && undefined.__awaiter) || function (thisArg, _arguments, P, generator) {
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : new P(function (resolve) { resolve(result.value); }).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
class TranslationService {
    /**
     * @param {?} locale
     * @param {?} configuration
     * @param {?} translationProvider
     * @param {?} translationHandler
     */
    constructor(locale, configuration, translationProvider, translationHandler) {
        this.locale = locale;
        this.configuration = configuration;
        this.translationProvider = translationProvider;
        this.translationHandler = translationHandler;
        this.translationChanged = new EventEmitter(true);
        this.translationError = new EventEmitter(true);
        /**
         * The translation data: {language: {key: value}}.
         */
        this.translationData = {};
        this.serviceState = ServiceState.isWaiting;
    }
    /**
     * @return {?}
     */
    addConfiguration() {
        return new TranslationConfigAPI(this.configuration);
    }
    /**
     * @return {?}
     */
    getConfiguration() {
        return this.configuration;
    }
    /**
     * @return {?}
     */
    init() {
        return __awaiter$2(this, void 0, void 0, function* () {
            // Waiting for LocaleService initialization.
            yield this.locale.init();
            if (this.configuration.providers.length > 0) {
                this.loadingMode = LoadingMode.Async;
            }
            else {
                this.loadingMode = LoadingMode.Direct;
            }
            // When the language changes, loads translation data.
            this.locale.loadTranslation.subscribe(() => { this.loadTranslation(); });
            yield this.loadTranslation();
        });
    }
    /**
     * @return {?}
     */
    getLanguage() {
        return this.language;
    }
    /**
     * @param {?} keys
     * @param {?=} args
     * @param {?=} lang
     * @return {?}
     */
    translate(keys, args = null, lang = this.language) {
        // If the service is not ready, returns the keys.
        if (this.serviceState != ServiceState.isReady)
            return keys;
        if (Array.isArray(keys)) {
            const /** @type {?} */ data = {};
            for (const /** @type {?} */ key of keys) {
                data[key] = this.translateKey(key, args, lang);
            }
            return data;
        }
        return this.translateKey(keys, args, lang);
    }
    /**
     * @param {?} keys
     * @param {?=} args
     * @param {?=} lang
     * @return {?}
     */
    translateAsync(keys, args, lang = this.language) {
        return Observable.create((observer) => {
            const /** @type {?} */ values = this.translate(keys, args, lang);
            observer.next(values);
            observer.complete();
        });
    }
    /**
     * @param {?} key
     * @param {?} args
     * @param {?} lang
     * @return {?}
     */
    translateKey(key, args, lang) {
        if (key == null || key == "")
            return null;
        // I18n plural.
        if (this.configuration.i18nPlural && /^\d+\b/.exec(key)) {
            return this.translateI18nPlural(key, args, lang);
        }
        return this.getValue(key, args, lang);
    }
    /**
     * @param {?} key
     * @param {?} args
     * @param {?} lang
     * @return {?}
     */
    getValue(key, args, lang) {
        const /** @type {?} */ path = key;
        let /** @type {?} */ value = null;
        if (this.translationData[lang]) {
            let /** @type {?} */ translation = this.translationData[lang];
            // Composed key.
            const /** @type {?} */ sequences = key.split(this.configuration.keySeparator);
            do {
                key = ((sequences.shift()));
                if (translation[key] && typeof translation[key] === "object") {
                    translation = translation[key];
                }
            } while (sequences.length > 0);
            value = translation[key] || translation[this.configuration.missingKey];
        }
        return this.translationHandler.parseValue(path, key, value, args, lang);
    }
    /**
     * @param {?} key
     * @param {?} args
     * @param {?} lang
     * @return {?}
     */
    translateI18nPlural(key, args, lang) {
        let /** @type {?} */ keyText = key.replace(/^\d+\b/, "");
        keyText = keyText.trim();
        const /** @type {?} */ keyNumber = parseFloat(key);
        key = key.replace(/^\d+/, this.translateNumber(keyNumber));
        return key.replace(keyText, this.getValue(keyText, args, lang));
    }
    /**
     * @param {?} keyNumber
     * @return {?}
     */
    translateNumber(keyNumber) {
        if (!isNaN(keyNumber) && IntlAPI.hasNumberFormat()) {
            const /** @type {?} */ localeNumber = new Intl.NumberFormat(this.language).format(keyNumber);
            return localeNumber;
        }
        return keyNumber.toString();
    }
    /**
     * @return {?}
     */
    loadTranslation() {
        return __awaiter$2(this, void 0, void 0, function* () {
            const /** @type {?} */ language = !this.configuration.localeAsLanguage
                ? this.locale.getCurrentLanguage()
                : this.locale.getCurrentLocale();
            if (language != null && language != this.language) {
                if (this.loadingMode == LoadingMode.Async) {
                    yield this.getTranslation(language).toPromise();
                }
                else {
                    this.translationData = {};
                    this.translationData[language] = this.configuration.translationData[language];
                    this.releaseTranslation(language);
                }
            }
        });
    }
    /**
     * @param {?} language
     * @return {?}
     */
    getTranslation(language) {
        return Observable.create((observer) => {
            this.translationData = {};
            this.serviceState = ServiceState.isLoading;
            const /** @type {?} */ sequencesOfTranslationData = [];
            for (const /** @type {?} */ provider of this.configuration.providers) {
                sequencesOfTranslationData.push(this.translationProvider.getTranslation(language, provider.args));
            }
            // Merges all the sequences into a single observable sequence.
            Observable.merge(...sequencesOfTranslationData).subscribe((data) => {
                this.addData(data, language);
            }, (error) => {
                // Sends an event for custom actions.
                this.translationError.emit(error);
                this.releaseTranslation(language);
                observer.next(null);
                observer.complete();
            }, () => {
                this.releaseTranslation(language);
                observer.next(null);
                observer.complete();
            });
        });
    }
    /**
     * @param {?} data
     * @param {?} language
     * @return {?}
     */
    addData(data, language) {
        this.translationData[language] = typeof this.translationData[language] !== "undefined"
            ? Object.assign({}, this.translationData[language], data) : data;
    }
    /**
     * @param {?} language
     * @return {?}
     */
    releaseTranslation(language) {
        this.serviceState = ServiceState.isReady;
        this.language = language;
        this.sendEvents();
    }
    /**
     * @return {?}
     */
    sendEvents() {
        // Sends an event for the components.
        this.translationChanged.emit(this.language);
    }
}
TranslationService.decorators = [
    { type: Injectable },
];
/**
 * @nocollapse
 */
TranslationService.ctorParameters = () => [
    { type: LocaleService, },
    { type: TranslationConfig, },
    { type: TranslationProvider, },
    { type: TranslationHandler, },
];
TranslationService.propDecorators = {
    'translationChanged': [{ type: Output },],
    'translationError': [{ type: Output },],
};

/**
 * Provides 'lang' to the translate pipe.
 */
class Translation {
    /**
     * @param {?=} translation
     * @param {?=} changeDetectorRef
     */
    constructor(translation = InjectorRef.get(TranslationService), changeDetectorRef) {
        this.translation = translation;
        this.changeDetectorRef = changeDetectorRef;
        this.paramSubscriptions = [];
        this.lang = this.translation.getLanguage();
        // When the language changes, subscribes to the event & updates lang property.
        this.paramSubscriptions.push(this.translation.translationChanged.subscribe((language) => {
            this.lang = language;
            // OnPush Change Detection strategy.
            if (this.changeDetectorRef) {
                this.changeDetectorRef.markForCheck();
            }
        }));
    }
    /**
     * @return {?}
     */
    cancelParamSubscriptions() {
        this.paramSubscriptions.forEach((subscription) => {
            if (typeof subscription !== "undefined") {
                subscription.unsubscribe();
            }
        });
    }
}
Translation.decorators = [
    { type: Injectable },
];
/**
 * @nocollapse
 */
Translation.ctorParameters = () => [
    { type: TranslationService, },
    { type: ChangeDetectorRef, },
];

/**
 * Provides 'lang' to the translate pipe,
'defaultLocale' & 'currency' to localeDecimal, localePercent & localeCurrency pipes.
 */
class Localization extends Translation {
    /**
     * @param {?=} locale
     * @param {?=} translation
     * @param {?=} changeDetectorRef
     */
    constructor(locale = InjectorRef.get(LocaleService), translation = InjectorRef.get(TranslationService), changeDetectorRef) {
        super(translation, changeDetectorRef);
        this.locale = locale;
        this.translation = translation;
        this.changeDetectorRef = changeDetectorRef;
        this.defaultLocale = this.locale.getDefaultLocale();
        // When the default locale changes, subscribes to the event & updates defaultLocale property.
        this.paramSubscriptions.push(this.locale.defaultLocaleChanged.subscribe((defaultLocale) => {
            this.defaultLocale = defaultLocale;
            // OnPush Change Detection strategy.
            if (this.changeDetectorRef) {
                this.changeDetectorRef.markForCheck();
            }
        }));
        this.currency = this.locale.getCurrentCurrency();
        // When the currency changes, subscribes to the event & updates currency property.
        this.paramSubscriptions.push(this.locale.currencyCodeChanged.subscribe((currency) => {
            this.currency = currency;
            // OnPush Change Detection strategy.
            if (this.changeDetectorRef) {
                this.changeDetectorRef.markForCheck();
            }
        }));
    }
}
Localization.decorators = [
    { type: Injectable },
];
/**
 * @nocollapse
 */
Localization.ctorParameters = () => [
    { type: LocaleService, },
    { type: TranslationService, },
    { type: ChangeDetectorRef, },
];

/**
 * @abstract
 */
class NumberCode {
    /**
     * @param {?} locale
     */
    constructor(locale) {
        this.locale = locale;
    }
    /**
     * @return {?}
     */
    get numberCodes() {
        const /** @type {?} */ numberCodes = [];
        for (let /** @type {?} */ num = 0; num <= 9; num++) {
            numberCodes.push(this.toUnicode(num.toString()));
        }
        if (IntlAPI.hasNumberFormat()) {
            for (let /** @type {?} */ num = 0; num <= 9; num++) {
                numberCodes[num] = this.toUnicode(new Intl.NumberFormat(this.locale.getDefaultLocale()).format(num));
            }
        }
        return numberCodes;
    }
    /**
     * @abstract
     * @param {?} s
     * @return {?}
     */
    parse(s) { }
    /**
     * @abstract
     * @param {?} digits
     * @return {?}
     */
    getRegExp(digits) { }
    /**
     * @param {?} pattern
     * @return {?}
     */
    toChar(pattern) {
        return pattern.replace(/\\u[\dA-F]{4}/gi, (match) => {
            return String.fromCharCode(parseInt(match.replace(/\\u/g, ""), 16));
        });
    }
    /**
     * @param {?} c
     * @return {?}
     */
    toUnicode(c) {
        return "\\u" + this.toHex(c.charCodeAt(0));
    }
    /**
     * @param {?} value
     * @return {?}
     */
    toHex(value) {
        let /** @type {?} */ hex = value.toString(16).toUpperCase();
        // With padding.
        hex = "0000".substr(0, 4 - hex.length) + hex;
        return hex;
    }
}
NumberCode.decorators = [
    { type: Injectable },
];
/**
 * @nocollapse
 */
NumberCode.ctorParameters = () => [
    { type: LocaleService, },
];

class DecimalCode extends NumberCode {
    /**
     * @param {?} locale
     */
    constructor(locale) {
        super(locale);
        this.locale = locale;
    }
    /**
     * @return {?}
     */
    get decimalCodes() {
        let /** @type {?} */ decimalCodes = {
            minusSign: this.toUnicode("-"),
            decimalSeparator: this.toUnicode("."),
            thousandSeparator: this.toUnicode(",")
        };
        if (IntlAPI.hasNumberFormat()) {
            const /** @type {?} */ value = -1000.9; // Reference value.
            const /** @type {?} */ localeValue = new Intl.NumberFormat(this.locale.getDefaultLocale()).format(value);
            const /** @type {?} */ unicodeChars = [];
            for (let /** @type {?} */ i = 0; i <= 7; i++) {
                unicodeChars.push(this.toUnicode(localeValue.charAt(i)));
            }
            // Right to left:
            // checks Unicode characters 'RIGHT-TO-LEFT MARK' (U+200F) & 'Arabic Letter Mark' (U+061C),
            // or the reverse order.
            // Left to right:
            // checks Unicode character 'LEFT-TO-RIGHT MARK' (U+200E).
            if (unicodeChars[0] == "\\u200F" || unicodeChars[0] == "\\u061C") {
                decimalCodes = {
                    minusSign: unicodeChars[1],
                    decimalSeparator: unicodeChars[7],
                    thousandSeparator: unicodeChars[3]
                };
            }
            else if (unicodeChars[0] == this.toUnicode(new Intl.NumberFormat(this.locale.getDefaultLocale()).format(1))) {
                decimalCodes = {
                    minusSign: unicodeChars[7],
                    decimalSeparator: unicodeChars[5],
                    thousandSeparator: unicodeChars[1]
                };
            }
            else if (unicodeChars[0] == "\\u200E") {
                decimalCodes = {
                    minusSign: unicodeChars[1],
                    decimalSeparator: unicodeChars[7],
                    thousandSeparator: unicodeChars[3]
                };
            }
            else {
                decimalCodes = {
                    minusSign: unicodeChars[0],
                    decimalSeparator: unicodeChars[6],
                    thousandSeparator: unicodeChars[2]
                };
            }
        }
        return decimalCodes;
    }
    /**
     * @param {?} s
     * @return {?}
     */
    parse(s) {
        let /** @type {?} */ value = "";
        const /** @type {?} */ decimalCodes = this.decimalCodes;
        const /** @type {?} */ characters = s.split("");
        for (const /** @type {?} */ char of characters) {
            const /** @type {?} */ charCode = this.toUnicode(char);
            const /** @type {?} */ index = this.numberCodes.indexOf(charCode);
            if (index != -1) {
                value += index;
            }
            else if (charCode == decimalCodes.minusSign) {
                value += "-";
            }
            else if (charCode == decimalCodes.decimalSeparator) {
                value += ".";
            }
            else if (charCode == decimalCodes.thousandSeparator) {
                continue;
            }
            else {
                return NaN;
            }
        }
        return parseFloat(value);
    }
    /**
     * @param {?} digits
     * @return {?}
     */
    getRegExp(digits) {
        let /** @type {?} */ minInt = 1;
        let /** @type {?} */ minFraction = 0;
        let /** @type {?} */ maxFraction = 3;
        if (!!digits) {
            const /** @type {?} */ NUMBER_FORMAT_REGEXP = /^(\d+)?\.((\d+)(\-(\d+))?)?$/;
            const /** @type {?} */ parts = digits.match(NUMBER_FORMAT_REGEXP);
            if (parts != null) {
                if (parts[1] != null) {
                    minInt = parseInt(parts[1]);
                }
                if (parts[3] != null) {
                    minFraction = parseInt(parts[3]);
                }
                if (parts[5] != null) {
                    maxFraction = parseInt(parts[5]);
                }
            }
        }
        const /** @type {?} */ minusSign = this.decimalCodes.minusSign;
        const /** @type {?} */ zero = this.numberCodes[0];
        const /** @type {?} */ decimalSeparator = this.decimalCodes.decimalSeparator;
        const /** @type {?} */ thousandSeparator = this.decimalCodes.thousandSeparator;
        const /** @type {?} */ nine = this.numberCodes[9];
        // Pattern for 1.2-2 digits: /^-?[0-9,]{1,}\.[0-9]{2,2}$/
        // Unicode pattern = "^\u002d?[\u0030-\u0039,\u002c]{1,}\\u002e[\u0030-\u0039]{2,2}$"
        let /** @type {?} */ pattern;
        if (minFraction > 0 && maxFraction > 0) {
            pattern = "^"
                + minusSign
                + "?[" + zero + "-" + nine + thousandSeparator
                + "]{" + minInt + ",}\\"
                + decimalSeparator
                + "[" + zero + "-" + nine
                + "]{" + minFraction + "," + maxFraction
                + "}$";
        }
        else if (minFraction == 0 && maxFraction > 0) {
            // Decimal separator is optional.
            pattern = "^"
                + minusSign
                + "?[" + zero + "-" + nine + thousandSeparator
                + "]{" + minInt + ",}\\"
                + decimalSeparator
                + "?[" + zero + "-" + nine
                + "]{" + minFraction + "," + maxFraction
                + "}$";
        }
        else {
            // Integer number.
            pattern = "^"
                + minusSign
                + "?[" + zero + "-" + nine + thousandSeparator
                + "]{" + minInt + ",}$";
        }
        pattern = this.toChar(pattern);
        const /** @type {?} */ regExp = new RegExp(pattern);
        return regExp;
    }
}
DecimalCode.decorators = [
    { type: Injectable },
];
/**
 * @nocollapse
 */
DecimalCode.ctorParameters = () => [
    { type: LocaleService, },
];

class LocaleValidation {
    /**
     * @param {?} decimalCode
     */
    constructor(decimalCode) {
        this.decimalCode = decimalCode;
    }
    /**
     * @param {?} s
     * @return {?}
     */
    parseNumber(s) {
        if (s == "") {
            return null;
        }
        return this.decimalCode.parse(s);
    }
}
LocaleValidation.decorators = [
    { type: Injectable },
];
/**
 * @nocollapse
 */
LocaleValidation.ctorParameters = () => [
    { type: DecimalCode, },
];

class Collator {
    /**
     * @param {?} locale
     * @param {?} translation
     */
    constructor(locale, translation) {
        this.locale = locale;
        this.translation = translation;
    }
    /**
     * @param {?} key1
     * @param {?} key2
     * @param {?=} extension
     * @param {?=} options
     * @return {?}
     */
    compare(key1, key2, extension, options = { usage: 'sort', sensitivity: 'variant' }) {
        if (!IntlAPI.hasCollator()) {
            return 0;
        }
        const /** @type {?} */ value1 = this.translation.translate(key1);
        const /** @type {?} */ value2 = this.translation.translate(key2);
        const /** @type {?} */ locale = this.addExtension(this.locale.getCurrentLocale(), extension);
        return new Intl.Collator(locale, options).compare(value1, value2);
    }
    /**
     * @param {?} list
     * @param {?} keyName
     * @param {?=} order
     * @param {?=} extension
     * @param {?=} options
     * @return {?}
     */
    sort(list, keyName, order = "asc", extension, options = { usage: 'sort', sensitivity: 'variant' }) {
        if (!list || !keyName || !IntlAPI.hasCollator()) {
            return list;
        }
        list.sort((key1, key2) => {
            return this.compare(key1[keyName], key2[keyName], extension, options);
        });
        if (order == "desc") {
            list.reverse();
        }
        return list;
    }
    /**
     * @param {?} list
     * @param {?} keyName
     * @param {?=} order
     * @param {?=} extension
     * @param {?=} options
     * @return {?}
     */
    sortAsync(list, keyName, order, extension, options = { usage: 'sort', sensitivity: 'variant' }) {
        return Observable.create((observer) => {
            observer.next(this.sort(list, keyName, order, extension, options));
            observer.complete();
        });
    }
    /**
     * @param {?} s
     * @param {?} list
     * @param {?} keyNames
     * @param {?=} options
     * @return {?}
     */
    search(s, list, keyNames, options = { usage: 'search' }) {
        if (!list || !keyNames || s == "" || s == null || !IntlAPI.hasCollator()) {
            return list;
        }
        const /** @type {?} */ locale = this.locale.getCurrentLocale();
        const /** @type {?} */ collator = new Intl.Collator(locale, options);
        const /** @type {?} */ matches = list.filter((key) => {
            let /** @type {?} */ found = false;
            for (let /** @type {?} */ i = 0; i < keyNames.length; i++) {
                if (this.match(key[keyNames[i]], s, collator)) {
                    found = true;
                    break;
                }
            }
            return found;
        });
        return matches;
    }
    /**
     * @param {?} s
     * @param {?} list
     * @param {?} keyNames
     * @param {?=} options
     * @return {?}
     */
    searchAsync(s, list, keyNames, options = { usage: 'search' }) {
        return Observable.create((observer) => {
            observer.next(this.search(s, list, keyNames, options));
            observer.complete();
        });
    }
    /**
     * @param {?} locale
     * @param {?=} extension
     * @return {?}
     */
    addExtension(locale, extension) {
        if (!!extension) {
            locale = locale + "-u-" + extension;
        }
        return locale;
    }
    /**
     * @param {?} key
     * @param {?} s
     * @param {?} collator
     * @return {?}
     */
    match(key, s, collator) {
        const /** @type {?} */ value = this.translation.translate(key);
        const /** @type {?} */ valueLength = value.length;
        const /** @type {?} */ sLength = s.length;
        if (sLength > valueLength) {
            return false;
        }
        if (sLength == valueLength) {
            return collator.compare(value, s) == 0;
        }
        let /** @type {?} */ found = false;
        for (let /** @type {?} */ i = 0; i < valueLength - (sLength - 1); i++) {
            const /** @type {?} */ str = value.substr(i, sLength);
            if (collator.compare(str, s) == 0) {
                found = true;
                break;
            }
        }
        return found;
    }
}
Collator.decorators = [
    { type: Injectable },
];
/**
 * @nocollapse
 */
Collator.ctorParameters = () => [
    { type: LocaleService, },
    { type: TranslationService, },
];

/**
 * Property decorator for components to provide the parameter to the translate pipe.
 * @return {?}
 */
function Language() {
    /**
     * @param {?} target
     * @param {?=} propertyKey
     * @return {?}
     */
    function DecoratorFactory(target, propertyKey) {
        let /** @type {?} */ subscription;
        const /** @type {?} */ targetNgOnInit = target.ngOnInit;
        /**
         * @this {?}
         * @return {?}
         */
        function ngOnInit() {
            const /** @type {?} */ translation = InjectorRef.get(TranslationService);
            if (typeof propertyKey !== "undefined") {
                this[propertyKey] = translation.getLanguage();
                // When the language changes, subscribes to the event & updates language property.
                subscription = translation.translationChanged.subscribe((language) => {
                    this[propertyKey] = language;
                });
            }
            if (targetNgOnInit) {
                targetNgOnInit.apply(this);
            }
        }
        target.ngOnInit = ngOnInit;
        const /** @type {?} */ targetNgOnDestroy = target.ngOnDestroy;
        /**
         * @this {?}
         * @return {?}
         */
        function ngOnDestroy() {
            if (typeof subscription !== "undefined") {
                subscription.unsubscribe();
            }
            if (targetNgOnDestroy) {
                targetNgOnDestroy.apply(this);
            }
        }
        target.ngOnDestroy = ngOnDestroy;
        if (typeof propertyKey !== "undefined") {
            Object.defineProperty(target, propertyKey, {
                writable: true,
                value: undefined
            });
        }
    }
    return DecoratorFactory;
}

/**
 * Property decorator for components to provide the parameter
to localeDecimal, localePercent & localeCurrency pipes.
 * @return {?}
 */
function DefaultLocale$1() {
    /**
     * @param {?} target
     * @param {?=} propertyKey
     * @return {?}
     */
    function DecoratorFactory(target, propertyKey) {
        let /** @type {?} */ subscription;
        const /** @type {?} */ targetNgOnInit = target.ngOnInit;
        /**
         * @this {?}
         * @return {?}
         */
        function ngOnInit() {
            const /** @type {?} */ locale = InjectorRef.get(LocaleService);
            if (typeof propertyKey !== "undefined") {
                this[propertyKey] = locale.getDefaultLocale();
                // When the default locale changes, subscribes to the event & updates defaultLocale property.
                subscription = locale.defaultLocaleChanged.subscribe((defaultLocale) => {
                    this[propertyKey] = defaultLocale;
                });
            }
            if (targetNgOnInit) {
                targetNgOnInit.apply(this);
            }
        }
        target.ngOnInit = ngOnInit;
        const /** @type {?} */ targetNgOnDestroy = target.ngOnDestroy;
        /**
         * @this {?}
         * @return {?}
         */
        function ngOnDestroy() {
            if (typeof subscription !== "undefined") {
                subscription.unsubscribe();
            }
            if (targetNgOnDestroy) {
                targetNgOnDestroy.apply(this);
            }
        }
        target.ngOnDestroy = ngOnDestroy;
        if (typeof propertyKey !== "undefined") {
            Object.defineProperty(target, propertyKey, {
                writable: true,
                value: undefined
            });
        }
    }
    return DecoratorFactory;
}

/**
 * Property decorator for components to provide the parameter to the localeCurrency pipe.
 * @return {?}
 */
function Currency() {
    /**
     * @param {?} target
     * @param {?=} propertyKey
     * @return {?}
     */
    function DecoratorFactory(target, propertyKey) {
        let /** @type {?} */ subscription;
        const /** @type {?} */ targetNgOnInit = target.ngOnInit;
        /**
         * @this {?}
         * @return {?}
         */
        function ngOnInit() {
            const /** @type {?} */ locale = InjectorRef.get(LocaleService);
            if (typeof propertyKey !== "undefined") {
                this[propertyKey] = locale.getCurrentCurrency();
                // When the currency changes, subscribes to the event & updates currency property.
                subscription = locale.currencyCodeChanged.subscribe((currency) => {
                    this[propertyKey] = currency;
                });
            }
            if (targetNgOnInit) {
                targetNgOnInit.apply(this);
            }
        }
        target.ngOnInit = ngOnInit;
        const /** @type {?} */ targetNgOnDestroy = target.ngOnDestroy;
        /**
         * @this {?}
         * @return {?}
         */
        function ngOnDestroy() {
            if (typeof subscription !== "undefined") {
                subscription.unsubscribe();
            }
            if (targetNgOnDestroy) {
                targetNgOnDestroy.apply(this);
            }
        }
        target.ngOnDestroy = ngOnDestroy;
        if (typeof propertyKey !== "undefined") {
            Object.defineProperty(target, propertyKey, {
                writable: true,
                value: undefined
            });
        }
    }
    return DecoratorFactory;
}

class TranslatePipe {
    /**
     * @param {?} translation
     */
    constructor(translation) {
        this.translation = translation;
    }
    /**
     * @param {?} key
     * @param {?} lang
     * @param {?=} args
     * @return {?}
     */
    transform(key, lang, args) {
        return this.translation.translate(key, args, lang);
    }
}
TranslatePipe.decorators = [
    { type: Pipe, args: [{
                name: 'translate',
                pure: true
            },] },
];
/**
 * @nocollapse
 */
TranslatePipe.ctorParameters = () => [
    { type: TranslationService, },
];

class LocaleDatePipe {
    /**
     * @param {?} value
     * @param {?} defaultLocale
     * @param {?=} pattern
     * @return {?}
     */
    transform(value, defaultLocale, pattern = 'mediumDate') {
        if (typeof defaultLocale === "undefined")
            return null;
        if (IntlAPI.hasDateTimeFormat()) {
            const /** @type {?} */ localeDate = new DatePipe(defaultLocale);
            return localeDate.transform(value, pattern);
        }
        // Returns the date without localization.
        return value;
    }
}
LocaleDatePipe.decorators = [
    { type: Pipe, args: [{
                name: 'localeDate',
                pure: true
            },] },
];
/**
 * @nocollapse
 */
LocaleDatePipe.ctorParameters = () => [];

class LocaleDecimalPipe {
    /**
     * @param {?} value
     * @param {?} defaultLocale
     * @param {?=} digits
     * @return {?}
     */
    transform(value, defaultLocale, digits) {
        if (typeof defaultLocale === "undefined")
            return null;
        if (IntlAPI.hasNumberFormat()) {
            const /** @type {?} */ localeDecimal = new DecimalPipe(defaultLocale);
            return localeDecimal.transform(value, digits);
        }
        // Returns the number without localization.
        return value;
    }
}
LocaleDecimalPipe.decorators = [
    { type: Pipe, args: [{
                name: 'localeDecimal',
                pure: true
            },] },
];
/**
 * @nocollapse
 */
LocaleDecimalPipe.ctorParameters = () => [];
class LocalePercentPipe {
    /**
     * @param {?} value
     * @param {?} defaultLocale
     * @param {?=} digits
     * @return {?}
     */
    transform(value, defaultLocale, digits) {
        if (typeof defaultLocale === "undefined")
            return null;
        if (IntlAPI.hasNumberFormat()) {
            const /** @type {?} */ localePercent = new PercentPipe(defaultLocale);
            return localePercent.transform(value, digits);
        }
        // Returns the number without localization.
        return value;
    }
}
LocalePercentPipe.decorators = [
    { type: Pipe, args: [{
                name: 'localePercent',
                pure: true
            },] },
];
/**
 * @nocollapse
 */
LocalePercentPipe.ctorParameters = () => [];
class LocaleCurrencyPipe {
    /**
     * @param {?} value
     * @param {?} defaultLocale
     * @param {?} currency
     * @param {?=} symbolDisplay
     * @param {?=} digits
     * @return {?}
     */
    transform(value, defaultLocale, currency, symbolDisplay = false, digits) {
        if (typeof defaultLocale === "undefined" || typeof currency === "undefined")
            return null;
        if (IntlAPI.hasNumberFormat()) {
            const /** @type {?} */ localeCurrency = new CurrencyPipe(defaultLocale);
            return localeCurrency.transform(value, currency, symbolDisplay, digits);
        }
        // Returns the number & currency without localization.
        return value + " " + currency;
    }
}
LocaleCurrencyPipe.decorators = [
    { type: Pipe, args: [{
                name: 'localeCurrency',
                pure: true
            },] },
];
/**
 * @nocollapse
 */
LocaleCurrencyPipe.ctorParameters = () => [];

/**
 * Breadth First Search (BFS) algorithm for traversing & searching tree data structure of DOM
 * explores the neighbor nodes first, before moving to the next level neighbors.
 * Time complexity: between O(1) and O(|V|^2).
 */
class BFS {
    /**
     * Target node is a non empty text node.
     * @param {?} rootNode
     * @return {?}
     */
    static getTargetNode(rootNode) {
        return this.walk(rootNode);
    }
    /**
     * @param {?} rootNode
     * @return {?}
     */
    static walk(rootNode) {
        const /** @type {?} */ queue = [];
        let /** @type {?} */ iNode;
        let /** @type {?} */ depth = 0;
        let /** @type {?} */ nodeToDepthIncrease = 1;
        queue.push(rootNode);
        while (queue.length > 0 && depth <= this.MAX_DEPTH) {
            iNode = queue.shift();
            if (this.isTargetNode(iNode)) {
                return iNode;
            }
            if (depth < this.MAX_DEPTH && iNode.childNodes != null) {
                for (const /** @type {?} */ child of iNode.childNodes) {
                    if (this.isValidNode(child)) {
                        queue.push(child);
                    }
                }
            }
            if (--nodeToDepthIncrease == 0) {
                depth++;
                nodeToDepthIncrease = queue.length;
            }
        }
        return rootNode;
    }
    /**
     * @param {?} node
     * @return {?}
     */
    static isTargetNode(node) {
        return typeof node !== "undefined" &&
            node.nodeType == 3 &&
            node.nodeValue != null &&
            node.nodeValue.trim() != "";
    }
    /**
     * A valid node is not marked for translation.
     * @param {?} node
     * @return {?}
     */
    static isValidNode(node) {
        if (typeof node !== "undefined" && node.nodeType == 1 && node.attributes) {
            for (const /** @type {?} */ attr of node.attributes) {
                if (this.SELECTOR.test(attr.name))
                    return false;
            }
        }
        return true;
    }
}
BFS.SELECTOR = /^l10n|translate|locale/;
BFS.MAX_DEPTH = 10;

/**
 * @abstract
 */
class BaseDirective {
    /**
     * @param {?} el
     * @param {?} renderer
     */
    constructor(el, renderer) {
        this.el = el;
        this.renderer = renderer;
        this.attributes = [];
        this.subscriptions = [];
        this.TEXT_MUTATION_CONFIG = { subtree: true, characterData: true };
        this.SELECTOR = /^l10n-/;
    }
    /**
     * @return {?}
     */
    ngAfterViewInit() {
        if (this.el && this.el.nativeElement) {
            this.element = this.el.nativeElement;
            this.renderNode = BFS.getTargetNode(this.element);
            this.getKey();
            this.getAttributes();
            this.addTextListener();
            this.setup();
        }
    }
    /**
     * @param {?} changes
     * @return {?}
     */
    ngOnChanges(changes) {
        if (!!this.key) {
            if (this.nodeValue == null || this.nodeValue == "") {
                if (!!this.valueAttribute) {
                    this.key = this.valueAttribute;
                }
                else if (!!this.innerHTMLProperty) {
                    this.key = this.innerHTMLProperty;
                }
            }
            this.replaceText();
        }
        if (this.attributes.length > 0) {
            this.replaceAttributes();
        }
    }
    /**
     * @return {?}
     */
    ngOnDestroy() {
        this.removeTextListener();
        this.cancelSubscriptions();
    }
    /**
     * @abstract
     * @return {?}
     */
    setup() { }
    /**
     * @abstract
     * @return {?}
     */
    replace() { }
    /**
     * @abstract
     * @return {?}
     */
    replaceText() { }
    /**
     * @abstract
     * @return {?}
     */
    replaceAttributes() { }
    /**
     * @return {?}
     */
    getAttributesData() {
        const /** @type {?} */ keys = this.getAttributesKeys();
        const /** @type {?} */ data = {};
        for (const /** @type {?} */ key of keys) {
            data[key] = this.getValues(key);
        }
        return data;
    }
    /**
     * @return {?}
     */
    getAttributesKeys() {
        return this.attributes.map((attr) => attr.key);
    }
    /**
     * @abstract
     * @param {?} keys
     * @return {?}
     */
    getValues(keys) { }
    /**
     * @param {?} value
     * @return {?}
     */
    setText(value) {
        if (!!value) {
            if (!!this.nodeValue && !!this.key) {
                this.removeTextListener();
                this.renderer.setValue(this.renderNode, this.nodeValue.replace(this.key, value));
                this.addTextListener();
            }
            else if (!!this.valueAttribute) {
                this.renderer.setAttribute(this.element, "value", value);
            }
            else if (!!this.innerHTMLProperty) {
                this.renderer.setProperty(this.element, "innerHTML", value);
            }
        }
    }
    /**
     * @param {?} data
     * @return {?}
     */
    setAttributes(data) {
        for (const /** @type {?} */ attr of this.attributes) {
            this.renderer.setAttribute(this.element, attr.name, data[attr.key]);
        }
    }
    /**
     * @return {?}
     */
    addTextListener() {
        if (typeof MutationObserver !== "undefined") {
            this.textObserver = new MutationObserver((mutations) => {
                this.renderNode = BFS.getTargetNode(this.element);
                this.getKey();
                this.replaceText();
            });
            this.textObserver.observe(this.renderNode, this.TEXT_MUTATION_CONFIG);
        }
    }
    /**
     * @return {?}
     */
    removeTextListener() {
        if (typeof this.textObserver !== "undefined") {
            this.textObserver.disconnect();
        }
    }
    /**
     * @return {?}
     */
    getText() {
        this.nodeValue = this.renderNode != null ? (this.renderNode.nodeValue) : "";
        return !!this.nodeValue ? this.nodeValue.trim() : "";
    }
    /**
     * @return {?}
     */
    getKey() {
        if (this.element.childNodes.length > 0) {
            this.key = this.getText();
        }
        else if (!!this.valueAttribute) {
            this.key = this.valueAttribute;
        }
        else if (!!this.innerHTMLProperty) {
            this.key = this.innerHTMLProperty;
        }
    }
    /**
     * @return {?}
     */
    getAttributes() {
        if (this.element.attributes) {
            for (const /** @type {?} */ attr of this.element.attributes) {
                if (this.SELECTOR.test(attr.name)) {
                    const /** @type {?} */ name = attr.name.substr(5);
                    for (const /** @type {?} */ targetAttr of this.element.attributes) {
                        if (new RegExp("^" + name + "$").test(targetAttr.name)) {
                            this.attributes.push({ name: name, key: targetAttr.value });
                        }
                    }
                }
            }
        }
    }
    /**
     * @return {?}
     */
    cancelSubscriptions() {
        this.subscriptions.forEach((subscription) => {
            if (typeof subscription !== "undefined") {
                subscription.unsubscribe();
            }
        });
    }
}
BaseDirective.propDecorators = {
    'valueAttribute': [{ type: Input, args: ['value',] },],
    'innerHTMLProperty': [{ type: Input, args: ['innerHTML',] },],
};

class TranslateDirective extends BaseDirective {
    /**
     * @param {?} translation
     * @param {?} el
     * @param {?} renderer
     */
    constructor(translation, el, renderer) {
        super(el, renderer);
        this.translation = translation;
        this.el = el;
        this.renderer = renderer;
    }
    /**
     * @param {?} params
     * @return {?}
     */
    set l10nTranslate(params) {
        this.params = params;
    }
    /**
     * @param {?} params
     * @return {?}
     */
    set translate(params) {
        this.params = params;
    }
    /**
     * @return {?}
     */
    setup() {
        this.replace();
        this.subscriptions.push(this.translation.translationChanged.subscribe(() => { this.replace(); }));
    }
    /**
     * @return {?}
     */
    replace() {
        this.replaceText();
        this.replaceAttributes();
    }
    /**
     * @return {?}
     */
    replaceText() {
        if (!!this.key) {
            this.setText(this.getValues(this.key));
        }
    }
    /**
     * @return {?}
     */
    replaceAttributes() {
        if (this.attributes.length > 0) {
            const /** @type {?} */ keys = this.getAttributesKeys();
            this.setAttributes(this.getValues(keys));
        }
    }
    /**
     * @param {?} keys
     * @return {?}
     */
    getValues(keys) {
        return this.translation.translate(keys, this.params);
    }
}
TranslateDirective.decorators = [
    { type: Directive, args: [{
                selector: '[l10nTranslate],[translate]'
            },] },
];
/**
 * @nocollapse
 */
TranslateDirective.ctorParameters = () => [
    { type: TranslationService, },
    { type: ElementRef, },
    { type: Renderer2, },
];
TranslateDirective.propDecorators = {
    'l10nTranslate': [{ type: Input },],
    'translate': [{ type: Input },],
};

class LocaleDateDirective extends BaseDirective {
    /**
     * @param {?} locale
     * @param {?} el
     * @param {?} renderer
     */
    constructor(locale, el, renderer) {
        super(el, renderer);
        this.locale = locale;
        this.el = el;
        this.renderer = renderer;
        this.defaultPattern = 'mediumDate';
        this.localeDatePipe = new LocaleDatePipe();
    }
    /**
     * @param {?} pattern
     * @return {?}
     */
    set l10nDate(pattern) {
        this.pattern = pattern;
    }
    /**
     * @param {?} pattern
     * @return {?}
     */
    set localeDate(pattern) {
        this.pattern = pattern;
    }
    /**
     * @return {?}
     */
    setup() {
        this.replace();
        this.subscriptions.push(this.locale.defaultLocaleChanged.subscribe(() => { this.replace(); }));
    }
    /**
     * @return {?}
     */
    replace() {
        this.replaceText();
        this.replaceAttributes();
    }
    /**
     * @return {?}
     */
    replaceText() {
        if (!!this.key) {
            this.setText(this.getValues(this.key));
        }
    }
    /**
     * @return {?}
     */
    replaceAttributes() {
        if (this.attributes.length > 0) {
            this.setAttributes(this.getAttributesData());
        }
    }
    /**
     * @param {?} keys
     * @return {?}
     */
    getValues(keys) {
        return this.localeDatePipe.transform(keys, this.locale.getDefaultLocale(), this.pattern || this.defaultPattern);
    }
}
LocaleDateDirective.decorators = [
    { type: Directive, args: [{
                selector: '[l10nDate],[localeDate]'
            },] },
];
/**
 * @nocollapse
 */
LocaleDateDirective.ctorParameters = () => [
    { type: LocaleService, },
    { type: ElementRef, },
    { type: Renderer2, },
];
LocaleDateDirective.propDecorators = {
    'l10nDate': [{ type: Input },],
    'localeDate': [{ type: Input },],
};

class LocaleDecimalDirective extends BaseDirective {
    /**
     * @param {?} locale
     * @param {?} el
     * @param {?} renderer
     */
    constructor(locale, el, renderer) {
        super(el, renderer);
        this.locale = locale;
        this.el = el;
        this.renderer = renderer;
        this.localeDecimalPipe = new LocaleDecimalPipe();
    }
    /**
     * @param {?} digits
     * @return {?}
     */
    set l10nDecimal(digits) {
        this.digits = digits;
    }
    /**
     * @param {?} digits
     * @return {?}
     */
    set localeDecimal(digits) {
        this.digits = digits;
    }
    /**
     * @return {?}
     */
    setup() {
        this.replace();
        this.subscriptions.push(this.locale.defaultLocaleChanged.subscribe(() => { this.replace(); }));
    }
    /**
     * @return {?}
     */
    replace() {
        this.replaceText();
        this.replaceAttributes();
    }
    /**
     * @return {?}
     */
    replaceText() {
        if (!!this.key) {
            this.setText(this.getValues(this.key));
        }
    }
    /**
     * @return {?}
     */
    replaceAttributes() {
        if (this.attributes.length > 0) {
            this.setAttributes(this.getAttributesData());
        }
    }
    /**
     * @param {?} keys
     * @return {?}
     */
    getValues(keys) {
        return this.localeDecimalPipe.transform(keys, this.locale.getDefaultLocale(), this.digits);
    }
}
LocaleDecimalDirective.decorators = [
    { type: Directive, args: [{
                selector: '[l10nDecimal],[localeDecimal]'
            },] },
];
/**
 * @nocollapse
 */
LocaleDecimalDirective.ctorParameters = () => [
    { type: LocaleService, },
    { type: ElementRef, },
    { type: Renderer2, },
];
LocaleDecimalDirective.propDecorators = {
    'l10nDecimal': [{ type: Input },],
    'localeDecimal': [{ type: Input },],
};
class LocalePercentDirective extends BaseDirective {
    /**
     * @param {?} locale
     * @param {?} el
     * @param {?} renderer
     */
    constructor(locale, el, renderer) {
        super(el, renderer);
        this.locale = locale;
        this.el = el;
        this.renderer = renderer;
        this.localePercentPipe = new LocalePercentPipe();
    }
    /**
     * @param {?} digits
     * @return {?}
     */
    set l10nPercent(digits) {
        this.digits = digits;
    }
    /**
     * @param {?} digits
     * @return {?}
     */
    set localePercent(digits) {
        this.digits = digits;
    }
    /**
     * @return {?}
     */
    setup() {
        this.replace();
        this.subscriptions.push(this.locale.defaultLocaleChanged.subscribe(() => { this.replace(); }));
    }
    /**
     * @return {?}
     */
    replace() {
        this.replaceText();
        this.replaceAttributes();
    }
    /**
     * @return {?}
     */
    replaceText() {
        if (!!this.key) {
            this.setText(this.getValues(this.key));
        }
    }
    /**
     * @return {?}
     */
    replaceAttributes() {
        if (this.attributes.length > 0) {
            this.setAttributes(this.getAttributesData());
        }
    }
    /**
     * @param {?} keys
     * @return {?}
     */
    getValues(keys) {
        return this.localePercentPipe.transform(keys, this.locale.getDefaultLocale(), this.digits);
    }
}
LocalePercentDirective.decorators = [
    { type: Directive, args: [{
                selector: '[l10nPercent],[localePercent]'
            },] },
];
/**
 * @nocollapse
 */
LocalePercentDirective.ctorParameters = () => [
    { type: LocaleService, },
    { type: ElementRef, },
    { type: Renderer2, },
];
LocalePercentDirective.propDecorators = {
    'l10nPercent': [{ type: Input },],
    'localePercent': [{ type: Input },],
};
class LocaleCurrencyDirective extends BaseDirective {
    /**
     * @param {?} locale
     * @param {?} el
     * @param {?} renderer
     */
    constructor(locale, el, renderer) {
        super(el, renderer);
        this.locale = locale;
        this.el = el;
        this.renderer = renderer;
        this.localeCurrencyPipe = new LocaleCurrencyPipe();
    }
    /**
     * @param {?} digits
     * @return {?}
     */
    set l10nCurrency(digits) {
        this.digits = digits;
    }
    /**
     * @param {?} digits
     * @return {?}
     */
    set localeCurrency(digits) {
        this.digits = digits;
    }
    /**
     * @return {?}
     */
    setup() {
        this.replace();
        this.subscriptions.push(this.locale.defaultLocaleChanged.subscribe(() => { this.replace(); }));
        this.subscriptions.push(this.locale.currencyCodeChanged.subscribe(() => { this.replace(); }));
    }
    /**
     * @return {?}
     */
    replace() {
        this.replaceText();
        this.replaceAttributes();
    }
    /**
     * @return {?}
     */
    replaceText() {
        if (!!this.key) {
            this.setText(this.getValues(this.key));
        }
    }
    /**
     * @return {?}
     */
    replaceAttributes() {
        if (this.attributes.length > 0) {
            this.setAttributes(this.getAttributesData());
        }
    }
    /**
     * @param {?} keys
     * @return {?}
     */
    getValues(keys) {
        return this.localeCurrencyPipe.transform(keys, this.locale.getDefaultLocale(), this.locale.getCurrentCurrency(), this.symbol, this.digits);
    }
}
LocaleCurrencyDirective.decorators = [
    { type: Directive, args: [{
                selector: '[l10nCurrency],[localeCurrency]'
            },] },
];
/**
 * @nocollapse
 */
LocaleCurrencyDirective.ctorParameters = () => [
    { type: LocaleService, },
    { type: ElementRef, },
    { type: Renderer2, },
];
LocaleCurrencyDirective.propDecorators = {
    'l10nCurrency': [{ type: Input },],
    'localeCurrency': [{ type: Input },],
    'symbol': [{ type: Input },],
};

/**
 * Function that takes a control and returns either null when it’s valid, or an error object if it’s not.
 * @param {?} digits The format of the number
 * @param {?=} MIN_VALUE The minimum value for the number
 * @param {?=} MAX_VALUE The maximum value for the number
 * @return {?} An error object: 'format', 'minValue' or 'maxValue'; null in case the value is valid
 */
function validateLocaleNumber(digits, MIN_VALUE = Number.MIN_VALUE, MAX_VALUE = Number.MAX_VALUE) {
    const /** @type {?} */ locale = InjectorRef.get(LocaleService);
    const /** @type {?} */ decimalCode = InjectorRef.get(DecimalCode);
    let /** @type {?} */ defaultLocale;
    let /** @type {?} */ NUMBER_REGEXP;
    return (formControl) => {
        if (formControl.value == null || formControl.value == "")
            return null;
        if (defaultLocale != locale.getDefaultLocale()) {
            NUMBER_REGEXP = decimalCode.getRegExp(digits);
            defaultLocale = locale.getDefaultLocale();
        }
        if (NUMBER_REGEXP.test(formControl.value)) {
            const /** @type {?} */ parsedValue = decimalCode.parse(formControl.value);
            if (parsedValue != null && parsedValue < MIN_VALUE) {
                return {
                    minValue: {
                        valid: false
                    }
                };
            }
            else if (parsedValue != null && parsedValue > MAX_VALUE) {
                return {
                    maxValue: {
                        valid: false
                    }
                };
            }
            return null; // The number is valid.
        }
        else {
            return {
                format: {
                    valid: false
                }
            };
        }
    };
}
class LocaleNumberValidatorDirective {
    constructor() {
        this.MIN_VALUE = Number.MIN_VALUE;
        this.MAX_VALUE = Number.MAX_VALUE;
    }
    /**
     * Format: {minIntegerDigits}.{minFractionDigits}-{maxFractionDigits}.
     * @param {?} digits
     * @return {?}
     */
    set l10nValidateNumber(digits) {
        this.digits = digits;
    }
    /**
     * @param {?} digits
     * @return {?}
     */
    set validateLocaleNumber(digits) {
        this.digits = digits;
    }
    /**
     * @return {?}
     */
    ngOnInit() {
        this.validator = validateLocaleNumber(this.digits, this.minValue || this.MIN_VALUE, this.maxValue || this.MAX_VALUE);
    }
    /**
     * @param {?} formControl
     * @return {?}
     */
    validate(formControl) {
        return this.validator(formControl);
    }
}
LocaleNumberValidatorDirective.decorators = [
    { type: Directive, args: [{
                selector: '[l10nValidateNumber][ngModel],[l10nValidateNumber][formControl], \
        [validateLocaleNumber][ngModel],[validateLocaleNumber][formControl]',
                providers: [
                    { provide: NG_VALIDATORS, useExisting: forwardRef(() => LocaleNumberValidatorDirective), multi: true }
                ]
            },] },
];
/**
 * @nocollapse
 */
LocaleNumberValidatorDirective.ctorParameters = () => [];
LocaleNumberValidatorDirective.propDecorators = {
    'l10nValidateNumber': [{ type: Input },],
    'validateLocaleNumber': [{ type: Input },],
    'minValue': [{ type: Input },],
    'maxValue': [{ type: Input },],
};

/**
 * Provides dependencies, pipes & directives for translating messages.
 */
class TranslationModule {
    /**
     * @param {?} injector
     */
    constructor(injector) {
        this.injector = injector;
        // Creates the instance of the InjectorRef, so that module dependencies are available.
    }
    /**
     * Use in AppModule: new instances of LocaleService & TranslationService.
     * @param {?=} token
     * @return {?}
     */
    static forRoot(token = {}) {
        return {
            ngModule: TranslationModule,
            providers: [
                InjectorRef,
                LocaleConfig,
                LocaleService,
                {
                    provide: LocaleStorage,
                    useClass: token.localeStorage || BrowserStorage
                },
                TranslationConfig,
                TranslationService,
                {
                    provide: TranslationProvider,
                    useClass: token.translationProvider || HttpTranslationProvider
                },
                {
                    provide: TranslationHandler,
                    useClass: token.translationHandler || DefaultTranslationHandler
                }
            ]
        };
    }
    /**
     * Use in feature modules with lazy loading: new instance of TranslationService.
     * @param {?=} token
     * @return {?}
     */
    static forChild(token = {}) {
        return {
            ngModule: TranslationModule,
            providers: [
                InjectorRef,
                TranslationConfig,
                TranslationService,
                {
                    provide: TranslationProvider,
                    useClass: token.translationProvider || HttpTranslationProvider
                },
                {
                    provide: TranslationHandler,
                    useClass: token.translationHandler || DefaultTranslationHandler
                }
            ]
        };
    }
}
TranslationModule.decorators = [
    { type: NgModule, args: [{
                declarations: [
                    TranslatePipe,
                    TranslateDirective
                ],
                exports: [
                    TranslatePipe,
                    TranslateDirective
                ]
            },] },
];
/**
 * @nocollapse
 */
TranslationModule.ctorParameters = () => [
    { type: InjectorRef, },
];

/**
 * Provides dependencies, pipes & directives for translating messages, dates & numbers.
 */
class LocalizationModule {
    /**
     * @param {?} injector
     */
    constructor(injector) {
        this.injector = injector;
        // Creates the instance of the InjectorRef, so that module dependencies are available.
    }
    /**
     * Use in AppModule: new instances of LocaleService & TranslationService.
     * @param {?=} token
     * @return {?}
     */
    static forRoot(token = {}) {
        return {
            ngModule: LocalizationModule,
            providers: [
                InjectorRef,
                LocaleConfig,
                LocaleService,
                {
                    provide: LocaleStorage,
                    useClass: token.localeStorage || BrowserStorage
                },
                TranslationConfig,
                TranslationService,
                {
                    provide: TranslationProvider,
                    useClass: token.translationProvider || HttpTranslationProvider
                },
                {
                    provide: TranslationHandler,
                    useClass: token.translationHandler || DefaultTranslationHandler
                }
            ]
        };
    }
    /**
     * Use in feature modules with lazy loading: new instance of TranslationService.
     * @param {?=} token
     * @return {?}
     */
    static forChild(token = {}) {
        return {
            ngModule: LocalizationModule,
            providers: [
                InjectorRef,
                TranslationConfig,
                TranslationService,
                {
                    provide: TranslationProvider,
                    useClass: token.translationProvider || HttpTranslationProvider
                },
                {
                    provide: TranslationHandler,
                    useClass: token.translationHandler || DefaultTranslationHandler
                }
            ]
        };
    }
}
LocalizationModule.decorators = [
    { type: NgModule, args: [{
                declarations: [
                    LocaleDatePipe,
                    LocaleDecimalPipe,
                    LocalePercentPipe,
                    LocaleCurrencyPipe,
                    LocaleDateDirective,
                    LocaleDecimalDirective,
                    LocalePercentDirective,
                    LocaleCurrencyDirective
                ],
                imports: [
                    TranslationModule
                ],
                exports: [
                    TranslationModule,
                    LocaleDatePipe,
                    LocaleDecimalPipe,
                    LocalePercentPipe,
                    LocaleCurrencyPipe,
                    LocaleDateDirective,
                    LocaleDecimalDirective,
                    LocalePercentDirective,
                    LocaleCurrencyDirective
                ]
            },] },
];
/**
 * @nocollapse
 */
LocalizationModule.ctorParameters = () => [
    { type: InjectorRef, },
];

/**
 * Provides dependencies & directives for locale validation.
 */
class LocaleValidationModule {
    /**
     * Use in AppModule: new instance of LocaleValidation.
     * @return {?}
     */
    static forRoot() {
        return {
            ngModule: LocaleValidationModule,
            providers: [DecimalCode, LocaleValidation]
        };
    }
}
LocaleValidationModule.decorators = [
    { type: NgModule, args: [{
                declarations: [
                    LocaleNumberValidatorDirective
                ],
                exports: [
                    LocaleNumberValidatorDirective
                ]
            },] },
];
/**
 * @nocollapse
 */
LocaleValidationModule.ctorParameters = () => [];

/**
 * Angular l10n.
 * An Angular library to translate messages, dates and numbers.
 * Written by Roberto Simonetti.
 * MIT license.
 * https://github.com/robisim74/angular-l10n
 */
/**
 * Entry point for all public APIs of the package.
 */

/**
 * Generated bundle index. Do not edit.
 */

export { InjectorRef, LocaleConfig, LocaleService, LocaleStorage, BrowserStorage, TranslationConfig, TranslationService, TranslationProvider, HttpTranslationProvider, TranslationHandler, DefaultTranslationHandler, Translation, Localization, NumberCode, DecimalCode, LocaleValidation, Collator, IntlAPI, Language, DefaultLocale$1 as DefaultLocale, Currency, TranslatePipe, LocaleDatePipe, LocaleDecimalPipe, LocalePercentPipe, LocaleCurrencyPipe, BaseDirective, TranslateDirective, LocaleDateDirective, LocaleDecimalDirective, LocalePercentDirective, LocaleCurrencyDirective, LocaleNumberValidatorDirective, validateLocaleNumber, TranslationModule, LocalizationModule, LocaleValidationModule };
//# sourceMappingURL=angular-l10n.js.map
