import { LocaleConfig } from './locale-config';
export interface ILocaleConfigAPI {
    /**
     * Adds a language to use in the app, specifying the layout direction.
     * @param languageCode ISO 639 two-letter or three-letter code of the language
     * @param textDirection Default is ltr (left to right)
     */
    addLanguage(languageCode: string, textDirection?: string): ILocaleConfigAPI;
    /**
     * Adds the languages to use in the app.
     * @param languageCodes Array of ISO 639 two-letter or three-letter codes of the languages
     */
    addLanguages(languageCodes: string[]): ILocaleConfigAPI;
    /**
     * Disables the browser storage for language, default locale & currency.
     */
    disableStorage(): ILocaleConfigAPI;
    /**
     * If the cookie expiration is omitted, the cookie becomes a session cookie.
     */
    setCookieExpiration(days?: number): ILocaleConfigAPI;
    /**
     * Sets browser LocalStorage as default for language, default locale & currency.
     */
    useLocalStorage(): ILocaleConfigAPI;
    /**
     * Sets browser SessionStorage as default for language, default locale & currency.
     */
    useSessionStorage(): ILocaleConfigAPI;
    /**
     * Defines the language to be used:
     *  - tries to get the language from the browser storage;
     *  - tries to get the language of the browser if it has been added;
     *  - otherwise gets the parameter language.
     * @param languageCode ISO 639 two-letter or three-letter code of the language
     */
    defineLanguage(languageCode: string): ILocaleConfigAPI;
    /**
     * Defines the default locale to be used, regardless of the browser language.
     * @param languageCode ISO 639 two-letter or three-letter code of the language
     * @param countryCode ISO 3166 two-letter, uppercase code of the country
     * @param scriptCode Optional ISO 15924 four-letter script code
     * @param numberingSystem Optional numbering system
     * @param calendar Optional calendar
     */
    defineDefaultLocale(languageCode: string, countryCode: string, scriptCode?: string, numberingSystem?: string, calendar?: string): ILocaleConfigAPI;
    /**
     * Defines the currency to be used.
     * @param defaultCurrency ISO 4217 three-letter code of the currency
     */
    defineCurrency(currencyCode: string): ILocaleConfigAPI;
}
export declare class LocaleConfigAPI implements ILocaleConfigAPI {
    private configuration;
    constructor(configuration: LocaleConfig);
    addLanguage(languageCode: string, textDirection?: string): ILocaleConfigAPI;
    addLanguages(languageCodes: string[]): ILocaleConfigAPI;
    disableStorage(): ILocaleConfigAPI;
    setCookieExpiration(days?: number): ILocaleConfigAPI;
    useLocalStorage(): ILocaleConfigAPI;
    useSessionStorage(): ILocaleConfigAPI;
    defineLanguage(languageCode: string): ILocaleConfigAPI;
    defineDefaultLocale(languageCode: string, countryCode: string, scriptCode?: string, numberingSystem?: string, calendar?: string): ILocaleConfigAPI;
    defineCurrency(currencyCode: string): ILocaleConfigAPI;
}
