import { Observable } from 'rxjs/Observable';
import { LocaleService } from './locale.service';
import { TranslationService } from './translation.service';
/**
 * Intl.Collator APIs.
 */
export interface ICollator {
    /**
     * Compares two keys by the value of translation according to the current language.
     * @param key1, key2 The keys of the values to compare
     * @param extension Unicode extension key, e.g. 'co-phonebk'
     * @param options Default is { usage: 'sort', sensitivity: 'variant' }
     * @return A negative value if the value of translation of key1 comes before the value of translation of key2;
     *         a positive value if key1 comes after key2;
     *         0 if they are considered equal or Intl.Collator is not supported
     * @see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Collator
     */
    compare(key1: string, key2: string, extension?: string, options?: any): number;
    /**
     * Sorts an array of objects or an array of arrays according to the current language.
     * @param list The array to be sorted
     * @param keyName The column that contains the keys of the values to be ordered
     * @param order 'asc' or 'desc'. The default value is 'asc'
     * @param extension Unicode extension key, e.g. 'co-phonebk'
     * @param options Default is { usage: 'sort', sensitivity: 'variant' }
     * @return The same sorted list or the same list if Intl.Collator is not supported
     * @see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Collator
     */
    sort(list: any[], keyName: any, order?: string, extension?: string, options?: any): any[];
    /**
     * Sorts asynchronously an array of objects or an array of arrays according to the current language.
     * @param list The array to be sorted
     * @param keyName The column that contains the keys of the values to be ordered
     * @param order 'asc' or 'desc'. The default value is 'asc'
     * @param extension Unicode extension key, e.g. 'co-phonebk'
     * @param options Default is { usage: 'sort', sensitivity: 'variant' }
     * @return An observable of the sorted list or of the same list if Intl.Collator is not supported
     * @see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Collator
     */
    sortAsync(list: any[], keyName: any, order?: string, extension?: string, options?: any): Observable<any[]>;
    /**
     * Matches a string into an array of objects or an array of arrays
     * according to the current language.
     * @param s The string to search
     * @param list The array in which to search
     * @param keyNames An array that contains the columns to look for
     * @param options Default is { usage: 'search' }
     * @return A filtered list or the same list if Intl.Collator is not supported
     * @see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Collator
     */
    search(s: string, list: any[], keyNames: any[], options?: any): any[];
    /**
     * Matches asynchronously a string into an array of objects or an array of arrays
     * according to the current language.
     * @param s The string to search
     * @param list The array in which to search
     * @param keyNames An array that contains the columns to look for
     * @param options Default is { usage: 'search' }
     * @return An observable of the filtered list or the same list if Intl.Collator is not supported
     * @see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Collator
     */
    searchAsync(s: string, list: any[], keyNames: any[], options?: any): Observable<any[]>;
}
export declare class Collator implements ICollator {
    private locale;
    private translation;
    constructor(locale: LocaleService, translation: TranslationService);
    compare(key1: string, key2: string, extension?: string, options?: any): number;
    sort(list: any[], keyName: any, order?: string, extension?: string, options?: any): any[];
    sortAsync(list: any[], keyName: any, order?: string, extension?: string, options?: any): Observable<any[]>;
    search(s: string, list: any[], keyNames: any[], options?: any): any[];
    searchAsync(s: string, list: any[], keyNames: any[], options?: any): Observable<any[]>;
    private addExtension(locale, extension?);
    private match(key, s, collator);
}
