# localForage [![Build Status](https://secure.travis-ci.org/mozilla/localForage.png?branch=master)](http://travis-ci.org/mozilla/localForage)

localForage is a JavaScript library that improves the offline experience of
your web app by using asynchronous storage (via IndexedDB or WebSQL
where available) with a simple, `localStorage`-like API.

localForage uses localStorage in browsers with no IndexedDB or
WebSQL support. Asynchronous storage is available in the current
versions of all major browsers: Chrome, Firefox, IE, and Safari
(including Safari Mobile). See below for detailed compatibility info.

To use localForage, just drop a single JavaScript file into your page:

```html
<script src="localforage.js"></script>
<script>localforage.getItem('something', myCallback);</script>
```

Download the [latest localForage from GitHub](https://github.com/mozilla/localForage/releases/latest), or install with
[npm](https://www.npmjs.org/) or [bower](http://bower.io):

```bash
npm install localforage
bower install localforage
```

localForage is compatible with [browserify](http://browserify.org/).

## Supported Browsers/Platforms

localForage works in all modern browsers (IE 8 and above).
_Asynchronous storage_ is available in all browsers **in bold**, with
localStorage fallback in parentheses.

* **Android Browser 2.1**
* **Blackberry 7**
* **Chrome 23** (Chrome 4.0+ with localStorage)
* **Chrome for Android 32**
* **Firefox 10** (Firefox 3.5+ with localStorage)
* **Firefox for Android 25**
* **Firefox OS 1.0**
* **IE 10** (IE 8+ with localStorage)
* **IE Mobile 10**
* **Opera 15** (Opera 10.5+ with localStorage)
* **Opera Mobile 11**
* **Phonegap/Apache Cordova 1.2.0**
* **Safari 3.1** (includes Mobile Safari)

Different browsers have [different storage limits](http://www.html5rocks.com/en/tutorials/offline/quota-research/#toc-overview), so plan accordingly.

Note that, thanks to WebSQL support, apps packaged with Phonegap will also
use asynchronous storage. Pretty slick!

## Support

Lost? Need help? Try the
[localForage API documentation](https://mozilla.github.io/localForage).

If you're stuck using the library, running the tests, or want to contribute,
to localForage, you can visit
[irc.mozilla.org](https://wiki.mozilla.org/IRC) and head to the `#apps`
channel to ask questions about localForage.

The best person to ask about localForage is [**tofumatt**][tofumatt], who
is usually online from 8am-10pm Eastern Time.

[tofumatt]: http://tofumatt.com/

# How to use localForage

## Callbacks

Because localForage uses async storage, it has an async API. It's otherwise
exactly the same as the
[localStorage API](https://hacks.mozilla.org/2009/06/localstorage/).

```javascript
// In localStorage, we would do:
localStorage.setItem('key', JSON.stringify('value'));
doSomethingElse();

// With localForage, we use callbacks:
localforage.setItem('key', 'value', doSomethingElse);
```

Similarly, please don't expect a return value from calls to
`localforage.getItem()`. Instead, use a callback:

```javascript
// Synchronous; slower!
var value = JSON.parse(localStorage.getItem('key'));
alert(value);

// Async, fast, and non-blocking!
localforage.getItem('key', function(err, value) { alert(value) });

```

Note that callbacks in localForage are Node-style (error argument first) since
`0.9.3`. This means if you're using callbacks, your code should look like this:

```javascript
// Use err as your first argument.
localforage.getItem('key', function(err, value) {
    if (err) {
        console.error('Oh noes!');
    } else {
        alert(value);
    }
});
```

You can store any type in localForage; you aren't limited to strings like in
localStorage. Even if localStorage is your storage backend, localForage
automatically does `JSON.parse()` and `JSON.stringify()` when getting/setting
values.

## Promises

Promises are pretty cool! If you'd rather use promises than callbacks,
localForage supports that too:

```javascript
function doSomethingElse(value) {
    console.log(value);
}

// With localForage, we allow promises:
localforage.setItem('key', 'value').then(doSomethingElse);
```

Note that with Promises, `err` is not the first argument to your function.
Instead, you handle an error with the rejection part of the Promise:

```javascript
// A full setItem() call with Promises.
localforage.setItem('key', 'value').then(function(value) {
    alert(value + ' was set!');
}, function(error) {
    console.error(error);
});
```

localForage relies on native [ES6 Promises](http://www.promisejs.org/), but
[ships with an awesome polyfill](https://github.com/jakearchibald/ES6-Promises)
for browsers that don't support ES6 Promises yet.

## Storing Blobs, TypedArrays, and other JS objects

localForage supports storing all native JS objects that can be serialized to
JSON, as well as ArrayBuffers, Blobs, and TypedArrays. Check the
[API docs][api] for a full list of types supported by localForage.

All types are supported in every storage backend, though storage limits in
localStorage make storing many large Blobs impossible.

[api]: https://mozilla.github.io/localForage/#setitem

## Driver Selection (i.e. forcing localStorage)

For development, it can be easier to use the
slower--but easier to debug--localStorage driver (mostly because localStorage
can easily be inspected from the console). You can use the `setDriver()` method
to change the driver localForage is using at any time.

```javascript
// If you aren't using JS modules, things are loaded synchronously.
localforage.setDriver(localforage.LOCALSTORAGE);
alert(localforage.driver());
  => 'localStorageWrapper'

// If you're using modules, things load asynchronously, so you should use
// callbacks or promises to ensure things have loaded.
localforage.setDriver(localforage.LOCALSTORAGE, function() {
    alert(localforage.driver());
});
  => 'localStorageWrapper'

// The promises version:
localforage.setDriver(localforage.LOCALSTORAGE).then(function() {
    alert(localforage.driver());
});
  => 'localStorageWrapper'
```

You can actually force any available driver with this method, but given that
the best driver will be selected automatically when the library is loaded, this
method is mostly useful in forcing localStorage.

Note that trying to load a driver unavailable on the current browser (like
trying to load WebSQL in Gecko) will fail and the previously loaded "best
choice" will continue to be used.

## Configuration

You can set database information with the `config()` method.
Available options are `driver`, `name`, `storeName`, `version`, and
`description`.

Example:
```javascript
localforage.config({
    driver      : localforage.WEBSQL, // Force WebSQL; same as using setDriver()
    name        : 'myApp',
    version     : 1.0,
    size        : 4980736, // Size of database, in bytes. WebSQL-only for now.
    storeName   : 'keyvaluepairs', // Should be alphanumeric, with underscores.
    description : 'some description'
});
```

**Note:** you must call `config()` _before_ you interact with your data. This
means calling `config()` before using `getItem()`, `setItem()`, `removeItem()`,
`clear()`, `key()`, `keys()` or `length()`.

## RequireJS

You can use localForage with [RequireJS](http://requirejs.org/), and even though
each driver will be loaded asynchronously with a `require()` call, you can use
localForage without having to confirm that it's ready:

```javascript
define(['localforage'], function(localforage) {
    // As a callback:
    localforage.setItem('mykey', 'myvalue', console.log);

    // With a Promise:
    localforage.setItem('mykey', 'myvalue').then(console.log);
});
```

In pre-1.0 versions you had to call `.ready()` to make sure the code was loaded,
but this is no longer necessary.

## Web Workers

Web Worker support in Firefox is blocked by [bug 701634][]. Until it is fixed,
web workers are not officially supported by localForage.

[bug 701634]: https://bugzilla.mozilla.org/show_bug.cgi?id=701634

## Framework Support

If you use a framework listed, there's a localForage storage driver for the
models in your framework so you can store data offline with localForage. We
have drivers for the following frameworks:

* [AngularJS](https://github.com/ocombe/angular-localForage)
* [Backbone](https://github.com/mozilla/localForage-backbone)
* [Ember](https://github.com/genkgo/ember-localforage-adapter)

If you have a driver you'd like listed, please
[open an issue](https://github.com/mozilla/localForage/issues/new) to have it
added to this list.

## Custom Drivers

You can create your own driver if you want; see the
[`defineDriver`](https://mozilla.github.io/localForage/#definedriver) API docs.

# Working on localForage

You'll need [node/npm](http://nodejs.org/),
[bower](http://bower.io/#installing-bower), and
[Grunt](http://gruntjs.com/getting-started#installing-the-cli).

To work on localForage, you should start by
[forking it](https://github.com/mozilla/localForage/fork) and installing its
dependencies. Replace `USERNAME` with your GitHub username and run the
following:

```bash
git clone git@github.com:USERNAME/localForage.git
cd localForage
npm install
bower install
```

Omitting the bower dependencies will cause the tests to fail!

## Running Tests

You need PhantomJS installed to run local tests. Run `npm test` (or,
directly: `grunt test`). Your code must also pass the
[linter](http://www.jshint.com/).

localForage is designed to run in the browser, so the tests explicitly require
a browser environment. Local tests are run on a headless WebKit (using
[PhantomJS](http://phantomjs.org)), but cross-browser tests are run using
[Sauce Labs](https://saucelabs.com/).

If you have Sauce Labs credentials on your machine, localForage will attempt
to connect to Sauce Labs to run the tests on Sauce Labs as well. To skip
Sauce Labs tests, run `grunt test:local`.

When you submit a pull request, tests will be run against all browsers that
localForage supports.

# License

This program is free software; it is distributed under an
[Apache License](https://github.com/mozilla/localForage/blob/master/LICENSE).

---

Copyright (c) 2013-2014 [Mozilla](https://mozilla.org)
([Contributors](https://github.com/mozilla/localForage/graphs/contributors)).
