export function getPointFromEvent(event) {
    // TouchEvent
    if (event.changedTouches !== undefined && event.changedTouches.length > 0) {
        return {
            x: event.changedTouches[0].clientX,
            y: event.changedTouches[0].clientY,
        };
    }
    // MouseEvent
    else if (event.clientX !== undefined && event.clientY !== undefined) {
        return {
            x: event.clientX,
            y: event.clientY,
        };
    }
    // KeyboardEvent
    else if (event.currentTarget !== undefined) {
        const gutterEl = event.currentTarget;
        return {
            x: gutterEl.offsetLeft,
            y: gutterEl.offsetTop,
        };
    }
    return null;
}
export function pointDeltaEquals(lhs, rhs, deltaPx) {
    return Math.abs(lhs.x - rhs.x) <= deltaPx && Math.abs(lhs.y - rhs.y) <= deltaPx;
}
export function getKeyboardEndpoint(event, direction) {
    // Return null if direction keys on the opposite axis were pressed
    if (direction === 'horizontal') {
        switch (event.key) {
            case 'ArrowLeft':
            case 'ArrowRight':
            case 'PageUp':
            case 'PageDown':
                break;
            default:
                return null;
        }
    }
    if (direction === 'vertical') {
        switch (event.key) {
            case 'ArrowUp':
            case 'ArrowDown':
            case 'PageUp':
            case 'PageDown':
                break;
            default:
                return null;
        }
    }
    const gutterEl = event.currentTarget;
    const offset = event.key === 'PageUp' || event.key === 'PageDown' ? 50 * 10 : 50;
    let offsetX = gutterEl.offsetLeft, offsetY = gutterEl.offsetTop;
    switch (event.key) {
        case 'ArrowLeft':
            offsetX -= offset;
            break;
        case 'ArrowRight':
            offsetX += offset;
            break;
        case 'ArrowUp':
            offsetY -= offset;
            break;
        case 'ArrowDown':
            offsetY += offset;
            break;
        case 'PageUp':
            if (direction === 'vertical') {
                offsetY -= offset;
            }
            else {
                offsetX += offset;
            }
            break;
        case 'PageDown':
            if (direction === 'vertical') {
                offsetY += offset;
            }
            else {
                offsetX -= offset;
            }
            break;
        default:
            return null;
    }
    return {
        x: offsetX,
        y: offsetY,
    };
}
export function getElementPixelSize(elRef, direction) {
    const rect = elRef.nativeElement.getBoundingClientRect();
    return direction === 'horizontal' ? rect.width : rect.height;
}
export function getInputBoolean(v) {
    return typeof v === 'boolean' ? v : v === 'false' ? false : true;
}
export function getInputPositiveNumber(v, defaultValue) {
    if (v === null || v === undefined)
        return defaultValue;
    v = Number(v);
    return !isNaN(v) && v >= 0 ? v : defaultValue;
}
export function isUserSizesValid(unit, sizes) {
    // All sizes have to be not null and total should be 100
    if (unit === 'percent') {
        const total = sizes.reduce((total, s) => (s !== null ? total + s : total), 0);
        return sizes.every((s) => s !== null) && total > 99.9 && total < 100.1;
    }
    // A size at null is mandatory but only one.
    if (unit === 'pixel') {
        return sizes.filter((s) => s === null).length === 1;
    }
}
export function getAreaMinSize(a) {
    if (a.size === null) {
        return null;
    }
    if (a.component.lockSize === true) {
        return a.size;
    }
    if (a.component.minSize === null) {
        return null;
    }
    if (a.component.minSize > a.size) {
        return a.size;
    }
    return a.component.minSize;
}
export function getAreaMaxSize(a) {
    if (a.size === null) {
        return null;
    }
    if (a.component.lockSize === true) {
        return a.size;
    }
    if (a.component.maxSize === null) {
        return null;
    }
    if (a.component.maxSize < a.size) {
        return a.size;
    }
    return a.component.maxSize;
}
export function getGutterSideAbsorptionCapacity(unit, sideAreas, pixels, allAreasSizePixel) {
    return sideAreas.reduce((acc, area) => {
        const res = getAreaAbsorptionCapacity(unit, area, acc.remain, allAreasSizePixel);
        acc.list.push(res);
        acc.remain = res.pixelRemain;
        return acc;
    }, { remain: pixels, list: [] });
}
function getAreaAbsorptionCapacity(unit, areaSnapshot, pixels, allAreasSizePixel) {
    // No pain no gain
    if (pixels === 0) {
        return {
            areaSnapshot,
            pixelAbsorb: 0,
            percentAfterAbsorption: areaSnapshot.sizePercentAtStart,
            pixelRemain: 0,
        };
    }
    // Area start at zero and need to be reduced, not possible
    if (areaSnapshot.sizePixelAtStart === 0 && pixels < 0) {
        return {
            areaSnapshot,
            pixelAbsorb: 0,
            percentAfterAbsorption: 0,
            pixelRemain: pixels,
        };
    }
    if (unit === 'percent') {
        return getAreaAbsorptionCapacityPercent(areaSnapshot, pixels, allAreasSizePixel);
    }
    if (unit === 'pixel') {
        return getAreaAbsorptionCapacityPixel(areaSnapshot, pixels, allAreasSizePixel);
    }
}
function getAreaAbsorptionCapacityPercent(areaSnapshot, pixels, allAreasSizePixel) {
    const tempPixelSize = areaSnapshot.sizePixelAtStart + pixels;
    const tempPercentSize = (tempPixelSize / allAreasSizePixel) * 100;
    // ENLARGE AREA
    if (pixels > 0) {
        // If maxSize & newSize bigger than it > absorb to max and return remaining pixels
        if (areaSnapshot.area.maxSize !== null && tempPercentSize > areaSnapshot.area.maxSize) {
            // Use area.area.maxSize as newPercentSize and return calculate pixels remaining
            const maxSizePixel = (areaSnapshot.area.maxSize / 100) * allAreasSizePixel;
            return {
                areaSnapshot,
                pixelAbsorb: maxSizePixel,
                percentAfterAbsorption: areaSnapshot.area.maxSize,
                pixelRemain: areaSnapshot.sizePixelAtStart + pixels - maxSizePixel,
            };
        }
        return {
            areaSnapshot,
            pixelAbsorb: pixels,
            percentAfterAbsorption: tempPercentSize > 100 ? 100 : tempPercentSize,
            pixelRemain: 0,
        };
    }
    // REDUCE AREA
    else if (pixels < 0) {
        // If minSize & newSize smaller than it > absorb to min and return remaining pixels
        if (areaSnapshot.area.minSize !== null && tempPercentSize < areaSnapshot.area.minSize) {
            // Use area.area.minSize as newPercentSize and return calculate pixels remaining
            const minSizePixel = (areaSnapshot.area.minSize / 100) * allAreasSizePixel;
            return {
                areaSnapshot,
                pixelAbsorb: minSizePixel,
                percentAfterAbsorption: areaSnapshot.area.minSize,
                pixelRemain: areaSnapshot.sizePixelAtStart + pixels - minSizePixel,
            };
        }
        // If reduced under zero > return remaining pixels
        else if (tempPercentSize < 0) {
            // Use 0 as newPercentSize and return calculate pixels remaining
            return {
                areaSnapshot,
                pixelAbsorb: -areaSnapshot.sizePixelAtStart,
                percentAfterAbsorption: 0,
                pixelRemain: pixels + areaSnapshot.sizePixelAtStart,
            };
        }
        return {
            areaSnapshot,
            pixelAbsorb: pixels,
            percentAfterAbsorption: tempPercentSize,
            pixelRemain: 0,
        };
    }
}
function getAreaAbsorptionCapacityPixel(areaSnapshot, pixels, containerSizePixel) {
    const tempPixelSize = areaSnapshot.sizePixelAtStart + pixels;
    // ENLARGE AREA
    if (pixels > 0) {
        // If maxSize & newSize bigger than it > absorb to max and return remaining pixels
        if (areaSnapshot.area.maxSize !== null && tempPixelSize > areaSnapshot.area.maxSize) {
            return {
                areaSnapshot,
                pixelAbsorb: areaSnapshot.area.maxSize - areaSnapshot.sizePixelAtStart,
                percentAfterAbsorption: -1,
                pixelRemain: tempPixelSize - areaSnapshot.area.maxSize,
            };
        }
        return {
            areaSnapshot,
            pixelAbsorb: pixels,
            percentAfterAbsorption: -1,
            pixelRemain: 0,
        };
    }
    // REDUCE AREA
    else if (pixels < 0) {
        // If minSize & newSize smaller than it > absorb to min and return remaining pixels
        if (areaSnapshot.area.minSize !== null && tempPixelSize < areaSnapshot.area.minSize) {
            return {
                areaSnapshot,
                pixelAbsorb: areaSnapshot.area.minSize + pixels - tempPixelSize,
                percentAfterAbsorption: -1,
                pixelRemain: tempPixelSize - areaSnapshot.area.minSize,
            };
        }
        // If reduced under zero > return remaining pixels
        else if (tempPixelSize < 0) {
            return {
                areaSnapshot,
                pixelAbsorb: -areaSnapshot.sizePixelAtStart,
                percentAfterAbsorption: -1,
                pixelRemain: pixels + areaSnapshot.sizePixelAtStart,
            };
        }
        return {
            areaSnapshot,
            pixelAbsorb: pixels,
            percentAfterAbsorption: -1,
            pixelRemain: 0,
        };
    }
}
export function updateAreaSize(unit, item) {
    if (unit === 'percent') {
        item.areaSnapshot.area.size = item.percentAfterAbsorption;
    }
    else if (unit === 'pixel') {
        // Update size except for the wildcard size area
        if (item.areaSnapshot.area.size !== null) {
            item.areaSnapshot.area.size = item.areaSnapshot.sizePixelAtStart + item.pixelAbsorb;
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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