"use strict";
var forIn = require('lodash.forin');
var isEqual = require('lodash.isequal');
var assign = require('lodash.assign');
var ApolloHandle = (function () {
    function ApolloHandle(_a) {
        var component = _a.component, client = _a.client, queries = _a.queries, mutations = _a.mutations;
        this.lastQueryVariables = {};
        this.queryHandles = {};
        this.querySubscriptions = {};
        this.component = component;
        this.client = client;
        this.queries = queries;
        this.mutations = mutations;
    }
    ApolloHandle.prototype.handleQueries = function () {
        var _this = this;
        if (!this.queries) {
            return;
        }
        forIn(this.queries(this.component), function (options, queryName) {
            if (_this.hasVariablesChanged(queryName, options.variables)) {
                if (_this.getQuery(queryName)) {
                    _this.reuseQuery(queryName, options);
                }
                else {
                    _this.createQuery(queryName, options);
                }
            }
        });
    };
    ApolloHandle.prototype.handleMutations = function () {
        var _this = this;
        if (!this.mutations) {
            return;
        }
        forIn(this.mutations(this.component), function (method, mutationName) {
            _this.createMutation(mutationName, method);
        });
    };
    ApolloHandle.prototype.unsubscribe = function (queryName) {
        var allQueries = this.getAllQueriesSubs();
        if (allQueries) {
            if (queryName) {
                var single = allQueries[queryName];
                // just one
                if (single) {
                    single.unsubscribe();
                }
            }
            else {
                // loop through all
                for (var name_1 in allQueries) {
                    if (allQueries.hasOwnProperty(name_1)) {
                        allQueries[name_1].unsubscribe();
                    }
                }
            }
        }
    };
    ApolloHandle.prototype.setQuery = function (name, handle) {
        this.queryHandles[name] = handle;
    };
    ApolloHandle.prototype.getQuery = function (name) {
        return this.queryHandles[name];
    };
    ApolloHandle.prototype.setQuerySub = function (name, sub) {
        this.querySubscriptions[name] = sub;
    };
    ApolloHandle.prototype.getQuerySub = function (name) {
        return this.querySubscriptions[name];
    };
    ApolloHandle.prototype.getAllQueriesSubs = function () {
        return this.querySubscriptions;
    };
    /**
     * Saves variables so they can be used in futher comparasion
     * @param {string} queryName Query's name
     * @param {any}    variables used variables
     */
    ApolloHandle.prototype.saveVariables = function (queryName, variables) {
        this.lastQueryVariables[queryName] = variables;
    };
    /**
     * Compares current variables with previous ones.
     * @param  {string}  queryName Query's name
     * @param  {any}     variables current variables
     * @return {boolean}           comparasion result
     */
    ApolloHandle.prototype.hasVariablesChanged = function (queryName, variables) {
        return !(this.lastQueryVariables.hasOwnProperty(queryName)
            && isEqual(this.lastQueryVariables[queryName], variables));
    };
    ApolloHandle.prototype.hasDataChanged = function (queryName, data) {
        var _this = this;
        var changed = false;
        forIn(data, function (value, key) {
            if (!isEqual(_this.component[queryName][key], value)) {
                changed = true;
                return;
            }
        });
        return changed;
    };
    ApolloHandle.prototype.createQuery = function (queryName, options) {
        // save variables so they can be used in futher comparasion
        this.saveVariables(queryName, options.variables);
        // assign to component's context
        this.subscribe(queryName, this.client.watchQuery(options));
    };
    ApolloHandle.prototype.createMutation = function (mutationName, method) {
        var _this = this;
        // assign to component's context
        this.component[mutationName] = function () {
            var args = [];
            for (var _i = 0; _i < arguments.length; _i++) {
                args[_i - 0] = arguments[_i];
            }
            var options = method.apply(_this.client, args);
            return _this.client.mutate(options);
        };
    };
    // XXX https://github.com/apollostack/apollo-client/pull/362
    ApolloHandle.prototype.backcompat = function (queryName, method) {
        var args = [];
        for (var _i = 2; _i < arguments.length; _i++) {
            args[_i - 2] = arguments[_i];
        }
        if (this.getQuerySub(queryName)[method]) {
            return (_a = this.getQuerySub(queryName))[method].apply(_a, args);
        }
        return (_b = this.getQuery(queryName))[method].apply(_b, args);
        var _a, _b;
    };
    ApolloHandle.prototype.missingCompat = function (queryName, method) {
        var args = [];
        for (var _i = 2; _i < arguments.length; _i++) {
            args[_i - 2] = arguments[_i];
        }
        if (!this.getQuery(queryName)[method]) {
            throw new Error("Your version of the ApolloClient does not support '" + method + "'. Try to update.");
        }
        return (_a = this.getQuery(queryName))[method].apply(_a, args);
        var _a;
    };
    ApolloHandle.prototype.subscribe = function (queryName, obs) {
        var _this = this;
        this.component[queryName] = {
            errors: null,
            loading: true,
            unsubscribe: function () { return _this.getQuerySub(queryName).unsubscribe(); },
            refetch: function () {
                var args = [];
                for (var _i = 0; _i < arguments.length; _i++) {
                    args[_i - 0] = arguments[_i];
                }
                return _this.backcompat.apply(_this, [queryName, 'refetch'].concat(args));
            },
            stopPolling: function () { return _this.backcompat(queryName, 'stopPolling'); },
            startPolling: function () {
                var args = [];
                for (var _i = 0; _i < arguments.length; _i++) {
                    args[_i - 0] = arguments[_i];
                }
                return _this.backcompat.apply(_this, [queryName, 'startPolling'].concat(args));
            },
            fetchMore: function () {
                var args = [];
                for (var _i = 0; _i < arguments.length; _i++) {
                    args[_i - 0] = arguments[_i];
                }
                return _this.missingCompat.apply(_this, [queryName, 'fetchMore'].concat(args));
            },
        };
        var setQuery = function (_a) {
            var errors = _a.errors, loading = _a.loading, _b = _a.data, data = _b === void 0 ? {} : _b;
            var changed = _this.hasDataChanged(queryName, data);
            assign(_this.component[queryName], {
                errors: errors,
                // XXX backwards compatibility of loading property
                loading: !!loading,
            }, changed ? data : {});
        };
        // we don't want to have multiple subscriptions
        this.unsubscribe(queryName);
        this.setQuery(queryName, obs);
        this.setQuerySub(queryName, this.getQuery(queryName).subscribe({
            next: setQuery,
            error: function (errors) {
                setQuery({ errors: errors });
            },
        }));
    };
    ApolloHandle.prototype.reuseQuery = function (queryName, _a) {
        var variables = _a.variables;
        // save variables so they can be used in futher comparasion
        this.saveVariables(queryName, variables);
        // refetch query
        this.backcompat(queryName, 'refetch', variables);
    };
    return ApolloHandle;
}());
exports.ApolloHandle = ApolloHandle;
//# sourceMappingURL=ApolloHandle.js.map