"use strict";
var Rx_1 = require('rxjs/Rx');
var Scroller = (function () {
    function Scroller(windowElement, $interval, $elementRef, infiniteScrollDownCallback, infiniteScrollUpCallback, infiniteScrollDownDistance, infiniteScrollUpDistance, infiniteScrollParent, infiniteScrollThrottle, isImmediate) {
        this.windowElement = windowElement;
        this.$interval = $interval;
        this.$elementRef = $elementRef;
        this.infiniteScrollDownCallback = infiniteScrollDownCallback;
        this.infiniteScrollUpCallback = infiniteScrollUpCallback;
        this.infiniteScrollThrottle = infiniteScrollThrottle;
        this.isImmediate = isImmediate;
        this.lastScrollPosition = 0;
        this.isContainerWindow = toString.call(this.windowElement).includes('Window');
        this.documentElement = this.isContainerWindow ? this.windowElement.document.documentElement : null;
        this.handleInfiniteScrollDistance(infiniteScrollDownDistance, infiniteScrollUpDistance);
        // if (attrs.infiniteScrollParent != null) {
        // 	attachEvent(angular.element(elem.parent()));
        // }
        this.handleInfiniteScrollDisabled(false);
        this.defineContainer();
        this.createInterval();
    }
    Scroller.prototype.defineContainer = function () {
        if (this.isContainerWindow) {
            this.attachEvent(this.windowElement);
        }
        else {
            this.container = this.windowElement.nativeElement;
        }
    };
    Scroller.prototype.createInterval = function () {
        var _this = this;
        this.checkInterval = this.$interval(function () {
            if (_this.isImmediate) {
                return _this.handler();
            }
        }, 0);
    };
    Scroller.prototype.height = function (elem) {
        // elem = elem.nativeElement;
        if (isNaN(elem.offsetHeight)) {
            return this.documentElement.clientHeight;
        }
        else {
            return elem.offsetHeight;
        }
    };
    Scroller.prototype.offsetTop = function (elem) {
        // elem = elem.nativeElement;
        if (!elem.getBoundingClientRect) {
            return;
        }
        return elem.getBoundingClientRect().top + this.pageYOffset(elem);
    };
    Scroller.prototype.pageYOffset = function (elem) {
        // elem = elem.nativeElement;
        if (isNaN(window.pageYOffset)) {
            return this.documentElement.scrollTop;
        }
        else if (elem.ownerDocument) {
            return elem.ownerDocument.defaultView.pageYOffset;
        }
        else {
            elem.offsetTop;
        }
    };
    Scroller.prototype.handler = function () {
        var container = this.calculatePoints();
        var scrollingDown = this.lastScrollPosition < container.scrolledUntilNow;
        this.lastScrollPosition = container.scrolledUntilNow;
        var remaining;
        var containerBreakpoint;
        if (scrollingDown) {
            remaining = container.totalToScroll - container.scrolledUntilNow;
            containerBreakpoint = container.height * this.scrollDownDistance + 1;
        }
        else {
            remaining = container.scrolledUntilNow;
            containerBreakpoint = container.height * this.scrollUpDistance + 1;
        }
        var shouldScroll = remaining <= containerBreakpoint;
        var triggerCallback = shouldScroll && this.scrollEnabled;
        var shouldClearInterval = shouldScroll && this.checkInterval;
        // if (this.useDocumentBottom) {
        // 	container.totalToScroll = this.height(this.$elementRef.nativeElement.ownerDocument);
        // }
        this.checkWhenEnabled = shouldScroll;
        if (triggerCallback) {
            if (scrollingDown) {
                this.infiniteScrollDownCallback();
            }
            else {
                this.infiniteScrollUpCallback();
            }
        }
        if (shouldClearInterval) {
            clearInterval(this.checkInterval);
        }
    };
    Scroller.prototype.calculatePoints = function () {
        return this.isContainerWindow
            ? this.calculatePointsForWindow()
            : this.calculatePointsForElement();
    };
    Scroller.prototype.calculatePointsForWindow = function () {
        // container's height
        var height = this.height(this.container);
        // scrolled until now / current y point
        var scrolledUntilNow = height + this.pageYOffset(this.documentElement);
        // total height / most bottom y point
        var totalToScroll = this.offsetTop(this.$elementRef.nativeElement) + this.height(this.$elementRef.nativeElement);
        return { height: height, scrolledUntilNow: scrolledUntilNow, totalToScroll: totalToScroll };
    };
    Scroller.prototype.calculatePointsForElement = function () {
        var height = this.height(this.container);
        // perhaps use this.container.offsetTop instead of 'scrollTop'
        var scrolledUntilNow = this.container.scrollTop;
        var containerTopOffset = 0;
        var offsetTop = this.offsetTop(this.container);
        if (offsetTop !== void 0) {
            containerTopOffset = offsetTop;
        }
        var totalToScroll = this.container.scrollHeight;
        // const totalToScroll = this.offsetTop(this.$elementRef.nativeElement) - containerTopOffset + this.height(this.$elementRef.nativeElement);
        return { height: height, scrolledUntilNow: scrolledUntilNow, totalToScroll: totalToScroll };
    };
    Scroller.prototype.handleInfiniteScrollDistance = function (scrollDownDistance, scrollUpDistance) {
        this.scrollDownDistance = parseFloat(scrollDownDistance) || 0;
        this.scrollUpDistance = parseFloat(scrollUpDistance) || 0;
    };
    Scroller.prototype.attachEvent = function (newContainer) {
        var _this = this;
        this.clean();
        this.container = newContainer;
        if (newContainer) {
            var throttle_1 = this.infiniteScrollThrottle;
            this.disposeScroll = Rx_1.Observable.fromEvent(this.container, 'scroll')
                .debounce(function (ev) { return Rx_1.Observable.timer(throttle_1); })
                .subscribe(function (ev) { return _this.handler(); });
        }
    };
    Scroller.prototype.clean = function () {
        if (this.disposeScroll) {
            this.disposeScroll.unsubscribe();
        }
    };
    Scroller.prototype.handleInfiniteScrollDisabled = function (enableScroll) {
        this.scrollEnabled = !enableScroll;
        // if (this.scrollEnabled && checkWhenEnabled) {
        // 	checkWhenEnabled = false;
        // 	return handler();
        // }
    };
    return Scroller;
}());
exports.Scroller = Scroller;
//# sourceMappingURL=data:application/json;base64,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