// alert
// prompt (alert + blocking)
// confirm (yes / not
"use strict";
var PRIVATE_PREFIX = '$$';
var RESERVED_REGEX = new RegExp("^" + PRIVATE_PREFIX + ".*");
function validateMethodName(name) {
    if (!name) {
        throw new Error("Illegal method name. Empty method name is not allowed");
    }
    else if (name in this) {
        if (RESERVED_REGEX.test(name)) {
            throw new Error("Illegal method name '" + name + "'.\n             Method name starting with $$ is not allowed.");
        }
        else {
            throw new Error("A member name '" + name + "' already defined.");
        }
    }
}
function getConfigPropertyNames(subject) {
    return Object.getOwnPropertyNames(subject)
        .filter(function (name) { return RESERVED_REGEX.test(name); })
        .map(function (name) { return name.substr(2); });
}
function setConfigMethod(obj, name) {
    validateMethodName.call(obj, name);
    Object.defineProperty(obj, name, {
        configurable: false,
        enumerable: false,
        writable: false,
        value: function (value) {
            this[PRIVATE_PREFIX + name] = value;
            return this;
        }
    });
}
/**
 * Represent a fluent API factory wrapper for defining ConfigBuilder instances.
 */
var ConfigBuilderFactory = (function () {
    function ConfigBuilderFactory(configBuilder) {
        this._configBuilder =
            configBuilder instanceof ConfigBuilder ? configBuilder : new ConfigBuilder();
    }
    /**
     * Create a setter on the configuration object.
     * @param name The name of the setter function.
     * @param defaultValue If set (not undefined) set's the value on the config object immediately.
     * @returns {ConfigBuilderFactory}
     */
    ConfigBuilderFactory.prototype.setMethod = function (name, defaultValue) {
        if (defaultValue === void 0) { defaultValue = undefined; }
        setConfigMethod(this._configBuilder, name);
        if (defaultValue !== undefined) {
            this._configBuilder[name](defaultValue);
        }
        return this;
    };
    Object.defineProperty(ConfigBuilderFactory.prototype, "configBuilder", {
        /**
         * The config builder object.
         * @returns {ConfigBuilder<T>}
         */
        get: function () {
            return this._configBuilder;
        },
        enumerable: true,
        configurable: true
    });
    return ConfigBuilderFactory;
}());
exports.ConfigBuilderFactory = ConfigBuilderFactory;
/**
 * Represent a fluent API configuration builder where setter functions return their context
 * which is the ConfigBuilder instance itself.
 */
var ConfigBuilder = (function () {
    /**
     *
     * @param defaultConfigValues An object representing default values for the config object.
     * Keys are the config properties and the values are the default values for the config.
     * @param initialSetters A list of initial setters for this ConfigBuilder.
     */
    function ConfigBuilder(defaultConfigValues, initialSetters) {
        var _this = this;
        if (defaultConfigValues === void 0) { defaultConfigValues = undefined; }
        if (initialSetters === void 0) { initialSetters = undefined; }
        if (defaultConfigValues) {
            Object.getOwnPropertyNames(defaultConfigValues)
                .forEach(function (name) { return _this[PRIVATE_PREFIX + name] = defaultConfigValues[name]; });
        }
        if (Array.isArray(initialSetters)) {
            initialSetters.forEach(function (name) { return setConfigMethod(_this, name); });
        }
    }
    /**
     * Returns a ConfigBuilderFactory ready to define a ConfigBuilder instance.
     * @param defaultConfigValues An object representing default values for the config object.
     * Keys are the config properties and the values are the default values for the config.
     * @param initialSetters A list of initial setters for this ConfigBuilder.
     * @returns {ConfigBuilderFactory<T>}
     */
    ConfigBuilder.compose = function (defaultConfigValues, initialSetters) {
        if (defaultConfigValues === void 0) { defaultConfigValues = undefined; }
        if (initialSetters === void 0) { initialSetters = undefined; }
        return new ConfigBuilderFactory(new ConfigBuilder(defaultConfigValues, initialSetters));
    };
    ConfigBuilder.composeWith = function (builderType, defaultConfigValues, initialSetters) {
        if (defaultConfigValues === void 0) { defaultConfigValues = undefined; }
        if (initialSetters === void 0) { initialSetters = undefined; }
        return new ConfigBuilderFactory(new builderType(defaultConfigValues, initialSetters));
    };
    ConfigBuilder.prototype.toJSON = function () {
        var _this = this;
        return getConfigPropertyNames(this)
            .reduce(function (obj, name) {
            obj[name] = _this[PRIVATE_PREFIX + name];
            return obj;
        }, {});
    };
    return ConfigBuilder;
}());
exports.ConfigBuilder = ConfigBuilder;
//# sourceMappingURL=data:application/json;base64,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