/**
 * angular2-modal - Angular2 Modal (dialog) window.
 * @version v2.0.0-beta.2
 * @link https://github.com/shlomiassaf/angular2-modal
 * @license MIT
 */
import { ComponentFactoryResolver } from '@angular/core';
import { Subject } from 'rxjs/Subject';
import { createComponent } from '../framework/createComponent';
const BROWSER_PREFIX = ['webkit', 'moz', 'MS', 'o', ''];
function register(eventName, element, cb) {
    BROWSER_PREFIX.forEach(p => {
        element.addEventListener(p ? p + eventName : eventName.toLowerCase(), cb, false);
    });
}
/**
 * A base class that expose customisation methods in derived components.
 * Capabilities: Add/Remove styls, Add/Remove classes, listen to animation/transition end event,
 * add components
 */
export class BaseDynamicComponent {
    constructor(sanitizer, el) {
        this.sanitizer = sanitizer;
        this.el = el;
        this.style = {};
        this.styleStr = '';
        this.cssClass = '';
        this.classArray = [];
    }
    activateAnimationListener() {
        if (this.animationEnd)
            return;
        this.animationEnd = new Subject();
        this.animationEnd$ = this.animationEnd.asObservable();
        register('TransitionEnd', this.el.nativeElement, this.onEndTransition.bind(this));
        register('AnimationEnd', this.el.nativeElement, this.onEndAnimation.bind(this));
    }
    /**
     * Set a specific inline style on the overlay host element.
     * @param prop The style key
     * @param value The value, undefined to remove
     * @returns {ModalOverlay}
     */
    setStyle(prop, value) {
        if (this.style[prop] !== value) {
            if (value === undefined) {
                delete this.style[prop];
            }
            else {
                this.style[prop] = value;
            }
            this.applyStyle();
        }
        return this;
    }
    /**
     * Remove's all inline styles from the overlay host element.
     */
    clearStyles() {
        this.style = {};
        this.applyStyle();
    }
    addClass(css, nextTurn=false) {
        if (typeof css === 'string') {
            css.split(' ').forEach(c => this._addClass(c, nextTurn));
        }
    }
    removeClass(css, nextTurn=false) {
        if (typeof css === 'string') {
            css.split(' ').forEach(c => this._removeClass(c, nextTurn));
        }
    }
    ngOnDestroy() {
        if (this.animationEnd && !this.animationEnd.isUnsubscribed) {
            this.animationEnd.complete();
        }
    }
    applyStyle() {
        this.styleStr = this.sanitizer.bypassSecurityTrustStyle(JSON.stringify(this.style)
            .replace('{', '')
            .replace('}', '')
            .replace(/,/g, ';')
            .replace(/"/g, ''));
    }
    applyClasses(nextTurn) {
        if (nextTurn === true) {
            if (!this.applyOnNextTurn) {
                this.applyOnNextTurn = true;
                setTimeout(() => {
                    this.applyOnNextTurn = false;
                    this.applyClasses(false);
                });
            }
        }
        else {
            this.cssClass = this.classArray.join(' ');
        }
    }
    /**
     * Add a component, supply a view container ref.
     * Note: The components vcRef will result in a sibling.
     * @param type The component to add
     * @param vcRef The container to add to
     * @param bindings Bindings to use (added on top of the ViewContainerRef)
     * @returns {Promise<ComponentRef<any>>}
     */
    _addComponent(type, vcRef, bindings) {
        const cmpRef = createComponent(vcRef.injector.get(ComponentFactoryResolver), type, vcRef, bindings || []);
        cmpRef.changeDetectorRef.detectChanges();
        return cmpRef;
    }
    onEndTransition() {
        if (!this.animationEnd.isUnsubscribed) {
            this.animationEnd.next('transition');
        }
    }
    onEndAnimation() {
        if (!this.animationEnd.isUnsubscribed) {
            this.animationEnd.next('animation');
        }
    }
    _addClass(css, nextTurn=false) {
        if (this.classArray.indexOf(css) > -1)
            return;
        this.classArray.push(css);
        this.applyClasses(nextTurn);
    }
    _removeClass(css, nextTurn=false) {
        const idx = this.classArray.indexOf(css);
        if (idx > -1) {
            this.classArray.splice(idx, 1);
            this.applyClasses(nextTurn);
        }
    }
}

//# sourceMappingURL=base-dynamic-component.js.map
