import { Directive, ElementRef, EventEmitter, Inject } from 'angular2/angular2';
import { NG1_COMPILE, NG1_SCOPE, NG1_HTTP_BACKEND, NG1_TEMPLATE_CACHE, NG1_CONTROLLER } from './constants';
import { controllerKey } from './util';
import * as angular from './angular_js';
const CAMEL_CASE = /([A-Z])/g;
const INITIAL_VALUE = {
    __UNINITIALIZED__: true
};
const NOT_SUPPORTED = 'NOT_SUPPORTED';
export class UpgradeNg1ComponentAdapterBuilder {
    constructor(name) {
        this.name = name;
        this.inputs = [];
        this.inputsRename = [];
        this.outputs = [];
        this.outputsRename = [];
        this.propertyOutputs = [];
        this.checkProperties = [];
        this.propertyMap = {};
        this.linkFn = null;
        this.directive = null;
        this.$controller = null;
        var selector = name.replace(CAMEL_CASE, (all, next) => '-' + next.toLowerCase());
        var self = this;
        this.type =
            Directive({ selector: selector, inputs: this.inputsRename, outputs: this.outputsRename })
                .Class({
                constructor: [
                    new Inject(NG1_SCOPE),
                    ElementRef,
                    function (scope, elementRef) {
                        return new UpgradeNg1ComponentAdapter(self.linkFn, scope, self.directive, elementRef, self.$controller, self.inputs, self.outputs, self.propertyOutputs, self.checkProperties, self.propertyMap);
                    }
                ],
                ngOnChanges: function () { },
                ngDoCheck: function () { }
            });
    }
    extractDirective(injector) {
        var directives = injector.get(this.name + 'Directive');
        if (directives.length > 1) {
            throw new Error('Only support single directive definition for: ' + this.name);
        }
        var directive = directives[0];
        if (directive.replace)
            this.notSupported('replace');
        if (directive.terminal)
            this.notSupported('terminal');
        var link = directive.link;
        if (typeof link == 'object') {
            if (link.post)
                this.notSupported('link.post');
        }
        return directive;
    }
    notSupported(feature) {
        throw new Error(`Upgraded directive '${this.name}' does not support '${feature}'.`);
    }
    extractBindings() {
        var scope = this.directive.scope;
        if (typeof scope == 'object') {
            for (var name in scope) {
                if (scope.hasOwnProperty(name)) {
                    var localName = scope[name];
                    var type = localName.charAt(0);
                    localName = localName.substr(1) || name;
                    var outputName = 'output_' + name;
                    var outputNameRename = outputName + ': ' + name;
                    var outputNameRenameChange = outputName + ': ' + name + 'Change';
                    var inputName = 'input_' + name;
                    var inputNameRename = inputName + ': ' + name;
                    switch (type) {
                        case '=':
                            this.propertyOutputs.push(outputName);
                            this.checkProperties.push(localName);
                            this.outputs.push(outputName);
                            this.outputsRename.push(outputNameRenameChange);
                            this.propertyMap[outputName] = localName;
                        // don't break; let it fall through to '@'
                        case '@':
                            this.inputs.push(inputName);
                            this.inputsRename.push(inputNameRename);
                            this.propertyMap[inputName] = localName;
                            break;
                        case '&':
                            this.outputs.push(outputName);
                            this.outputsRename.push(outputNameRename);
                            this.propertyMap[outputName] = localName;
                            break;
                        default:
                            var json = JSON.stringify(scope);
                            throw new Error(`Unexpected mapping '${type}' in '${json}' in '${this.name}' directive.`);
                    }
                }
            }
        }
    }
    compileTemplate(compile, templateCache, httpBackend) {
        if (this.directive.template) {
            this.linkFn = compileHtml(this.directive.template);
        }
        else if (this.directive.templateUrl) {
            var url = this.directive.templateUrl;
            var html = templateCache.get(url);
            if (html !== undefined) {
                this.linkFn = compileHtml(html);
            }
            else {
                return new Promise((resolve, err) => {
                    httpBackend('GET', url, null, (status, response) => {
                        if (status == 200) {
                            resolve(this.linkFn = compileHtml(templateCache.put(url, response)));
                        }
                        else {
                            err(`GET ${url} returned ${status}: ${response}`);
                        }
                    });
                });
            }
        }
        else {
            throw new Error(`Directive '${this.name}' is not a component, it is missing template.`);
        }
        return null;
        function compileHtml(html) {
            var div = document.createElement('div');
            div.innerHTML = html;
            return compile(div.childNodes);
        }
    }
    static resolve(exportedComponents, injector) {
        var promises = [];
        var compile = injector.get(NG1_COMPILE);
        var templateCache = injector.get(NG1_TEMPLATE_CACHE);
        var httpBackend = injector.get(NG1_HTTP_BACKEND);
        var $controller = injector.get(NG1_CONTROLLER);
        for (var name in exportedComponents) {
            if (exportedComponents.hasOwnProperty(name)) {
                var exportedComponent = exportedComponents[name];
                exportedComponent.directive = exportedComponent.extractDirective(injector);
                exportedComponent.$controller = $controller;
                exportedComponent.extractBindings();
                var promise = exportedComponent.compileTemplate(compile, templateCache, httpBackend);
                if (promise)
                    promises.push(promise);
            }
        }
        return Promise.all(promises);
    }
}
class UpgradeNg1ComponentAdapter {
    constructor(linkFn, scope, directive, elementRef, $controller, inputs, outputs, propOuts, checkProperties, propertyMap) {
        this.directive = directive;
        this.inputs = inputs;
        this.outputs = outputs;
        this.propOuts = propOuts;
        this.checkProperties = checkProperties;
        this.propertyMap = propertyMap;
        this.destinationObj = null;
        this.checkLastValues = [];
        var element = elementRef.nativeElement;
        var childNodes = [];
        var childNode;
        while (childNode = element.firstChild) {
            element.removeChild(childNode);
            childNodes.push(childNode);
        }
        var componentScope = scope.$new(!!directive.scope);
        var $element = angular.element(element);
        var controllerType = directive.controller;
        var controller = null;
        if (controllerType) {
            var locals = { $scope: componentScope, $element: $element };
            controller = $controller(controllerType, locals, null, directive.controllerAs);
            $element.data(controllerKey(directive.name), controller);
        }
        var link = directive.link;
        if (typeof link == 'object')
            link = link.pre;
        if (link) {
            var attrs = NOT_SUPPORTED;
            var transcludeFn = NOT_SUPPORTED;
            var linkController = this.resolveRequired($element, directive.require);
            directive.link(componentScope, $element, attrs, linkController, transcludeFn);
        }
        this.destinationObj = directive.bindToController && controller ? controller : componentScope;
        linkFn(componentScope, (clonedElement, scope) => {
            for (var i = 0, ii = clonedElement.length; i < ii; i++) {
                element.appendChild(clonedElement[i]);
            }
        }, { parentBoundTranscludeFn: (scope, cloneAttach) => { cloneAttach(childNodes); } });
        for (var i = 0; i < inputs.length; i++) {
            this[inputs[i]] = null;
        }
        for (var j = 0; j < outputs.length; j++) {
            var emitter = this[outputs[j]] = new EventEmitter();
            this.setComponentProperty(outputs[j], ((emitter) => (value) => emitter.emit(value))(emitter));
        }
        for (var k = 0; k < propOuts.length; k++) {
            this[propOuts[k]] = new EventEmitter();
            this.checkLastValues.push(INITIAL_VALUE);
        }
    }
    ngOnChanges(changes) {
        for (var name in changes) {
            if (changes.hasOwnProperty(name)) {
                var change = changes[name];
                this.setComponentProperty(name, change.currentValue);
            }
        }
    }
    ngDoCheck() {
        var count = 0;
        var destinationObj = this.destinationObj;
        var lastValues = this.checkLastValues;
        var checkProperties = this.checkProperties;
        for (var i = 0; i < checkProperties.length; i++) {
            var value = destinationObj[checkProperties[i]];
            var last = lastValues[i];
            if (value !== last) {
                if (typeof value == 'number' && isNaN(value) && typeof last == 'number' && isNaN(last)) {
                }
                else {
                    var eventEmitter = this[this.propOuts[i]];
                    eventEmitter.emit(lastValues[i] = value);
                }
            }
        }
        return count;
    }
    setComponentProperty(name, value) {
        this.destinationObj[this.propertyMap[name]] = value;
    }
    resolveRequired($element, require) {
        if (!require) {
            return undefined;
        }
        else if (typeof require == 'string') {
            var name = require;
            var isOptional = false;
            var startParent = false;
            var searchParents = false;
            var ch;
            if (name.charAt(0) == '?') {
                isOptional = true;
                name = name.substr(1);
            }
            if (name.charAt(0) == '^') {
                searchParents = true;
                name = name.substr(1);
            }
            if (name.charAt(0) == '^') {
                startParent = true;
                name = name.substr(1);
            }
            var key = controllerKey(name);
            if (startParent)
                $element = $element.parent();
            var dep = searchParents ? $element.inheritedData(key) : $element.data(key);
            if (!dep && !isOptional) {
                throw new Error(`Can not locate '${require}' in '${this.directive.name}'.`);
            }
            return dep;
        }
        else if (require instanceof Array) {
            var deps = [];
            for (var i = 0; i < require.length; i++) {
                deps.push(this.resolveRequired($element, require[i]));
            }
            return deps;
        }
        throw new Error(`Directive '${this.directive.name}' require syntax unrecognized: ${this.directive.require}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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