import { assertionsEnabled, isPresent, isBlank } from 'angular2/src/facade/lang';
import { ListWrapper } from 'angular2/src/facade/collection';
import { ChangeDetectionUtil } from './change_detection_util';
import { ChangeDetectorRef_ } from './change_detector_ref';
import { ChangeDetectionError, ExpressionChangedAfterItHasBeenCheckedException, DehydratedException } from './exceptions';
import { ChangeDetectionStrategy, ChangeDetectorState } from './constants';
import { wtfCreateScope, wtfLeave } from '../profile/profile';
import { isObservable } from './observable_facade';
var _scope_check = wtfCreateScope(`ChangeDetector#check(ascii id, bool throwOnChange)`);
class _Context {
    constructor(element, componentElement, context, locals, injector, expression) {
        this.element = element;
        this.componentElement = componentElement;
        this.context = context;
        this.locals = locals;
        this.injector = injector;
        this.expression = expression;
    }
}
export class AbstractChangeDetector {
    constructor(id, dispatcher, numberOfPropertyProtoRecords, bindingTargets, directiveIndices, strategy) {
        this.id = id;
        this.dispatcher = dispatcher;
        this.numberOfPropertyProtoRecords = numberOfPropertyProtoRecords;
        this.bindingTargets = bindingTargets;
        this.directiveIndices = directiveIndices;
        this.strategy = strategy;
        this.contentChildren = [];
        this.viewChildren = [];
        // The names of the below fields must be kept in sync with codegen_name_util.ts or
        // change detection will fail.
        this.state = ChangeDetectorState.NeverChecked;
        this.locals = null;
        this.mode = null;
        this.pipes = null;
        this.ref = new ChangeDetectorRef_(this);
    }
    addContentChild(cd) {
        this.contentChildren.push(cd);
        cd.parent = this;
    }
    removeContentChild(cd) { ListWrapper.remove(this.contentChildren, cd); }
    addViewChild(cd) {
        this.viewChildren.push(cd);
        cd.parent = this;
    }
    removeViewChild(cd) { ListWrapper.remove(this.viewChildren, cd); }
    remove() { this.parent.removeContentChild(this); }
    handleEvent(eventName, elIndex, locals) {
        var res = this.handleEventInternal(eventName, elIndex, locals);
        this.markPathToRootAsCheckOnce();
        return res;
    }
    handleEventInternal(eventName, elIndex, locals) { return false; }
    detectChanges() { this.runDetectChanges(false); }
    checkNoChanges() {
        if (assertionsEnabled()) {
            this.runDetectChanges(true);
        }
    }
    runDetectChanges(throwOnChange) {
        if (this.mode === ChangeDetectionStrategy.Detached ||
            this.mode === ChangeDetectionStrategy.Checked || this.state === ChangeDetectorState.Errored)
            return;
        var s = _scope_check(this.id, throwOnChange);
        this.detectChangesInRecords(throwOnChange);
        this._detectChangesContentChildren(throwOnChange);
        if (!throwOnChange)
            this.afterContentLifecycleCallbacks();
        this._detectChangesInViewChildren(throwOnChange);
        if (!throwOnChange)
            this.afterViewLifecycleCallbacks();
        if (this.mode === ChangeDetectionStrategy.CheckOnce)
            this.mode = ChangeDetectionStrategy.Checked;
        this.state = ChangeDetectorState.CheckedBefore;
        wtfLeave(s);
    }
    // This method is not intended to be overridden. Subclasses should instead provide an
    // implementation of `detectChangesInRecordsInternal` which does the work of detecting changes
    // and which this method will call.
    // This method expects that `detectChangesInRecordsInternal` will set the property
    // `this.propertyBindingIndex` to the propertyBindingIndex of the first proto record. This is to
    // facilitate error reporting.
    detectChangesInRecords(throwOnChange) {
        if (!this.hydrated()) {
            this.throwDehydratedError();
        }
        try {
            this.detectChangesInRecordsInternal(throwOnChange);
        }
        catch (e) {
            // throwOnChange errors aren't counted as fatal errors.
            if (!(e instanceof ExpressionChangedAfterItHasBeenCheckedException)) {
                this.state = ChangeDetectorState.Errored;
            }
            this._throwError(e, e.stack);
        }
    }
    // Subclasses should override this method to perform any work necessary to detect and report
    // changes. For example, changes should be reported via `ChangeDetectionUtil.addChange`, lifecycle
    // methods should be called, etc.
    // This implementation should also set `this.propertyBindingIndex` to the propertyBindingIndex of
    // the
    // first proto record to facilitate error reporting. See {@link #detectChangesInRecords}.
    detectChangesInRecordsInternal(throwOnChange) { }
    // This method is not intended to be overridden. Subclasses should instead provide an
    // implementation of `hydrateDirectives`.
    hydrate(context, locals, directives, pipes) {
        this.mode = ChangeDetectionUtil.changeDetectionMode(this.strategy);
        this.context = context;
        if (this.strategy === ChangeDetectionStrategy.OnPushObserve) {
            this.observeComponent(context);
        }
        this.locals = locals;
        this.pipes = pipes;
        this.hydrateDirectives(directives);
        this.state = ChangeDetectorState.NeverChecked;
    }
    // Subclasses should override this method to hydrate any directives.
    hydrateDirectives(directives) { }
    // This method is not intended to be overridden. Subclasses should instead provide an
    // implementation of `dehydrateDirectives`.
    dehydrate() {
        this.dehydrateDirectives(true);
        // This is an experimental feature. Works only in Dart.
        if (this.strategy === ChangeDetectionStrategy.OnPushObserve) {
            this._unsubsribeFromObservables();
        }
        this.context = null;
        this.locals = null;
        this.pipes = null;
    }
    // Subclasses should override this method to dehydrate any directives. This method should reverse
    // any work done in `hydrateDirectives`.
    dehydrateDirectives(destroyPipes) { }
    hydrated() { return isPresent(this.context); }
    afterContentLifecycleCallbacks() {
        this.dispatcher.notifyAfterContentChecked();
        this.afterContentLifecycleCallbacksInternal();
    }
    afterContentLifecycleCallbacksInternal() { }
    afterViewLifecycleCallbacks() {
        this.dispatcher.notifyAfterViewChecked();
        this.afterViewLifecycleCallbacksInternal();
    }
    afterViewLifecycleCallbacksInternal() { }
    /** @internal */
    _detectChangesContentChildren(throwOnChange) {
        var c = this.contentChildren;
        for (var i = 0; i < c.length; ++i) {
            c[i].runDetectChanges(throwOnChange);
        }
    }
    /** @internal */
    _detectChangesInViewChildren(throwOnChange) {
        var c = this.viewChildren;
        for (var i = 0; i < c.length; ++i) {
            c[i].runDetectChanges(throwOnChange);
        }
    }
    markAsCheckOnce() { this.mode = ChangeDetectionStrategy.CheckOnce; }
    markPathToRootAsCheckOnce() {
        var c = this;
        while (isPresent(c) && c.mode !== ChangeDetectionStrategy.Detached) {
            if (c.mode === ChangeDetectionStrategy.Checked)
                c.mode = ChangeDetectionStrategy.CheckOnce;
            c = c.parent;
        }
    }
    // This is an experimental feature. Works only in Dart.
    _unsubsribeFromObservables() {
        if (isPresent(this.subscriptions)) {
            for (var i = 0; i < this.subscriptions.length; ++i) {
                var s = this.subscriptions[i];
                if (isPresent(this.subscriptions[i])) {
                    s.cancel();
                    this.subscriptions[i] = null;
                }
            }
        }
    }
    // This is an experimental feature. Works only in Dart.
    observeValue(value, index) {
        if (isObservable(value)) {
            this._createArrayToStoreObservables();
            if (isBlank(this.subscriptions[index])) {
                this.streams[index] = value.changes;
                this.subscriptions[index] = value.changes.listen((_) => this.ref.markForCheck());
            }
            else if (this.streams[index] !== value.changes) {
                this.subscriptions[index].cancel();
                this.streams[index] = value.changes;
                this.subscriptions[index] = value.changes.listen((_) => this.ref.markForCheck());
            }
        }
        return value;
    }
    // This is an experimental feature. Works only in Dart.
    observeDirective(value, index) {
        if (isObservable(value)) {
            this._createArrayToStoreObservables();
            var arrayIndex = this.numberOfPropertyProtoRecords + index + 2; // +1 is component
            this.streams[arrayIndex] = value.changes;
            this.subscriptions[arrayIndex] = value.changes.listen((_) => this.ref.markForCheck());
        }
        return value;
    }
    // This is an experimental feature. Works only in Dart.
    observeComponent(value) {
        if (isObservable(value)) {
            this._createArrayToStoreObservables();
            var index = this.numberOfPropertyProtoRecords + 1;
            this.streams[index] = value.changes;
            this.subscriptions[index] = value.changes.listen((_) => this.ref.markForCheck());
        }
        return value;
    }
    _createArrayToStoreObservables() {
        if (isBlank(this.subscriptions)) {
            this.subscriptions = ListWrapper.createFixedSize(this.numberOfPropertyProtoRecords +
                this.directiveIndices.length + 2);
            this.streams = ListWrapper.createFixedSize(this.numberOfPropertyProtoRecords +
                this.directiveIndices.length + 2);
        }
    }
    getDirectiveFor(directives, index) {
        return directives.getDirectiveFor(this.directiveIndices[index]);
    }
    getDetectorFor(directives, index) {
        return directives.getDetectorFor(this.directiveIndices[index]);
    }
    notifyDispatcher(value) {
        this.dispatcher.notifyOnBinding(this._currentBinding(), value);
    }
    logBindingUpdate(value) {
        this.dispatcher.logBindingUpdate(this._currentBinding(), value);
    }
    addChange(changes, oldValue, newValue) {
        if (isBlank(changes)) {
            changes = {};
        }
        changes[this._currentBinding().name] = ChangeDetectionUtil.simpleChange(oldValue, newValue);
        return changes;
    }
    _throwError(exception, stack) {
        var error;
        try {
            var c = this.dispatcher.getDebugContext(this._currentBinding().elementIndex, null);
            var context = isPresent(c) ? new _Context(c.element, c.componentElement, c.context, c.locals, c.injector, this._currentBinding().debug) :
                null;
            error = new ChangeDetectionError(this._currentBinding().debug, exception, stack, context);
        }
        catch (e) {
            // if an error happens during getting the debug context, we throw a ChangeDetectionError
            // without the extra information.
            error = new ChangeDetectionError(null, exception, stack, null);
        }
        throw error;
    }
    throwOnChangeError(oldValue, newValue) {
        throw new ExpressionChangedAfterItHasBeenCheckedException(this._currentBinding().debug, oldValue, newValue, null);
    }
    throwDehydratedError() { throw new DehydratedException(); }
    _currentBinding() {
        return this.bindingTargets[this.propertyBindingIndex];
    }
}
//# sourceMappingURL=data:application/json;base64,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