import { isPresent, isBlank } from 'angular2/src/facade/lang';
import { BaseException } from 'angular2/src/facade/exceptions';
import { PromiseWrapper } from 'angular2/src/facade/promise';
import { Map } from 'angular2/src/facade/collection';
import { ComponentInstruction } from './instruction';
import { PathRecognizer } from './path_recognizer';
export class RouteMatch {
}
export class PathMatch extends RouteMatch {
    constructor(instruction, remaining, remainingAux) {
        super();
        this.instruction = instruction;
        this.remaining = remaining;
        this.remainingAux = remainingAux;
    }
}
export class RedirectMatch extends RouteMatch {
    constructor(redirectTo, specificity) {
        super();
        this.redirectTo = redirectTo;
        this.specificity = specificity;
    }
}
export class RedirectRecognizer {
    constructor(path, redirectTo) {
        this.path = path;
        this.redirectTo = redirectTo;
        this._pathRecognizer = new PathRecognizer(path);
        this.hash = this._pathRecognizer.hash;
    }
    /**
     * Returns `null` or a `ParsedUrl` representing the new path to match
     */
    recognize(beginningSegment) {
        var match = null;
        if (isPresent(this._pathRecognizer.recognize(beginningSegment))) {
            match = new RedirectMatch(this.redirectTo, this._pathRecognizer.specificity);
        }
        return PromiseWrapper.resolve(match);
    }
    generate(params) {
        throw new BaseException(`Tried to generate a redirect.`);
    }
}
// represents something like '/foo/:bar'
export class RouteRecognizer {
    // TODO: cache component instruction instances by params and by ParsedUrl instance
    constructor(path, handler) {
        this.path = path;
        this.handler = handler;
        this.terminal = true;
        this._cache = new Map();
        this._pathRecognizer = new PathRecognizer(path);
        this.specificity = this._pathRecognizer.specificity;
        this.hash = this._pathRecognizer.hash;
        this.terminal = this._pathRecognizer.terminal;
    }
    recognize(beginningSegment) {
        var res = this._pathRecognizer.recognize(beginningSegment);
        if (isBlank(res)) {
            return null;
        }
        return this.handler.resolveComponentType().then((_) => {
            var componentInstruction = this._getInstruction(res['urlPath'], res['urlParams'], res['allParams']);
            return new PathMatch(componentInstruction, res['nextSegment'], res['auxiliary']);
        });
    }
    generate(params) {
        var generated = this._pathRecognizer.generate(params);
        var urlPath = generated['urlPath'];
        var urlParams = generated['urlParams'];
        return this._getInstruction(urlPath, urlParams, params);
    }
    generateComponentPathValues(params) {
        return this._pathRecognizer.generate(params);
    }
    _getInstruction(urlPath, urlParams, params) {
        if (isBlank(this.handler.componentType)) {
            throw new BaseException(`Tried to get instruction before the type was loaded.`);
        }
        var hashKey = urlPath + '?' + urlParams.join('?');
        if (this._cache.has(hashKey)) {
            return this._cache.get(hashKey);
        }
        var instruction = new ComponentInstruction(urlPath, urlParams, this.handler.data, this.handler.componentType, this.terminal, this.specificity, params);
        this._cache.set(hashKey, instruction);
        return instruction;
    }
}
//# sourceMappingURL=data:application/json;base64,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