import { isJsObject, global, isPresent, isBlank, isArray, getSymbolIterator } from 'angular2/src/facade/lang';
export var Map = global.Map;
export var Set = global.Set;
// Safari and Internet Explorer do not support the iterable parameter to the
// Map constructor.  We work around that by manually adding the items.
var createMapFromPairs = (function () {
    try {
        if (new Map([[1, 2]]).size === 1) {
            return function createMapFromPairs(pairs) { return new Map(pairs); };
        }
    }
    catch (e) {
    }
    return function createMapAndPopulateFromPairs(pairs) {
        var map = new Map();
        for (var i = 0; i < pairs.length; i++) {
            var pair = pairs[i];
            map.set(pair[0], pair[1]);
        }
        return map;
    };
})();
var createMapFromMap = (function () {
    try {
        if (new Map(new Map())) {
            return function createMapFromMap(m) { return new Map(m); };
        }
    }
    catch (e) {
    }
    return function createMapAndPopulateFromMap(m) {
        var map = new Map();
        m.forEach((v, k) => { map.set(k, v); });
        return map;
    };
})();
var _clearValues = (function () {
    if ((new Map()).keys().next) {
        return function _clearValues(m) {
            var keyIterator = m.keys();
            var k;
            while (!((k = keyIterator.next()).done)) {
                m.set(k.value, null);
            }
        };
    }
    else {
        return function _clearValuesWithForeEach(m) {
            m.forEach((v, k) => { m.set(k, null); });
        };
    }
})();
// Safari doesn't implement MapIterator.next(), which is used is Traceur's polyfill of Array.from
// TODO(mlaval): remove the work around once we have a working polyfill of Array.from
var _arrayFromMap = (function () {
    try {
        if ((new Map()).values().next) {
            return function createArrayFromMap(m, getValues) {
                return getValues ? Array.from(m.values()) : Array.from(m.keys());
            };
        }
    }
    catch (e) {
    }
    return function createArrayFromMapWithForeach(m, getValues) {
        var res = ListWrapper.createFixedSize(m.size), i = 0;
        m.forEach((v, k) => {
            res[i] = getValues ? v : k;
            i++;
        });
        return res;
    };
})();
export class MapWrapper {
    static clone(m) { return createMapFromMap(m); }
    static createFromStringMap(stringMap) {
        var result = new Map();
        for (var prop in stringMap) {
            result.set(prop, stringMap[prop]);
        }
        return result;
    }
    static toStringMap(m) {
        var r = {};
        m.forEach((v, k) => r[k] = v);
        return r;
    }
    static createFromPairs(pairs) { return createMapFromPairs(pairs); }
    static clearValues(m) { _clearValues(m); }
    static iterable(m) { return m; }
    static keys(m) { return _arrayFromMap(m, false); }
    static values(m) { return _arrayFromMap(m, true); }
}
/**
 * Wraps Javascript Objects
 */
export class StringMapWrapper {
    static create() {
        // Note: We are not using Object.create(null) here due to
        // performance!
        // http://jsperf.com/ng2-object-create-null
        return {};
    }
    static contains(map, key) {
        return map.hasOwnProperty(key);
    }
    static get(map, key) {
        return map.hasOwnProperty(key) ? map[key] : undefined;
    }
    static set(map, key, value) { map[key] = value; }
    static keys(map) { return Object.keys(map); }
    static values(map) {
        return Object.keys(map).reduce((r, a) => {
            r.push(map[a]);
            return r;
        }, []);
    }
    static isEmpty(map) {
        for (var prop in map) {
            return false;
        }
        return true;
    }
    static delete(map, key) { delete map[key]; }
    static forEach(map, callback) {
        for (var prop in map) {
            if (map.hasOwnProperty(prop)) {
                callback(map[prop], prop);
            }
        }
    }
    static merge(m1, m2) {
        var m = {};
        for (var attr in m1) {
            if (m1.hasOwnProperty(attr)) {
                m[attr] = m1[attr];
            }
        }
        for (var attr in m2) {
            if (m2.hasOwnProperty(attr)) {
                m[attr] = m2[attr];
            }
        }
        return m;
    }
    static equals(m1, m2) {
        var k1 = Object.keys(m1);
        var k2 = Object.keys(m2);
        if (k1.length != k2.length) {
            return false;
        }
        var key;
        for (var i = 0; i < k1.length; i++) {
            key = k1[i];
            if (m1[key] !== m2[key]) {
                return false;
            }
        }
        return true;
    }
}
export class ListWrapper {
    // JS has no way to express a statically fixed size list, but dart does so we
    // keep both methods.
    static createFixedSize(size) { return new Array(size); }
    static createGrowableSize(size) { return new Array(size); }
    static clone(array) { return array.slice(0); }
    static createImmutable(array) {
        var result = ListWrapper.clone(array);
        Object.seal(result);
        return result;
    }
    static forEachWithIndex(array, fn) {
        for (var i = 0; i < array.length; i++) {
            fn(array[i], i);
        }
    }
    static first(array) {
        if (!array)
            return null;
        return array[0];
    }
    static last(array) {
        if (!array || array.length == 0)
            return null;
        return array[array.length - 1];
    }
    static indexOf(array, value, startIndex = 0) {
        return array.indexOf(value, startIndex);
    }
    static contains(list, el) { return list.indexOf(el) !== -1; }
    static reversed(array) {
        var a = ListWrapper.clone(array);
        return a.reverse();
    }
    static concat(a, b) { return a.concat(b); }
    static insert(list, index, value) { list.splice(index, 0, value); }
    static removeAt(list, index) {
        var res = list[index];
        list.splice(index, 1);
        return res;
    }
    static removeAll(list, items) {
        for (var i = 0; i < items.length; ++i) {
            var index = list.indexOf(items[i]);
            list.splice(index, 1);
        }
    }
    static remove(list, el) {
        var index = list.indexOf(el);
        if (index > -1) {
            list.splice(index, 1);
            return true;
        }
        return false;
    }
    static clear(list) { list.length = 0; }
    static isEmpty(list) { return list.length == 0; }
    static fill(list, value, start = 0, end = null) {
        list.fill(value, start, end === null ? list.length : end);
    }
    static equals(a, b) {
        if (a.length != b.length)
            return false;
        for (var i = 0; i < a.length; ++i) {
            if (a[i] !== b[i])
                return false;
        }
        return true;
    }
    static slice(l, from = 0, to = null) {
        return l.slice(from, to === null ? undefined : to);
    }
    static splice(l, from, length) { return l.splice(from, length); }
    static sort(l, compareFn) {
        if (isPresent(compareFn)) {
            l.sort(compareFn);
        }
        else {
            l.sort();
        }
    }
    static toString(l) { return l.toString(); }
    static toJSON(l) { return JSON.stringify(l); }
    static maximum(list, predicate) {
        if (list.length == 0) {
            return null;
        }
        var solution = null;
        var maxValue = -Infinity;
        for (var index = 0; index < list.length; index++) {
            var candidate = list[index];
            if (isBlank(candidate)) {
                continue;
            }
            var candidateValue = predicate(candidate);
            if (candidateValue > maxValue) {
                solution = candidate;
                maxValue = candidateValue;
            }
        }
        return solution;
    }
    static isImmutable(list) { return Object.isSealed(list); }
}
export function isListLikeIterable(obj) {
    if (!isJsObject(obj))
        return false;
    return isArray(obj) ||
        (!(obj instanceof Map) &&
            getSymbolIterator() in obj); // JS Iterable have a Symbol.iterator prop
}
export function areIterablesEqual(a, b, comparator) {
    var iterator1 = a[getSymbolIterator()]();
    var iterator2 = b[getSymbolIterator()]();
    while (true) {
        let item1 = iterator1.next();
        let item2 = iterator2.next();
        if (item1.done && item2.done)
            return true;
        if (item1.done || item2.done)
            return false;
        if (!comparator(item1.value, item2.value))
            return false;
    }
}
export function iterateListLike(obj, fn) {
    if (isArray(obj)) {
        for (var i = 0; i < obj.length; i++) {
            fn(obj[i]);
        }
    }
    else {
        var iterator = obj[getSymbolIterator()]();
        var item;
        while (!((item = iterator.next()).done)) {
            fn(item.value);
        }
    }
}
// Safari and Internet Explorer do not support the iterable parameter to the
// Set constructor.  We work around that by manually adding the items.
var createSetFromList = (function () {
    var test = new Set([1, 2, 3]);
    if (test.size === 3) {
        return function createSetFromList(lst) { return new Set(lst); };
    }
    else {
        return function createSetAndPopulateFromList(lst) {
            var res = new Set(lst);
            if (res.size !== lst.length) {
                for (var i = 0; i < lst.length; i++) {
                    res.add(lst[i]);
                }
            }
            return res;
        };
    }
})();
export class SetWrapper {
    static createFromList(lst) { return createSetFromList(lst); }
    static has(s, key) { return s.has(key); }
    static delete(m, k) { m.delete(k); }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29sbGVjdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImFuZ3VsYXIyL3NyYy9mYWNhZGUvY29sbGVjdGlvbi50cyJdLCJuYW1lcyI6WyJjcmVhdGVNYXBGcm9tUGFpcnMiLCJjcmVhdGVNYXBBbmRQb3B1bGF0ZUZyb21QYWlycyIsImNyZWF0ZU1hcEZyb21NYXAiLCJjcmVhdGVNYXBBbmRQb3B1bGF0ZUZyb21NYXAiLCJfY2xlYXJWYWx1ZXMiLCJfY2xlYXJWYWx1ZXNXaXRoRm9yZUVhY2giLCJjcmVhdGVBcnJheUZyb21NYXAiLCJjcmVhdGVBcnJheUZyb21NYXBXaXRoRm9yZWFjaCIsIk1hcFdyYXBwZXIiLCJNYXBXcmFwcGVyLmNsb25lIiwiTWFwV3JhcHBlci5jcmVhdGVGcm9tU3RyaW5nTWFwIiwiTWFwV3JhcHBlci50b1N0cmluZ01hcCIsIk1hcFdyYXBwZXIuY3JlYXRlRnJvbVBhaXJzIiwiTWFwV3JhcHBlci5jbGVhclZhbHVlcyIsIk1hcFdyYXBwZXIuaXRlcmFibGUiLCJNYXBXcmFwcGVyLmtleXMiLCJNYXBXcmFwcGVyLnZhbHVlcyIsIlN0cmluZ01hcFdyYXBwZXIiLCJTdHJpbmdNYXBXcmFwcGVyLmNyZWF0ZSIsIlN0cmluZ01hcFdyYXBwZXIuY29udGFpbnMiLCJTdHJpbmdNYXBXcmFwcGVyLmdldCIsIlN0cmluZ01hcFdyYXBwZXIuc2V0IiwiU3RyaW5nTWFwV3JhcHBlci5rZXlzIiwiU3RyaW5nTWFwV3JhcHBlci52YWx1ZXMiLCJTdHJpbmdNYXBXcmFwcGVyLmlzRW1wdHkiLCJTdHJpbmdNYXBXcmFwcGVyLmRlbGV0ZSIsIlN0cmluZ01hcFdyYXBwZXIuZm9yRWFjaCIsIlN0cmluZ01hcFdyYXBwZXIubWVyZ2UiLCJTdHJpbmdNYXBXcmFwcGVyLmVxdWFscyIsIkxpc3RXcmFwcGVyIiwiTGlzdFdyYXBwZXIuY3JlYXRlRml4ZWRTaXplIiwiTGlzdFdyYXBwZXIuY3JlYXRlR3Jvd2FibGVTaXplIiwiTGlzdFdyYXBwZXIuY2xvbmUiLCJMaXN0V3JhcHBlci5jcmVhdGVJbW11dGFibGUiLCJMaXN0V3JhcHBlci5mb3JFYWNoV2l0aEluZGV4IiwiTGlzdFdyYXBwZXIuZmlyc3QiLCJMaXN0V3JhcHBlci5sYXN0IiwiTGlzdFdyYXBwZXIuaW5kZXhPZiIsIkxpc3RXcmFwcGVyLmNvbnRhaW5zIiwiTGlzdFdyYXBwZXIucmV2ZXJzZWQiLCJMaXN0V3JhcHBlci5jb25jYXQiLCJMaXN0V3JhcHBlci5pbnNlcnQiLCJMaXN0V3JhcHBlci5yZW1vdmVBdCIsIkxpc3RXcmFwcGVyLnJlbW92ZUFsbCIsIkxpc3RXcmFwcGVyLnJlbW92ZSIsIkxpc3RXcmFwcGVyLmNsZWFyIiwiTGlzdFdyYXBwZXIuaXNFbXB0eSIsIkxpc3RXcmFwcGVyLmZpbGwiLCJMaXN0V3JhcHBlci5lcXVhbHMiLCJMaXN0V3JhcHBlci5zbGljZSIsIkxpc3RXcmFwcGVyLnNwbGljZSIsIkxpc3RXcmFwcGVyLnNvcnQiLCJMaXN0V3JhcHBlci50b1N0cmluZyIsIkxpc3RXcmFwcGVyLnRvSlNPTiIsIkxpc3RXcmFwcGVyLm1heGltdW0iLCJMaXN0V3JhcHBlci5pc0ltbXV0YWJsZSIsImlzTGlzdExpa2VJdGVyYWJsZSIsImFyZUl0ZXJhYmxlc0VxdWFsIiwiaXRlcmF0ZUxpc3RMaWtlIiwiY3JlYXRlU2V0RnJvbUxpc3QiLCJjcmVhdGVTZXRBbmRQb3B1bGF0ZUZyb21MaXN0IiwiU2V0V3JhcHBlciIsIlNldFdyYXBwZXIuY3JlYXRlRnJvbUxpc3QiLCJTZXRXcmFwcGVyLmhhcyIsIlNldFdyYXBwZXIuZGVsZXRlIl0sIm1hcHBpbmdzIjoiT0FBTyxFQUNMLFVBQVUsRUFDVixNQUFNLEVBQ04sU0FBUyxFQUNULE9BQU8sRUFDUCxPQUFPLEVBQ1AsaUJBQWlCLEVBQ2xCLE1BQU0sMEJBQTBCO0FBRWpDLFdBQVcsR0FBRyxHQUFHLE1BQU0sQ0FBQyxHQUFHLENBQUM7QUFDNUIsV0FBVyxHQUFHLEdBQUcsTUFBTSxDQUFDLEdBQUcsQ0FBQztBQUU1Qiw0RUFBNEU7QUFDNUUsc0VBQXNFO0FBQ3RFLElBQUksa0JBQWtCLEdBQW9DLENBQUM7SUFDekQsSUFBSSxDQUFDO1FBQ0gsRUFBRSxDQUFDLENBQUMsSUFBSSxHQUFHLENBQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDdEMsTUFBTSxDQUFDLDRCQUE0QixLQUFZLElBQW1CQSxNQUFNQSxDQUFDQSxJQUFJQSxHQUFHQSxDQUFDQSxLQUFLQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDO1FBQzdGLENBQUM7SUFDSCxDQUFFO0lBQUEsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUNiLENBQUM7SUFDRCxNQUFNLENBQUMsdUNBQXVDLEtBQVk7UUFDeERDLElBQUlBLEdBQUdBLEdBQUdBLElBQUlBLEdBQUdBLEVBQUVBLENBQUNBO1FBQ3BCQSxHQUFHQSxDQUFDQSxDQUFDQSxHQUFHQSxDQUFDQSxDQUFDQSxHQUFHQSxDQUFDQSxFQUFFQSxDQUFDQSxHQUFHQSxLQUFLQSxDQUFDQSxNQUFNQSxFQUFFQSxDQUFDQSxFQUFFQSxFQUFFQSxDQUFDQTtZQUN0Q0EsSUFBSUEsSUFBSUEsR0FBR0EsS0FBS0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0E7WUFDcEJBLEdBQUdBLENBQUNBLEdBQUdBLENBQUNBLElBQUlBLENBQUNBLENBQUNBLENBQUNBLEVBQUVBLElBQUlBLENBQUNBLENBQUNBLENBQUNBLENBQUNBLENBQUNBO1FBQzVCQSxDQUFDQTtRQUNEQSxNQUFNQSxDQUFDQSxHQUFHQSxDQUFDQTtJQUNiQSxDQUFDQSxDQUFDO0FBQ0osQ0FBQyxDQUFDLEVBQUUsQ0FBQztBQUNMLElBQUksZ0JBQWdCLEdBQXdDLENBQUM7SUFDM0QsSUFBSSxDQUFDO1FBQ0gsRUFBRSxDQUFDLENBQUMsSUFBSSxHQUFHLENBQU0sSUFBSSxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUM1QixNQUFNLENBQUMsMEJBQTBCLENBQWdCLElBQW1CQyxNQUFNQSxDQUFDQSxJQUFJQSxHQUFHQSxDQUFNQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDO1FBQ2hHLENBQUM7SUFDSCxDQUFFO0lBQUEsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUNiLENBQUM7SUFDRCxNQUFNLENBQUMscUNBQXFDLENBQWdCO1FBQzFEQyxJQUFJQSxHQUFHQSxHQUFHQSxJQUFJQSxHQUFHQSxFQUFFQSxDQUFDQTtRQUNwQkEsQ0FBQ0EsQ0FBQ0EsT0FBT0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsRUFBRUEsQ0FBQ0EsT0FBT0EsR0FBR0EsQ0FBQ0EsR0FBR0EsQ0FBQ0EsQ0FBQ0EsRUFBRUEsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0E7UUFDeENBLE1BQU1BLENBQUNBLEdBQUdBLENBQUNBO0lBQ2JBLENBQUNBLENBQUM7QUFDSixDQUFDLENBQUMsRUFBRSxDQUFDO0FBQ0wsSUFBSSxZQUFZLEdBQXlCLENBQUM7SUFDeEMsRUFBRSxDQUFDLENBQU8sQ0FBQyxJQUFJLEdBQUcsRUFBRSxDQUFDLENBQUMsSUFBSSxFQUFHLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztRQUNuQyxNQUFNLENBQUMsc0JBQXNCLENBQWdCO1lBQzNDQyxJQUFJQSxXQUFXQSxHQUFHQSxDQUFDQSxDQUFDQSxJQUFJQSxFQUFFQSxDQUFDQTtZQUMzQkEsSUFBSUEsQ0FBQ0EsQ0FBQ0E7WUFDTkEsT0FBT0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsR0FBU0EsV0FBWUEsQ0FBQ0EsSUFBSUEsRUFBRUEsQ0FBQ0EsQ0FBQ0EsSUFBSUEsQ0FBQ0EsRUFBRUEsQ0FBQ0E7Z0JBQy9DQSxDQUFDQSxDQUFDQSxHQUFHQSxDQUFDQSxDQUFDQSxDQUFDQSxLQUFLQSxFQUFFQSxJQUFJQSxDQUFDQSxDQUFDQTtZQUN2QkEsQ0FBQ0E7UUFDSEEsQ0FBQ0EsQ0FBQztJQUNKLENBQUM7SUFBQyxJQUFJLENBQUMsQ0FBQztRQUNOLE1BQU0sQ0FBQyxrQ0FBa0MsQ0FBZ0I7WUFDdkRDLENBQUNBLENBQUNBLE9BQU9BLENBQUNBLENBQUNBLENBQUNBLEVBQUVBLENBQUNBLE9BQU9BLENBQUNBLENBQUNBLEdBQUdBLENBQUNBLENBQUNBLEVBQUVBLElBQUlBLENBQUNBLENBQUNBLENBQUNBLENBQUNBLENBQUNBLENBQUNBO1FBQzNDQSxDQUFDQSxDQUFDO0lBQ0osQ0FBQztBQUNILENBQUMsQ0FBQyxFQUFFLENBQUM7QUFDTCxpR0FBaUc7QUFDakcscUZBQXFGO0FBQ3JGLElBQUksYUFBYSxHQUFvRCxDQUFDO0lBQ3BFLElBQUksQ0FBQztRQUNILEVBQUUsQ0FBQyxDQUFPLENBQUMsSUFBSSxHQUFHLEVBQUUsQ0FBQyxDQUFDLE1BQU0sRUFBRyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7WUFDckMsTUFBTSxDQUFDLDRCQUE0QixDQUFnQixFQUFFLFNBQWtCO2dCQUNyRUMsTUFBTUEsQ0FBQ0EsU0FBU0EsR0FBU0EsS0FBTUEsQ0FBQ0EsSUFBSUEsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsTUFBTUEsRUFBRUEsQ0FBQ0EsR0FBU0EsS0FBTUEsQ0FBQ0EsSUFBSUEsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsSUFBSUEsRUFBRUEsQ0FBQ0EsQ0FBQ0E7WUFDakZBLENBQUNBLENBQUM7UUFDSixDQUFDO0lBQ0gsQ0FBRTtJQUFBLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDYixDQUFDO0lBQ0QsTUFBTSxDQUFDLHVDQUF1QyxDQUFnQixFQUFFLFNBQWtCO1FBQ2hGQyxJQUFJQSxHQUFHQSxHQUFHQSxXQUFXQSxDQUFDQSxlQUFlQSxDQUFDQSxDQUFDQSxDQUFDQSxJQUFJQSxDQUFDQSxFQUFFQSxDQUFDQSxHQUFHQSxDQUFDQSxDQUFDQTtRQUNyREEsQ0FBQ0EsQ0FBQ0EsT0FBT0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsRUFBRUEsQ0FBQ0E7WUFDYkEsR0FBR0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsR0FBR0EsU0FBU0EsR0FBR0EsQ0FBQ0EsR0FBR0EsQ0FBQ0EsQ0FBQ0E7WUFDM0JBLENBQUNBLEVBQUVBLENBQUNBO1FBQ05BLENBQUNBLENBQUNBLENBQUNBO1FBQ0hBLE1BQU1BLENBQUNBLEdBQUdBLENBQUNBO0lBQ2JBLENBQUNBLENBQUM7QUFDSixDQUFDLENBQUMsRUFBRSxDQUFDO0FBRUw7SUFDRUMsT0FBT0EsS0FBS0EsQ0FBT0EsQ0FBWUEsSUFBZUMsTUFBTUEsQ0FBQ0EsZ0JBQWdCQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQTtJQUMzRUQsT0FBT0EsbUJBQW1CQSxDQUFJQSxTQUE2QkE7UUFDekRFLElBQUlBLE1BQU1BLEdBQUdBLElBQUlBLEdBQUdBLEVBQWFBLENBQUNBO1FBQ2xDQSxHQUFHQSxDQUFDQSxDQUFDQSxHQUFHQSxDQUFDQSxJQUFJQSxJQUFJQSxTQUFTQSxDQUFDQSxDQUFDQSxDQUFDQTtZQUMzQkEsTUFBTUEsQ0FBQ0EsR0FBR0EsQ0FBQ0EsSUFBSUEsRUFBRUEsU0FBU0EsQ0FBQ0EsSUFBSUEsQ0FBQ0EsQ0FBQ0EsQ0FBQ0E7UUFDcENBLENBQUNBO1FBQ0RBLE1BQU1BLENBQUNBLE1BQU1BLENBQUNBO0lBQ2hCQSxDQUFDQTtJQUNERixPQUFPQSxXQUFXQSxDQUFJQSxDQUFpQkE7UUFDckNHLElBQUlBLENBQUNBLEdBQXVCQSxFQUFFQSxDQUFDQTtRQUMvQkEsQ0FBQ0EsQ0FBQ0EsT0FBT0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsRUFBRUEsQ0FBQ0EsS0FBS0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsR0FBR0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0E7UUFDOUJBLE1BQU1BLENBQUNBLENBQUNBLENBQUNBO0lBQ1hBLENBQUNBO0lBQ0RILE9BQU9BLGVBQWVBLENBQUNBLEtBQVlBLElBQW1CSSxNQUFNQSxDQUFDQSxrQkFBa0JBLENBQUNBLEtBQUtBLENBQUNBLENBQUNBLENBQUNBLENBQUNBO0lBQ3pGSixPQUFPQSxXQUFXQSxDQUFDQSxDQUFnQkEsSUFBSUssWUFBWUEsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0E7SUFDekRMLE9BQU9BLFFBQVFBLENBQUlBLENBQUlBLElBQU9NLE1BQU1BLENBQUNBLENBQUNBLENBQUNBLENBQUNBLENBQUNBO0lBQ3pDTixPQUFPQSxJQUFJQSxDQUFJQSxDQUFjQSxJQUFTTyxNQUFNQSxDQUFDQSxhQUFhQSxDQUFDQSxDQUFDQSxFQUFFQSxLQUFLQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQTtJQUN2RVAsT0FBT0EsTUFBTUEsQ0FBSUEsQ0FBY0EsSUFBU1EsTUFBTUEsQ0FBQ0EsYUFBYUEsQ0FBQ0EsQ0FBQ0EsRUFBRUEsSUFBSUEsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0E7QUFDMUVSLENBQUNBO0FBRUQ7O0dBRUc7QUFDSDtJQUNFUyxPQUFPQSxNQUFNQTtRQUNYQyx5REFBeURBO1FBQ3pEQSxlQUFlQTtRQUNmQSwyQ0FBMkNBO1FBQzNDQSxNQUFNQSxDQUFDQSxFQUFFQSxDQUFDQTtJQUNaQSxDQUFDQTtJQUNERCxPQUFPQSxRQUFRQSxDQUFDQSxHQUF5QkEsRUFBRUEsR0FBV0E7UUFDcERFLE1BQU1BLENBQUNBLEdBQUdBLENBQUNBLGNBQWNBLENBQUNBLEdBQUdBLENBQUNBLENBQUNBO0lBQ2pDQSxDQUFDQTtJQUNERixPQUFPQSxHQUFHQSxDQUFJQSxHQUF1QkEsRUFBRUEsR0FBV0E7UUFDaERHLE1BQU1BLENBQUNBLEdBQUdBLENBQUNBLGNBQWNBLENBQUNBLEdBQUdBLENBQUNBLEdBQUdBLEdBQUdBLENBQUNBLEdBQUdBLENBQUNBLEdBQUdBLFNBQVNBLENBQUNBO0lBQ3hEQSxDQUFDQTtJQUNESCxPQUFPQSxHQUFHQSxDQUFJQSxHQUF1QkEsRUFBRUEsR0FBV0EsRUFBRUEsS0FBUUEsSUFBSUksR0FBR0EsQ0FBQ0EsR0FBR0EsQ0FBQ0EsR0FBR0EsS0FBS0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0E7SUFDbkZKLE9BQU9BLElBQUlBLENBQUNBLEdBQXlCQSxJQUFjSyxNQUFNQSxDQUFDQSxNQUFNQSxDQUFDQSxJQUFJQSxDQUFDQSxHQUFHQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQTtJQUM3RUwsT0FBT0EsTUFBTUEsQ0FBSUEsR0FBdUJBO1FBQ3RDTSxNQUFNQSxDQUFDQSxNQUFNQSxDQUFDQSxJQUFJQSxDQUFDQSxHQUFHQSxDQUFDQSxDQUFDQSxNQUFNQSxDQUFDQSxDQUFDQSxDQUFDQSxFQUFFQSxDQUFDQTtZQUNsQ0EsQ0FBQ0EsQ0FBQ0EsSUFBSUEsQ0FBQ0EsR0FBR0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0E7WUFDZkEsTUFBTUEsQ0FBQ0EsQ0FBQ0EsQ0FBQ0E7UUFDWEEsQ0FBQ0EsRUFBRUEsRUFBRUEsQ0FBQ0EsQ0FBQ0E7SUFDVEEsQ0FBQ0E7SUFDRE4sT0FBT0EsT0FBT0EsQ0FBQ0EsR0FBeUJBO1FBQ3RDTyxHQUFHQSxDQUFDQSxDQUFDQSxHQUFHQSxDQUFDQSxJQUFJQSxJQUFJQSxHQUFHQSxDQUFDQSxDQUFDQSxDQUFDQTtZQUNyQkEsTUFBTUEsQ0FBQ0EsS0FBS0EsQ0FBQ0E7UUFDZkEsQ0FBQ0E7UUFDREEsTUFBTUEsQ0FBQ0EsSUFBSUEsQ0FBQ0E7SUFDZEEsQ0FBQ0E7SUFDRFAsT0FBT0EsTUFBTUEsQ0FBRUEsR0FBeUJBLEVBQUVBLEdBQVdBLElBQUlRLE9BQU9BLEdBQUdBLENBQUNBLEdBQUdBLENBQUNBLENBQUNBLENBQUNBLENBQUNBO0lBQzNFUixPQUFPQSxPQUFPQSxDQUFPQSxHQUF1QkEsRUFBRUEsUUFBcUNBO1FBQ2pGUyxHQUFHQSxDQUFDQSxDQUFDQSxHQUFHQSxDQUFDQSxJQUFJQSxJQUFJQSxHQUFHQSxDQUFDQSxDQUFDQSxDQUFDQTtZQUNyQkEsRUFBRUEsQ0FBQ0EsQ0FBQ0EsR0FBR0EsQ0FBQ0EsY0FBY0EsQ0FBQ0EsSUFBSUEsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0E7Z0JBQzdCQSxRQUFRQSxDQUFDQSxHQUFHQSxDQUFDQSxJQUFJQSxDQUFDQSxFQUFFQSxJQUFJQSxDQUFDQSxDQUFDQTtZQUM1QkEsQ0FBQ0E7UUFDSEEsQ0FBQ0E7SUFDSEEsQ0FBQ0E7SUFFRFQsT0FBT0EsS0FBS0EsQ0FBSUEsRUFBc0JBLEVBQUVBLEVBQXNCQTtRQUM1RFUsSUFBSUEsQ0FBQ0EsR0FBdUJBLEVBQUVBLENBQUNBO1FBRS9CQSxHQUFHQSxDQUFDQSxDQUFDQSxHQUFHQSxDQUFDQSxJQUFJQSxJQUFJQSxFQUFFQSxDQUFDQSxDQUFDQSxDQUFDQTtZQUNwQkEsRUFBRUEsQ0FBQ0EsQ0FBQ0EsRUFBRUEsQ0FBQ0EsY0FBY0EsQ0FBQ0EsSUFBSUEsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0E7Z0JBQzVCQSxDQUFDQSxDQUFDQSxJQUFJQSxDQUFDQSxHQUFHQSxFQUFFQSxDQUFDQSxJQUFJQSxDQUFDQSxDQUFDQTtZQUNyQkEsQ0FBQ0E7UUFDSEEsQ0FBQ0E7UUFFREEsR0FBR0EsQ0FBQ0EsQ0FBQ0EsR0FBR0EsQ0FBQ0EsSUFBSUEsSUFBSUEsRUFBRUEsQ0FBQ0EsQ0FBQ0EsQ0FBQ0E7WUFDcEJBLEVBQUVBLENBQUNBLENBQUNBLEVBQUVBLENBQUNBLGNBQWNBLENBQUNBLElBQUlBLENBQUNBLENBQUNBLENBQUNBLENBQUNBO2dCQUM1QkEsQ0FBQ0EsQ0FBQ0EsSUFBSUEsQ0FBQ0EsR0FBR0EsRUFBRUEsQ0FBQ0EsSUFBSUEsQ0FBQ0EsQ0FBQ0E7WUFDckJBLENBQUNBO1FBQ0hBLENBQUNBO1FBRURBLE1BQU1BLENBQUNBLENBQUNBLENBQUNBO0lBQ1hBLENBQUNBO0lBRURWLE9BQU9BLE1BQU1BLENBQUlBLEVBQXNCQSxFQUFFQSxFQUFzQkE7UUFDN0RXLElBQUlBLEVBQUVBLEdBQUdBLE1BQU1BLENBQUNBLElBQUlBLENBQUNBLEVBQUVBLENBQUNBLENBQUNBO1FBQ3pCQSxJQUFJQSxFQUFFQSxHQUFHQSxNQUFNQSxDQUFDQSxJQUFJQSxDQUFDQSxFQUFFQSxDQUFDQSxDQUFDQTtRQUN6QkEsRUFBRUEsQ0FBQ0EsQ0FBQ0EsRUFBRUEsQ0FBQ0EsTUFBTUEsSUFBSUEsRUFBRUEsQ0FBQ0EsTUFBTUEsQ0FBQ0EsQ0FBQ0EsQ0FBQ0E7WUFDM0JBLE1BQU1BLENBQUNBLEtBQUtBLENBQUNBO1FBQ2ZBLENBQUNBO1FBQ0RBLElBQUlBLEdBQUdBLENBQUNBO1FBQ1JBLEdBQUdBLENBQUNBLENBQUNBLEdBQUdBLENBQUNBLENBQUNBLEdBQUdBLENBQUNBLEVBQUVBLENBQUNBLEdBQUdBLEVBQUVBLENBQUNBLE1BQU1BLEVBQUVBLENBQUNBLEVBQUVBLEVBQUVBLENBQUNBO1lBQ25DQSxHQUFHQSxHQUFHQSxFQUFFQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQTtZQUNaQSxFQUFFQSxDQUFDQSxDQUFDQSxFQUFFQSxDQUFDQSxHQUFHQSxDQUFDQSxLQUFLQSxFQUFFQSxDQUFDQSxHQUFHQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQTtnQkFDeEJBLE1BQU1BLENBQUNBLEtBQUtBLENBQUNBO1lBQ2ZBLENBQUNBO1FBQ0hBLENBQUNBO1FBQ0RBLE1BQU1BLENBQUNBLElBQUlBLENBQUNBO0lBQ2RBLENBQUNBO0FBQ0hYLENBQUNBO0FBUUQ7SUFDRVksNkVBQTZFQTtJQUM3RUEscUJBQXFCQTtJQUNyQkEsT0FBT0EsZUFBZUEsQ0FBQ0EsSUFBWUEsSUFBV0MsTUFBTUEsQ0FBQ0EsSUFBSUEsS0FBS0EsQ0FBQ0EsSUFBSUEsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0E7SUFDdkVELE9BQU9BLGtCQUFrQkEsQ0FBQ0EsSUFBWUEsSUFBV0UsTUFBTUEsQ0FBQ0EsSUFBSUEsS0FBS0EsQ0FBQ0EsSUFBSUEsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0E7SUFDMUVGLE9BQU9BLEtBQUtBLENBQUlBLEtBQVVBLElBQVNHLE1BQU1BLENBQUNBLEtBQUtBLENBQUNBLEtBQUtBLENBQUNBLENBQUNBLENBQUNBLENBQUNBLENBQUNBLENBQUNBO0lBQzNESCxPQUFPQSxlQUFlQSxDQUFJQSxLQUFVQTtRQUNsQ0ksSUFBSUEsTUFBTUEsR0FBR0EsV0FBV0EsQ0FBQ0EsS0FBS0EsQ0FBQ0EsS0FBS0EsQ0FBQ0EsQ0FBQ0E7UUFDdENBLE1BQU1BLENBQUNBLElBQUlBLENBQUNBLE1BQU1BLENBQUNBLENBQUNBO1FBQ3BCQSxNQUFNQSxDQUFDQSxNQUFNQSxDQUFDQTtJQUNoQkEsQ0FBQ0E7SUFDREosT0FBT0EsZ0JBQWdCQSxDQUFJQSxLQUFVQSxFQUFFQSxFQUE2QkE7UUFDbEVLLEdBQUdBLENBQUNBLENBQUNBLEdBQUdBLENBQUNBLENBQUNBLEdBQUdBLENBQUNBLEVBQUVBLENBQUNBLEdBQUdBLEtBQUtBLENBQUNBLE1BQU1BLEVBQUVBLENBQUNBLEVBQUVBLEVBQUVBLENBQUNBO1lBQ3RDQSxFQUFFQSxDQUFDQSxLQUFLQSxDQUFDQSxDQUFDQSxDQUFDQSxFQUFFQSxDQUFDQSxDQUFDQSxDQUFDQTtRQUNsQkEsQ0FBQ0E7SUFDSEEsQ0FBQ0E7SUFDREwsT0FBT0EsS0FBS0EsQ0FBSUEsS0FBVUE7UUFDeEJNLEVBQUVBLENBQUNBLENBQUNBLENBQUNBLEtBQUtBLENBQUNBO1lBQUNBLE1BQU1BLENBQUNBLElBQUlBLENBQUNBO1FBQ3hCQSxNQUFNQSxDQUFDQSxLQUFLQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQTtJQUNsQkEsQ0FBQ0E7SUFDRE4sT0FBT0EsSUFBSUEsQ0FBSUEsS0FBVUE7UUFDdkJPLEVBQUVBLENBQUNBLENBQUNBLENBQUNBLEtBQUtBLElBQUlBLEtBQUtBLENBQUNBLE1BQU1BLElBQUlBLENBQUNBLENBQUNBO1lBQUNBLE1BQU1BLENBQUNBLElBQUlBLENBQUNBO1FBQzdDQSxNQUFNQSxDQUFDQSxLQUFLQSxDQUFDQSxLQUFLQSxDQUFDQSxNQUFNQSxHQUFHQSxDQUFDQSxDQUFDQSxDQUFDQTtJQUNqQ0EsQ0FBQ0E7SUFDRFAsT0FBT0EsT0FBT0EsQ0FBSUEsS0FBVUEsRUFBRUEsS0FBUUEsRUFBRUEsVUFBVUEsR0FBV0EsQ0FBQ0E7UUFDNURRLE1BQU1BLENBQUNBLEtBQUtBLENBQUNBLE9BQU9BLENBQUNBLEtBQUtBLEVBQUVBLFVBQVVBLENBQUNBLENBQUNBO0lBQzFDQSxDQUFDQTtJQUNEUixPQUFPQSxRQUFRQSxDQUFJQSxJQUFTQSxFQUFFQSxFQUFLQSxJQUFhUyxNQUFNQSxDQUFDQSxJQUFJQSxDQUFDQSxPQUFPQSxDQUFDQSxFQUFFQSxDQUFDQSxLQUFLQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQTtJQUNqRlQsT0FBT0EsUUFBUUEsQ0FBSUEsS0FBVUE7UUFDM0JVLElBQUlBLENBQUNBLEdBQUdBLFdBQVdBLENBQUNBLEtBQUtBLENBQUNBLEtBQUtBLENBQUNBLENBQUNBO1FBQ2pDQSxNQUFNQSxDQUFDQSxDQUFDQSxDQUFDQSxPQUFPQSxFQUFFQSxDQUFDQTtJQUNyQkEsQ0FBQ0E7SUFDRFYsT0FBT0EsTUFBTUEsQ0FBQ0EsQ0FBUUEsRUFBRUEsQ0FBUUEsSUFBV1csTUFBTUEsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsTUFBTUEsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0E7SUFDaEVYLE9BQU9BLE1BQU1BLENBQUlBLElBQVNBLEVBQUVBLEtBQWFBLEVBQUVBLEtBQVFBLElBQUlZLElBQUlBLENBQUNBLE1BQU1BLENBQUNBLEtBQUtBLEVBQUVBLENBQUNBLEVBQUVBLEtBQUtBLENBQUNBLENBQUNBLENBQUNBLENBQUNBO0lBQ3RGWixPQUFPQSxRQUFRQSxDQUFJQSxJQUFTQSxFQUFFQSxLQUFhQTtRQUN6Q2EsSUFBSUEsR0FBR0EsR0FBR0EsSUFBSUEsQ0FBQ0EsS0FBS0EsQ0FBQ0EsQ0FBQ0E7UUFDdEJBLElBQUlBLENBQUNBLE1BQU1BLENBQUNBLEtBQUtBLEVBQUVBLENBQUNBLENBQUNBLENBQUNBO1FBQ3RCQSxNQUFNQSxDQUFDQSxHQUFHQSxDQUFDQTtJQUNiQSxDQUFDQTtJQUNEYixPQUFPQSxTQUFTQSxDQUFJQSxJQUFTQSxFQUFFQSxLQUFVQTtRQUN2Q2MsR0FBR0EsQ0FBQ0EsQ0FBQ0EsR0FBR0EsQ0FBQ0EsQ0FBQ0EsR0FBR0EsQ0FBQ0EsRUFBRUEsQ0FBQ0EsR0FBR0EsS0FBS0EsQ0FBQ0EsTUFBTUEsRUFBRUEsRUFBRUEsQ0FBQ0EsRUFBRUEsQ0FBQ0E7WUFDdENBLElBQUlBLEtBQUtBLEdBQUdBLElBQUlBLENBQUNBLE9BQU9BLENBQUNBLEtBQUtBLENBQUNBLENBQUNBLENBQUNBLENBQUNBLENBQUNBO1lBQ25DQSxJQUFJQSxDQUFDQSxNQUFNQSxDQUFDQSxLQUFLQSxFQUFFQSxDQUFDQSxDQUFDQSxDQUFDQTtRQUN4QkEsQ0FBQ0E7SUFDSEEsQ0FBQ0E7SUFDRGQsT0FBT0EsTUFBTUEsQ0FBSUEsSUFBU0EsRUFBRUEsRUFBS0E7UUFDL0JlLElBQUlBLEtBQUtBLEdBQUdBLElBQUlBLENBQUNBLE9BQU9BLENBQUNBLEVBQUVBLENBQUNBLENBQUNBO1FBQzdCQSxFQUFFQSxDQUFDQSxDQUFDQSxLQUFLQSxHQUFHQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQTtZQUNmQSxJQUFJQSxDQUFDQSxNQUFNQSxDQUFDQSxLQUFLQSxFQUFFQSxDQUFDQSxDQUFDQSxDQUFDQTtZQUN0QkEsTUFBTUEsQ0FBQ0EsSUFBSUEsQ0FBQ0E7UUFDZEEsQ0FBQ0E7UUFDREEsTUFBTUEsQ0FBQ0EsS0FBS0EsQ0FBQ0E7SUFDZkEsQ0FBQ0E7SUFDRGYsT0FBT0EsS0FBS0EsQ0FBQ0EsSUFBV0EsSUFBSWdCLElBQUlBLENBQUNBLE1BQU1BLEdBQUdBLENBQUNBLENBQUNBLENBQUNBLENBQUNBO0lBQzlDaEIsT0FBT0EsT0FBT0EsQ0FBQ0EsSUFBV0EsSUFBYWlCLE1BQU1BLENBQUNBLElBQUlBLENBQUNBLE1BQU1BLElBQUlBLENBQUNBLENBQUNBLENBQUNBLENBQUNBO0lBQ2pFakIsT0FBT0EsSUFBSUEsQ0FBQ0EsSUFBV0EsRUFBRUEsS0FBVUEsRUFBRUEsS0FBS0EsR0FBV0EsQ0FBQ0EsRUFBRUEsR0FBR0EsR0FBV0EsSUFBSUE7UUFDeEVrQixJQUFJQSxDQUFDQSxJQUFJQSxDQUFDQSxLQUFLQSxFQUFFQSxLQUFLQSxFQUFFQSxHQUFHQSxLQUFLQSxJQUFJQSxHQUFHQSxJQUFJQSxDQUFDQSxNQUFNQSxHQUFHQSxHQUFHQSxDQUFDQSxDQUFDQTtJQUM1REEsQ0FBQ0E7SUFDRGxCLE9BQU9BLE1BQU1BLENBQUNBLENBQVFBLEVBQUVBLENBQVFBO1FBQzlCbUIsRUFBRUEsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsTUFBTUEsSUFBSUEsQ0FBQ0EsQ0FBQ0EsTUFBTUEsQ0FBQ0E7WUFBQ0EsTUFBTUEsQ0FBQ0EsS0FBS0EsQ0FBQ0E7UUFDdkNBLEdBQUdBLENBQUNBLENBQUNBLEdBQUdBLENBQUNBLENBQUNBLEdBQUdBLENBQUNBLEVBQUVBLENBQUNBLEdBQUdBLENBQUNBLENBQUNBLE1BQU1BLEVBQUVBLEVBQUVBLENBQUNBLEVBQUVBLENBQUNBO1lBQ2xDQSxFQUFFQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQSxLQUFLQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQTtnQkFBQ0EsTUFBTUEsQ0FBQ0EsS0FBS0EsQ0FBQ0E7UUFDbENBLENBQUNBO1FBQ0RBLE1BQU1BLENBQUNBLElBQUlBLENBQUNBO0lBQ2RBLENBQUNBO0lBQ0RuQixPQUFPQSxLQUFLQSxDQUFJQSxDQUFNQSxFQUFFQSxJQUFJQSxHQUFXQSxDQUFDQSxFQUFFQSxFQUFFQSxHQUFXQSxJQUFJQTtRQUN6RG9CLE1BQU1BLENBQUNBLENBQUNBLENBQUNBLEtBQUtBLENBQUNBLElBQUlBLEVBQUVBLEVBQUVBLEtBQUtBLElBQUlBLEdBQUdBLFNBQVNBLEdBQUdBLEVBQUVBLENBQUNBLENBQUNBO0lBQ3JEQSxDQUFDQTtJQUNEcEIsT0FBT0EsTUFBTUEsQ0FBSUEsQ0FBTUEsRUFBRUEsSUFBWUEsRUFBRUEsTUFBY0EsSUFBU3FCLE1BQU1BLENBQUNBLENBQUNBLENBQUNBLE1BQU1BLENBQUNBLElBQUlBLEVBQUVBLE1BQU1BLENBQUNBLENBQUNBLENBQUNBLENBQUNBO0lBQzlGckIsT0FBT0EsSUFBSUEsQ0FBSUEsQ0FBTUEsRUFBRUEsU0FBa0NBO1FBQ3ZEc0IsRUFBRUEsQ0FBQ0EsQ0FBQ0EsU0FBU0EsQ0FBQ0EsU0FBU0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0E7WUFDekJBLENBQUNBLENBQUNBLElBQUlBLENBQUNBLFNBQVNBLENBQUNBLENBQUNBO1FBQ3BCQSxDQUFDQTtRQUFDQSxJQUFJQSxDQUFDQSxDQUFDQTtZQUNOQSxDQUFDQSxDQUFDQSxJQUFJQSxFQUFFQSxDQUFDQTtRQUNYQSxDQUFDQTtJQUNIQSxDQUFDQTtJQUNEdEIsT0FBT0EsUUFBUUEsQ0FBSUEsQ0FBTUEsSUFBWXVCLE1BQU1BLENBQUNBLENBQUNBLENBQUNBLFFBQVFBLEVBQUVBLENBQUNBLENBQUNBLENBQUNBO0lBQzNEdkIsT0FBT0EsTUFBTUEsQ0FBSUEsQ0FBTUEsSUFBWXdCLE1BQU1BLENBQUNBLElBQUlBLENBQUNBLFNBQVNBLENBQUNBLENBQUNBLENBQUNBLENBQUNBLENBQUNBLENBQUNBO0lBRTlEeEIsT0FBT0EsT0FBT0EsQ0FBSUEsSUFBU0EsRUFBRUEsU0FBMkJBO1FBQ3REeUIsRUFBRUEsQ0FBQ0EsQ0FBQ0EsSUFBSUEsQ0FBQ0EsTUFBTUEsSUFBSUEsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0E7WUFDckJBLE1BQU1BLENBQUNBLElBQUlBLENBQUNBO1FBQ2RBLENBQUNBO1FBQ0RBLElBQUlBLFFBQVFBLEdBQUdBLElBQUlBLENBQUNBO1FBQ3BCQSxJQUFJQSxRQUFRQSxHQUFHQSxDQUFDQSxRQUFRQSxDQUFDQTtRQUN6QkEsR0FBR0EsQ0FBQ0EsQ0FBQ0EsR0FBR0EsQ0FBQ0EsS0FBS0EsR0FBR0EsQ0FBQ0EsRUFBRUEsS0FBS0EsR0FBR0EsSUFBSUEsQ0FBQ0EsTUFBTUEsRUFBRUEsS0FBS0EsRUFBRUEsRUFBRUEsQ0FBQ0E7WUFDakRBLElBQUlBLFNBQVNBLEdBQUdBLElBQUlBLENBQUNBLEtBQUtBLENBQUNBLENBQUNBO1lBQzVCQSxFQUFFQSxDQUFDQSxDQUFDQSxPQUFPQSxDQUFDQSxTQUFTQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQTtnQkFDdkJBLFFBQVFBLENBQUNBO1lBQ1hBLENBQUNBO1lBQ0RBLElBQUlBLGNBQWNBLEdBQUdBLFNBQVNBLENBQUNBLFNBQVNBLENBQUNBLENBQUNBO1lBQzFDQSxFQUFFQSxDQUFDQSxDQUFDQSxjQUFjQSxHQUFHQSxRQUFRQSxDQUFDQSxDQUFDQSxDQUFDQTtnQkFDOUJBLFFBQVFBLEdBQUdBLFNBQVNBLENBQUNBO2dCQUNyQkEsUUFBUUEsR0FBR0EsY0FBY0EsQ0FBQ0E7WUFDNUJBLENBQUNBO1FBQ0hBLENBQUNBO1FBQ0RBLE1BQU1BLENBQUNBLFFBQVFBLENBQUNBO0lBQ2xCQSxDQUFDQTtJQUVEekIsT0FBT0EsV0FBV0EsQ0FBQ0EsSUFBV0EsSUFBYTBCLE1BQU1BLENBQUNBLE1BQU1BLENBQUNBLFFBQVFBLENBQUNBLElBQUlBLENBQUNBLENBQUNBLENBQUNBLENBQUNBO0FBQzVFMUIsQ0FBQ0E7QUFFRCxtQ0FBbUMsR0FBUTtJQUN6QzJCLEVBQUVBLENBQUNBLENBQUNBLENBQUNBLFVBQVVBLENBQUNBLEdBQUdBLENBQUNBLENBQUNBO1FBQUNBLE1BQU1BLENBQUNBLEtBQUtBLENBQUNBO0lBQ25DQSxNQUFNQSxDQUFDQSxPQUFPQSxDQUFDQSxHQUFHQSxDQUFDQTtRQUNaQSxDQUFDQSxDQUFDQSxDQUFDQSxHQUFHQSxZQUFZQSxHQUFHQSxDQUFDQTtZQUNyQkEsaUJBQWlCQSxFQUFFQSxJQUFJQSxHQUFHQSxDQUFDQSxDQUFDQSxDQUFFQSwwQ0FBMENBO0FBQ2xGQSxDQUFDQTtBQUVELGtDQUFrQyxDQUFNLEVBQUUsQ0FBTSxFQUFFLFVBQW9CO0lBQ3BFQyxJQUFJQSxTQUFTQSxHQUFHQSxDQUFDQSxDQUFDQSxpQkFBaUJBLEVBQUVBLENBQUNBLEVBQUVBLENBQUNBO0lBQ3pDQSxJQUFJQSxTQUFTQSxHQUFHQSxDQUFDQSxDQUFDQSxpQkFBaUJBLEVBQUVBLENBQUNBLEVBQUVBLENBQUNBO0lBRXpDQSxPQUFPQSxJQUFJQSxFQUFFQSxDQUFDQTtRQUNaQSxJQUFJQSxLQUFLQSxHQUFHQSxTQUFTQSxDQUFDQSxJQUFJQSxFQUFFQSxDQUFDQTtRQUM3QkEsSUFBSUEsS0FBS0EsR0FBR0EsU0FBU0EsQ0FBQ0EsSUFBSUEsRUFBRUEsQ0FBQ0E7UUFDN0JBLEVBQUVBLENBQUNBLENBQUNBLEtBQUtBLENBQUNBLElBQUlBLElBQUlBLEtBQUtBLENBQUNBLElBQUlBLENBQUNBO1lBQUNBLE1BQU1BLENBQUNBLElBQUlBLENBQUNBO1FBQzFDQSxFQUFFQSxDQUFDQSxDQUFDQSxLQUFLQSxDQUFDQSxJQUFJQSxJQUFJQSxLQUFLQSxDQUFDQSxJQUFJQSxDQUFDQTtZQUFDQSxNQUFNQSxDQUFDQSxLQUFLQSxDQUFDQTtRQUMzQ0EsRUFBRUEsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsVUFBVUEsQ0FBQ0EsS0FBS0EsQ0FBQ0EsS0FBS0EsRUFBRUEsS0FBS0EsQ0FBQ0EsS0FBS0EsQ0FBQ0EsQ0FBQ0E7WUFBQ0EsTUFBTUEsQ0FBQ0EsS0FBS0EsQ0FBQ0E7SUFDMURBLENBQUNBO0FBQ0hBLENBQUNBO0FBRUQsZ0NBQWdDLEdBQVEsRUFBRSxFQUFZO0lBQ3BEQyxFQUFFQSxDQUFDQSxDQUFDQSxPQUFPQSxDQUFDQSxHQUFHQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQTtRQUNqQkEsR0FBR0EsQ0FBQ0EsQ0FBQ0EsR0FBR0EsQ0FBQ0EsQ0FBQ0EsR0FBR0EsQ0FBQ0EsRUFBRUEsQ0FBQ0EsR0FBR0EsR0FBR0EsQ0FBQ0EsTUFBTUEsRUFBRUEsQ0FBQ0EsRUFBRUEsRUFBRUEsQ0FBQ0E7WUFDcENBLEVBQUVBLENBQUNBLEdBQUdBLENBQUNBLENBQUNBLENBQUNBLENBQUNBLENBQUNBO1FBQ2JBLENBQUNBO0lBQ0hBLENBQUNBO0lBQUNBLElBQUlBLENBQUNBLENBQUNBO1FBQ05BLElBQUlBLFFBQVFBLEdBQUdBLEdBQUdBLENBQUNBLGlCQUFpQkEsRUFBRUEsQ0FBQ0EsRUFBRUEsQ0FBQ0E7UUFDMUNBLElBQUlBLElBQUlBLENBQUNBO1FBQ1RBLE9BQU9BLENBQUNBLENBQUNBLENBQUNBLElBQUlBLEdBQUdBLFFBQVFBLENBQUNBLElBQUlBLEVBQUVBLENBQUNBLENBQUNBLElBQUlBLENBQUNBLEVBQUVBLENBQUNBO1lBQ3hDQSxFQUFFQSxDQUFDQSxJQUFJQSxDQUFDQSxLQUFLQSxDQUFDQSxDQUFDQTtRQUNqQkEsQ0FBQ0E7SUFDSEEsQ0FBQ0E7QUFDSEEsQ0FBQ0E7QUFFRCw0RUFBNEU7QUFDNUUsc0VBQXNFO0FBQ3RFLElBQUksaUJBQWlCLEdBQTZCLENBQUM7SUFDakQsSUFBSSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDOUIsRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3BCLE1BQU0sQ0FBQywyQkFBMkIsR0FBVSxJQUFjQyxNQUFNQSxDQUFDQSxJQUFJQSxHQUFHQSxDQUFDQSxHQUFHQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDO0lBQ25GLENBQUM7SUFBQyxJQUFJLENBQUMsQ0FBQztRQUNOLE1BQU0sQ0FBQyxzQ0FBc0MsR0FBVTtZQUNyREMsSUFBSUEsR0FBR0EsR0FBR0EsSUFBSUEsR0FBR0EsQ0FBQ0EsR0FBR0EsQ0FBQ0EsQ0FBQ0E7WUFDdkJBLEVBQUVBLENBQUNBLENBQUNBLEdBQUdBLENBQUNBLElBQUlBLEtBQUtBLEdBQUdBLENBQUNBLE1BQU1BLENBQUNBLENBQUNBLENBQUNBO2dCQUM1QkEsR0FBR0EsQ0FBQ0EsQ0FBQ0EsR0FBR0EsQ0FBQ0EsQ0FBQ0EsR0FBR0EsQ0FBQ0EsRUFBRUEsQ0FBQ0EsR0FBR0EsR0FBR0EsQ0FBQ0EsTUFBTUEsRUFBRUEsQ0FBQ0EsRUFBRUEsRUFBRUEsQ0FBQ0E7b0JBQ3BDQSxHQUFHQSxDQUFDQSxHQUFHQSxDQUFDQSxHQUFHQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQTtnQkFDbEJBLENBQUNBO1lBQ0hBLENBQUNBO1lBQ0RBLE1BQU1BLENBQUNBLEdBQUdBLENBQUNBO1FBQ2JBLENBQUNBLENBQUM7SUFDSixDQUFDO0FBQ0gsQ0FBQyxDQUFDLEVBQUUsQ0FBQztBQUNMO0lBQ0VDLE9BQU9BLGNBQWNBLENBQUlBLEdBQVFBLElBQVlDLE1BQU1BLENBQUNBLGlCQUFpQkEsQ0FBQ0EsR0FBR0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0EsQ0FBQ0E7SUFDN0VELE9BQU9BLEdBQUdBLENBQUlBLENBQVNBLEVBQUVBLEdBQU1BLElBQWFFLE1BQU1BLENBQUNBLENBQUNBLENBQUNBLEdBQUdBLENBQUNBLEdBQUdBLENBQUNBLENBQUNBLENBQUNBLENBQUNBO0lBQ2hFRixPQUFPQSxNQUFNQSxDQUFJQSxDQUFTQSxFQUFFQSxDQUFJQSxJQUFJRyxDQUFDQSxDQUFDQSxNQUFNQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQSxDQUFDQTtBQUNwREgsQ0FBQ0E7QUFBQSIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7XG4gIGlzSnNPYmplY3QsXG4gIGdsb2JhbCxcbiAgaXNQcmVzZW50LFxuICBpc0JsYW5rLFxuICBpc0FycmF5LFxuICBnZXRTeW1ib2xJdGVyYXRvclxufSBmcm9tICdhbmd1bGFyMi9zcmMvZmFjYWRlL2xhbmcnO1xuXG5leHBvcnQgdmFyIE1hcCA9IGdsb2JhbC5NYXA7XG5leHBvcnQgdmFyIFNldCA9IGdsb2JhbC5TZXQ7XG5cbi8vIFNhZmFyaSBhbmQgSW50ZXJuZXQgRXhwbG9yZXIgZG8gbm90IHN1cHBvcnQgdGhlIGl0ZXJhYmxlIHBhcmFtZXRlciB0byB0aGVcbi8vIE1hcCBjb25zdHJ1Y3Rvci4gIFdlIHdvcmsgYXJvdW5kIHRoYXQgYnkgbWFudWFsbHkgYWRkaW5nIHRoZSBpdGVtcy5cbnZhciBjcmVhdGVNYXBGcm9tUGFpcnM6IHsocGFpcnM6IGFueVtdKTogTWFwPGFueSwgYW55Pn0gPSAoZnVuY3Rpb24oKSB7XG4gIHRyeSB7XG4gICAgaWYgKG5ldyBNYXAoPGFueT5bWzEsIDJdXSkuc2l6ZSA9PT0gMSkge1xuICAgICAgcmV0dXJuIGZ1bmN0aW9uIGNyZWF0ZU1hcEZyb21QYWlycyhwYWlyczogYW55W10pOiBNYXA8YW55LCBhbnk+IHsgcmV0dXJuIG5ldyBNYXAocGFpcnMpOyB9O1xuICAgIH1cbiAgfSBjYXRjaCAoZSkge1xuICB9XG4gIHJldHVybiBmdW5jdGlvbiBjcmVhdGVNYXBBbmRQb3B1bGF0ZUZyb21QYWlycyhwYWlyczogYW55W10pOiBNYXA8YW55LCBhbnk+IHtcbiAgICB2YXIgbWFwID0gbmV3IE1hcCgpO1xuICAgIGZvciAodmFyIGkgPSAwOyBpIDwgcGFpcnMubGVuZ3RoOyBpKyspIHtcbiAgICAgIHZhciBwYWlyID0gcGFpcnNbaV07XG4gICAgICBtYXAuc2V0KHBhaXJbMF0sIHBhaXJbMV0pO1xuICAgIH1cbiAgICByZXR1cm4gbWFwO1xuICB9O1xufSkoKTtcbnZhciBjcmVhdGVNYXBGcm9tTWFwOiB7KG06IE1hcDxhbnksIGFueT4pOiBNYXA8YW55LCBhbnk+fSA9IChmdW5jdGlvbigpIHtcbiAgdHJ5IHtcbiAgICBpZiAobmV3IE1hcCg8YW55Pm5ldyBNYXAoKSkpIHtcbiAgICAgIHJldHVybiBmdW5jdGlvbiBjcmVhdGVNYXBGcm9tTWFwKG06IE1hcDxhbnksIGFueT4pOiBNYXA8YW55LCBhbnk+IHsgcmV0dXJuIG5ldyBNYXAoPGFueT5tKTsgfTtcbiAgICB9XG4gIH0gY2F0Y2ggKGUpIHtcbiAgfVxuICByZXR1cm4gZnVuY3Rpb24gY3JlYXRlTWFwQW5kUG9wdWxhdGVGcm9tTWFwKG06IE1hcDxhbnksIGFueT4pOiBNYXA8YW55LCBhbnk+IHtcbiAgICB2YXIgbWFwID0gbmV3IE1hcCgpO1xuICAgIG0uZm9yRWFjaCgodiwgaykgPT4geyBtYXAuc2V0KGssIHYpOyB9KTtcbiAgICByZXR1cm4gbWFwO1xuICB9O1xufSkoKTtcbnZhciBfY2xlYXJWYWx1ZXM6IHsobTogTWFwPGFueSwgYW55Pil9ID0gKGZ1bmN0aW9uKCkge1xuICBpZiAoKDxhbnk+KG5ldyBNYXAoKSkua2V5cygpKS5uZXh0KSB7XG4gICAgcmV0dXJuIGZ1bmN0aW9uIF9jbGVhclZhbHVlcyhtOiBNYXA8YW55LCBhbnk+KSB7XG4gICAgICB2YXIga2V5SXRlcmF0b3IgPSBtLmtleXMoKTtcbiAgICAgIHZhciBrO1xuICAgICAgd2hpbGUgKCEoKGsgPSAoPGFueT5rZXlJdGVyYXRvcikubmV4dCgpKS5kb25lKSkge1xuICAgICAgICBtLnNldChrLnZhbHVlLCBudWxsKTtcbiAgICAgIH1cbiAgICB9O1xuICB9IGVsc2Uge1xuICAgIHJldHVybiBmdW5jdGlvbiBfY2xlYXJWYWx1ZXNXaXRoRm9yZUVhY2gobTogTWFwPGFueSwgYW55Pikge1xuICAgICAgbS5mb3JFYWNoKCh2LCBrKSA9PiB7IG0uc2V0KGssIG51bGwpOyB9KTtcbiAgICB9O1xuICB9XG59KSgpO1xuLy8gU2FmYXJpIGRvZXNuJ3QgaW1wbGVtZW50IE1hcEl0ZXJhdG9yLm5leHQoKSwgd2hpY2ggaXMgdXNlZCBpcyBUcmFjZXVyJ3MgcG9seWZpbGwgb2YgQXJyYXkuZnJvbVxuLy8gVE9ETyhtbGF2YWwpOiByZW1vdmUgdGhlIHdvcmsgYXJvdW5kIG9uY2Ugd2UgaGF2ZSBhIHdvcmtpbmcgcG9seWZpbGwgb2YgQXJyYXkuZnJvbVxudmFyIF9hcnJheUZyb21NYXA6IHsobTogTWFwPGFueSwgYW55PiwgZ2V0VmFsdWVzOiBib29sZWFuKTogYW55W119ID0gKGZ1bmN0aW9uKCkge1xuICB0cnkge1xuICAgIGlmICgoPGFueT4obmV3IE1hcCgpKS52YWx1ZXMoKSkubmV4dCkge1xuICAgICAgcmV0dXJuIGZ1bmN0aW9uIGNyZWF0ZUFycmF5RnJvbU1hcChtOiBNYXA8YW55LCBhbnk+LCBnZXRWYWx1ZXM6IGJvb2xlYW4pOiBhbnlbXSB7XG4gICAgICAgIHJldHVybiBnZXRWYWx1ZXMgPyAoPGFueT5BcnJheSkuZnJvbShtLnZhbHVlcygpKSA6ICg8YW55PkFycmF5KS5mcm9tKG0ua2V5cygpKTtcbiAgICAgIH07XG4gICAgfVxuICB9IGNhdGNoIChlKSB7XG4gIH1cbiAgcmV0dXJuIGZ1bmN0aW9uIGNyZWF0ZUFycmF5RnJvbU1hcFdpdGhGb3JlYWNoKG06IE1hcDxhbnksIGFueT4sIGdldFZhbHVlczogYm9vbGVhbik6IGFueVtdIHtcbiAgICB2YXIgcmVzID0gTGlzdFdyYXBwZXIuY3JlYXRlRml4ZWRTaXplKG0uc2l6ZSksIGkgPSAwO1xuICAgIG0uZm9yRWFjaCgodiwgaykgPT4ge1xuICAgICAgcmVzW2ldID0gZ2V0VmFsdWVzID8gdiA6IGs7XG4gICAgICBpKys7XG4gICAgfSk7XG4gICAgcmV0dXJuIHJlcztcbiAgfTtcbn0pKCk7XG5cbmV4cG9ydCBjbGFzcyBNYXBXcmFwcGVyIHtcbiAgc3RhdGljIGNsb25lPEssIFY+KG06IE1hcDxLLCBWPik6IE1hcDxLLCBWPiB7IHJldHVybiBjcmVhdGVNYXBGcm9tTWFwKG0pOyB9XG4gIHN0YXRpYyBjcmVhdGVGcm9tU3RyaW5nTWFwPFQ+KHN0cmluZ01hcDoge1trZXk6IHN0cmluZ106IFR9KTogTWFwPHN0cmluZywgVD4ge1xuICAgIHZhciByZXN1bHQgPSBuZXcgTWFwPHN0cmluZywgVD4oKTtcbiAgICBmb3IgKHZhciBwcm9wIGluIHN0cmluZ01hcCkge1xuICAgICAgcmVzdWx0LnNldChwcm9wLCBzdHJpbmdNYXBbcHJvcF0pO1xuICAgIH1cbiAgICByZXR1cm4gcmVzdWx0O1xuICB9XG4gIHN0YXRpYyB0b1N0cmluZ01hcDxUPihtOiBNYXA8c3RyaW5nLCBUPik6IHtba2V5OiBzdHJpbmddOiBUfSB7XG4gICAgdmFyIHI6IHtba2V5OiBzdHJpbmddOiBUfSA9IHt9O1xuICAgIG0uZm9yRWFjaCgodiwgaykgPT4gcltrXSA9IHYpO1xuICAgIHJldHVybiByO1xuICB9XG4gIHN0YXRpYyBjcmVhdGVGcm9tUGFpcnMocGFpcnM6IGFueVtdKTogTWFwPGFueSwgYW55PiB7IHJldHVybiBjcmVhdGVNYXBGcm9tUGFpcnMocGFpcnMpOyB9XG4gIHN0YXRpYyBjbGVhclZhbHVlcyhtOiBNYXA8YW55LCBhbnk+KSB7IF9jbGVhclZhbHVlcyhtKTsgfVxuICBzdGF0aWMgaXRlcmFibGU8VD4obTogVCk6IFQgeyByZXR1cm4gbTsgfVxuICBzdGF0aWMga2V5czxLPihtOiBNYXA8SywgYW55Pik6IEtbXSB7IHJldHVybiBfYXJyYXlGcm9tTWFwKG0sIGZhbHNlKTsgfVxuICBzdGF0aWMgdmFsdWVzPFY+KG06IE1hcDxhbnksIFY+KTogVltdIHsgcmV0dXJuIF9hcnJheUZyb21NYXAobSwgdHJ1ZSk7IH1cbn1cblxuLyoqXG4gKiBXcmFwcyBKYXZhc2NyaXB0IE9iamVjdHNcbiAqL1xuZXhwb3J0IGNsYXNzIFN0cmluZ01hcFdyYXBwZXIge1xuICBzdGF0aWMgY3JlYXRlKCk6IHtbazogLyphbnkqLyBzdHJpbmddOiBhbnl9IHtcbiAgICAvLyBOb3RlOiBXZSBhcmUgbm90IHVzaW5nIE9iamVjdC5jcmVhdGUobnVsbCkgaGVyZSBkdWUgdG9cbiAgICAvLyBwZXJmb3JtYW5jZSFcbiAgICAvLyBodHRwOi8vanNwZXJmLmNvbS9uZzItb2JqZWN0LWNyZWF0ZS1udWxsXG4gICAgcmV0dXJuIHt9O1xuICB9XG4gIHN0YXRpYyBjb250YWlucyhtYXA6IHtba2V5OiBzdHJpbmddOiBhbnl9LCBrZXk6IHN0cmluZyk6IGJvb2xlYW4ge1xuICAgIHJldHVybiBtYXAuaGFzT3duUHJvcGVydHkoa2V5KTtcbiAgfVxuICBzdGF0aWMgZ2V0PFY+KG1hcDoge1trZXk6IHN0cmluZ106IFZ9LCBrZXk6IHN0cmluZyk6IFYge1xuICAgIHJldHVybiBtYXAuaGFzT3duUHJvcGVydHkoa2V5KSA/IG1hcFtrZXldIDogdW5kZWZpbmVkO1xuICB9XG4gIHN0YXRpYyBzZXQ8Vj4obWFwOiB7W2tleTogc3RyaW5nXTogVn0sIGtleTogc3RyaW5nLCB2YWx1ZTogVikgeyBtYXBba2V5XSA9IHZhbHVlOyB9XG4gIHN0YXRpYyBrZXlzKG1hcDoge1trZXk6IHN0cmluZ106IGFueX0pOiBzdHJpbmdbXSB7IHJldHVybiBPYmplY3Qua2V5cyhtYXApOyB9XG4gIHN0YXRpYyB2YWx1ZXM8VD4obWFwOiB7W2tleTogc3RyaW5nXTogVH0pOiBUW10ge1xuICAgIHJldHVybiBPYmplY3Qua2V5cyhtYXApLnJlZHVjZSgociwgYSkgPT4ge1xuICAgICAgci5wdXNoKG1hcFthXSk7XG4gICAgICByZXR1cm4gcjtcbiAgICB9LCBbXSk7XG4gIH1cbiAgc3RhdGljIGlzRW1wdHkobWFwOiB7W2tleTogc3RyaW5nXTogYW55fSk6IGJvb2xlYW4ge1xuICAgIGZvciAodmFyIHByb3AgaW4gbWFwKSB7XG4gICAgICByZXR1cm4gZmFsc2U7XG4gICAgfVxuICAgIHJldHVybiB0cnVlO1xuICB9XG4gIHN0YXRpYyBkZWxldGUgKG1hcDoge1trZXk6IHN0cmluZ106IGFueX0sIGtleTogc3RyaW5nKSB7IGRlbGV0ZSBtYXBba2V5XTsgfVxuICBzdGF0aWMgZm9yRWFjaDxLLCBWPihtYXA6IHtba2V5OiBzdHJpbmddOiBWfSwgY2FsbGJhY2s6IC8qKFYsIEspID0+IHZvaWQqLyBGdW5jdGlvbikge1xuICAgIGZvciAodmFyIHByb3AgaW4gbWFwKSB7XG4gICAgICBpZiAobWFwLmhhc093blByb3BlcnR5KHByb3ApKSB7XG4gICAgICAgIGNhbGxiYWNrKG1hcFtwcm9wXSwgcHJvcCk7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgc3RhdGljIG1lcmdlPFY+KG0xOiB7W2tleTogc3RyaW5nXTogVn0sIG0yOiB7W2tleTogc3RyaW5nXTogVn0pOiB7W2tleTogc3RyaW5nXTogVn0ge1xuICAgIHZhciBtOiB7W2tleTogc3RyaW5nXTogVn0gPSB7fTtcblxuICAgIGZvciAodmFyIGF0dHIgaW4gbTEpIHtcbiAgICAgIGlmIChtMS5oYXNPd25Qcm9wZXJ0eShhdHRyKSkge1xuICAgICAgICBtW2F0dHJdID0gbTFbYXR0cl07XG4gICAgICB9XG4gICAgfVxuXG4gICAgZm9yICh2YXIgYXR0ciBpbiBtMikge1xuICAgICAgaWYgKG0yLmhhc093blByb3BlcnR5KGF0dHIpKSB7XG4gICAgICAgIG1bYXR0cl0gPSBtMlthdHRyXTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICByZXR1cm4gbTtcbiAgfVxuXG4gIHN0YXRpYyBlcXVhbHM8Vj4obTE6IHtba2V5OiBzdHJpbmddOiBWfSwgbTI6IHtba2V5OiBzdHJpbmddOiBWfSk6IGJvb2xlYW4ge1xuICAgIHZhciBrMSA9IE9iamVjdC5rZXlzKG0xKTtcbiAgICB2YXIgazIgPSBPYmplY3Qua2V5cyhtMik7XG4gICAgaWYgKGsxLmxlbmd0aCAhPSBrMi5sZW5ndGgpIHtcbiAgICAgIHJldHVybiBmYWxzZTtcbiAgICB9XG4gICAgdmFyIGtleTtcbiAgICBmb3IgKHZhciBpID0gMDsgaSA8IGsxLmxlbmd0aDsgaSsrKSB7XG4gICAgICBrZXkgPSBrMVtpXTtcbiAgICAgIGlmIChtMVtrZXldICE9PSBtMltrZXldKSB7XG4gICAgICAgIHJldHVybiBmYWxzZTtcbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIHRydWU7XG4gIH1cbn1cblxuLyoqXG4gKiBBIGJvb2xlYW4tdmFsdWVkIGZ1bmN0aW9uIG92ZXIgYSB2YWx1ZSwgcG9zc2libHkgaW5jbHVkaW5nIGNvbnRleHQgaW5mb3JtYXRpb25cbiAqIHJlZ2FyZGluZyB0aGF0IHZhbHVlJ3MgcG9zaXRpb24gaW4gYW4gYXJyYXkuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUHJlZGljYXRlPFQ+IHsgKHZhbHVlOiBULCBpbmRleD86IG51bWJlciwgYXJyYXk/OiBUW10pOiBib29sZWFuOyB9XG5cbmV4cG9ydCBjbGFzcyBMaXN0V3JhcHBlciB7XG4gIC8vIEpTIGhhcyBubyB3YXkgdG8gZXhwcmVzcyBhIHN0YXRpY2FsbHkgZml4ZWQgc2l6ZSBsaXN0LCBidXQgZGFydCBkb2VzIHNvIHdlXG4gIC8vIGtlZXAgYm90aCBtZXRob2RzLlxuICBzdGF0aWMgY3JlYXRlRml4ZWRTaXplKHNpemU6IG51bWJlcik6IGFueVtdIHsgcmV0dXJuIG5ldyBBcnJheShzaXplKTsgfVxuICBzdGF0aWMgY3JlYXRlR3Jvd2FibGVTaXplKHNpemU6IG51bWJlcik6IGFueVtdIHsgcmV0dXJuIG5ldyBBcnJheShzaXplKTsgfVxuICBzdGF0aWMgY2xvbmU8VD4oYXJyYXk6IFRbXSk6IFRbXSB7IHJldHVybiBhcnJheS5zbGljZSgwKTsgfVxuICBzdGF0aWMgY3JlYXRlSW1tdXRhYmxlPFQ+KGFycmF5OiBUW10pOiBUW10ge1xuICAgIHZhciByZXN1bHQgPSBMaXN0V3JhcHBlci5jbG9uZShhcnJheSk7XG4gICAgT2JqZWN0LnNlYWwocmVzdWx0KTtcbiAgICByZXR1cm4gcmVzdWx0O1xuICB9XG4gIHN0YXRpYyBmb3JFYWNoV2l0aEluZGV4PFQ+KGFycmF5OiBUW10sIGZuOiAodDogVCwgbjogbnVtYmVyKSA9PiB2b2lkKSB7XG4gICAgZm9yICh2YXIgaSA9IDA7IGkgPCBhcnJheS5sZW5ndGg7IGkrKykge1xuICAgICAgZm4oYXJyYXlbaV0sIGkpO1xuICAgIH1cbiAgfVxuICBzdGF0aWMgZmlyc3Q8VD4oYXJyYXk6IFRbXSk6IFQge1xuICAgIGlmICghYXJyYXkpIHJldHVybiBudWxsO1xuICAgIHJldHVybiBhcnJheVswXTtcbiAgfVxuICBzdGF0aWMgbGFzdDxUPihhcnJheTogVFtdKTogVCB7XG4gICAgaWYgKCFhcnJheSB8fCBhcnJheS5sZW5ndGggPT0gMCkgcmV0dXJuIG51bGw7XG4gICAgcmV0dXJuIGFycmF5W2FycmF5Lmxlbmd0aCAtIDFdO1xuICB9XG4gIHN0YXRpYyBpbmRleE9mPFQ+KGFycmF5OiBUW10sIHZhbHVlOiBULCBzdGFydEluZGV4OiBudW1iZXIgPSAwKTogbnVtYmVyIHtcbiAgICByZXR1cm4gYXJyYXkuaW5kZXhPZih2YWx1ZSwgc3RhcnRJbmRleCk7XG4gIH1cbiAgc3RhdGljIGNvbnRhaW5zPFQ+KGxpc3Q6IFRbXSwgZWw6IFQpOiBib29sZWFuIHsgcmV0dXJuIGxpc3QuaW5kZXhPZihlbCkgIT09IC0xOyB9XG4gIHN0YXRpYyByZXZlcnNlZDxUPihhcnJheTogVFtdKTogVFtdIHtcbiAgICB2YXIgYSA9IExpc3RXcmFwcGVyLmNsb25lKGFycmF5KTtcbiAgICByZXR1cm4gYS5yZXZlcnNlKCk7XG4gIH1cbiAgc3RhdGljIGNvbmNhdChhOiBhbnlbXSwgYjogYW55W10pOiBhbnlbXSB7IHJldHVybiBhLmNvbmNhdChiKTsgfVxuICBzdGF0aWMgaW5zZXJ0PFQ+KGxpc3Q6IFRbXSwgaW5kZXg6IG51bWJlciwgdmFsdWU6IFQpIHsgbGlzdC5zcGxpY2UoaW5kZXgsIDAsIHZhbHVlKTsgfVxuICBzdGF0aWMgcmVtb3ZlQXQ8VD4obGlzdDogVFtdLCBpbmRleDogbnVtYmVyKTogVCB7XG4gICAgdmFyIHJlcyA9IGxpc3RbaW5kZXhdO1xuICAgIGxpc3Quc3BsaWNlKGluZGV4LCAxKTtcbiAgICByZXR1cm4gcmVzO1xuICB9XG4gIHN0YXRpYyByZW1vdmVBbGw8VD4obGlzdDogVFtdLCBpdGVtczogVFtdKSB7XG4gICAgZm9yICh2YXIgaSA9IDA7IGkgPCBpdGVtcy5sZW5ndGg7ICsraSkge1xuICAgICAgdmFyIGluZGV4ID0gbGlzdC5pbmRleE9mKGl0ZW1zW2ldKTtcbiAgICAgIGxpc3Quc3BsaWNlKGluZGV4LCAxKTtcbiAgICB9XG4gIH1cbiAgc3RhdGljIHJlbW92ZTxUPihsaXN0OiBUW10sIGVsOiBUKTogYm9vbGVhbiB7XG4gICAgdmFyIGluZGV4ID0gbGlzdC5pbmRleE9mKGVsKTtcbiAgICBpZiAoaW5kZXggPiAtMSkge1xuICAgICAgbGlzdC5zcGxpY2UoaW5kZXgsIDEpO1xuICAgICAgcmV0dXJuIHRydWU7XG4gICAgfVxuICAgIHJldHVybiBmYWxzZTtcbiAgfVxuICBzdGF0aWMgY2xlYXIobGlzdDogYW55W10pIHsgbGlzdC5sZW5ndGggPSAwOyB9XG4gIHN0YXRpYyBpc0VtcHR5KGxpc3Q6IGFueVtdKTogYm9vbGVhbiB7IHJldHVybiBsaXN0Lmxlbmd0aCA9PSAwOyB9XG4gIHN0YXRpYyBmaWxsKGxpc3Q6IGFueVtdLCB2YWx1ZTogYW55LCBzdGFydDogbnVtYmVyID0gMCwgZW5kOiBudW1iZXIgPSBudWxsKSB7XG4gICAgbGlzdC5maWxsKHZhbHVlLCBzdGFydCwgZW5kID09PSBudWxsID8gbGlzdC5sZW5ndGggOiBlbmQpO1xuICB9XG4gIHN0YXRpYyBlcXVhbHMoYTogYW55W10sIGI6IGFueVtdKTogYm9vbGVhbiB7XG4gICAgaWYgKGEubGVuZ3RoICE9IGIubGVuZ3RoKSByZXR1cm4gZmFsc2U7XG4gICAgZm9yICh2YXIgaSA9IDA7IGkgPCBhLmxlbmd0aDsgKytpKSB7XG4gICAgICBpZiAoYVtpXSAhPT0gYltpXSkgcmV0dXJuIGZhbHNlO1xuICAgIH1cbiAgICByZXR1cm4gdHJ1ZTtcbiAgfVxuICBzdGF0aWMgc2xpY2U8VD4obDogVFtdLCBmcm9tOiBudW1iZXIgPSAwLCB0bzogbnVtYmVyID0gbnVsbCk6IFRbXSB7XG4gICAgcmV0dXJuIGwuc2xpY2UoZnJvbSwgdG8gPT09IG51bGwgPyB1bmRlZmluZWQgOiB0byk7XG4gIH1cbiAgc3RhdGljIHNwbGljZTxUPihsOiBUW10sIGZyb206IG51bWJlciwgbGVuZ3RoOiBudW1iZXIpOiBUW10geyByZXR1cm4gbC5zcGxpY2UoZnJvbSwgbGVuZ3RoKTsgfVxuICBzdGF0aWMgc29ydDxUPihsOiBUW10sIGNvbXBhcmVGbj86IChhOiBULCBiOiBUKSA9PiBudW1iZXIpIHtcbiAgICBpZiAoaXNQcmVzZW50KGNvbXBhcmVGbikpIHtcbiAgICAgIGwuc29ydChjb21wYXJlRm4pO1xuICAgIH0gZWxzZSB7XG4gICAgICBsLnNvcnQoKTtcbiAgICB9XG4gIH1cbiAgc3RhdGljIHRvU3RyaW5nPFQ+KGw6IFRbXSk6IHN0cmluZyB7IHJldHVybiBsLnRvU3RyaW5nKCk7IH1cbiAgc3RhdGljIHRvSlNPTjxUPihsOiBUW10pOiBzdHJpbmcgeyByZXR1cm4gSlNPTi5zdHJpbmdpZnkobCk7IH1cblxuICBzdGF0aWMgbWF4aW11bTxUPihsaXN0OiBUW10sIHByZWRpY2F0ZTogKHQ6IFQpID0+IG51bWJlcik6IFQge1xuICAgIGlmIChsaXN0Lmxlbmd0aCA9PSAwKSB7XG4gICAgICByZXR1cm4gbnVsbDtcbiAgICB9XG4gICAgdmFyIHNvbHV0aW9uID0gbnVsbDtcbiAgICB2YXIgbWF4VmFsdWUgPSAtSW5maW5pdHk7XG4gICAgZm9yICh2YXIgaW5kZXggPSAwOyBpbmRleCA8IGxpc3QubGVuZ3RoOyBpbmRleCsrKSB7XG4gICAgICB2YXIgY2FuZGlkYXRlID0gbGlzdFtpbmRleF07XG4gICAgICBpZiAoaXNCbGFuayhjYW5kaWRhdGUpKSB7XG4gICAgICAgIGNvbnRpbnVlO1xuICAgICAgfVxuICAgICAgdmFyIGNhbmRpZGF0ZVZhbHVlID0gcHJlZGljYXRlKGNhbmRpZGF0ZSk7XG4gICAgICBpZiAoY2FuZGlkYXRlVmFsdWUgPiBtYXhWYWx1ZSkge1xuICAgICAgICBzb2x1dGlvbiA9IGNhbmRpZGF0ZTtcbiAgICAgICAgbWF4VmFsdWUgPSBjYW5kaWRhdGVWYWx1ZTtcbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIHNvbHV0aW9uO1xuICB9XG5cbiAgc3RhdGljIGlzSW1tdXRhYmxlKGxpc3Q6IGFueVtdKTogYm9vbGVhbiB7IHJldHVybiBPYmplY3QuaXNTZWFsZWQobGlzdCk7IH1cbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGlzTGlzdExpa2VJdGVyYWJsZShvYmo6IGFueSk6IGJvb2xlYW4ge1xuICBpZiAoIWlzSnNPYmplY3Qob2JqKSkgcmV0dXJuIGZhbHNlO1xuICByZXR1cm4gaXNBcnJheShvYmopIHx8XG4gICAgICAgICAoIShvYmogaW5zdGFuY2VvZiBNYXApICYmICAgICAgLy8gSlMgTWFwIGFyZSBpdGVyYWJsZXMgYnV0IHJldHVybiBlbnRyaWVzIGFzIFtrLCB2XVxuICAgICAgICAgIGdldFN5bWJvbEl0ZXJhdG9yKCkgaW4gb2JqKTsgIC8vIEpTIEl0ZXJhYmxlIGhhdmUgYSBTeW1ib2wuaXRlcmF0b3IgcHJvcFxufVxuXG5leHBvcnQgZnVuY3Rpb24gYXJlSXRlcmFibGVzRXF1YWwoYTogYW55LCBiOiBhbnksIGNvbXBhcmF0b3I6IEZ1bmN0aW9uKTogYm9vbGVhbiB7XG4gIHZhciBpdGVyYXRvcjEgPSBhW2dldFN5bWJvbEl0ZXJhdG9yKCldKCk7XG4gIHZhciBpdGVyYXRvcjIgPSBiW2dldFN5bWJvbEl0ZXJhdG9yKCldKCk7XG5cbiAgd2hpbGUgKHRydWUpIHtcbiAgICBsZXQgaXRlbTEgPSBpdGVyYXRvcjEubmV4dCgpO1xuICAgIGxldCBpdGVtMiA9IGl0ZXJhdG9yMi5uZXh0KCk7XG4gICAgaWYgKGl0ZW0xLmRvbmUgJiYgaXRlbTIuZG9uZSkgcmV0dXJuIHRydWU7XG4gICAgaWYgKGl0ZW0xLmRvbmUgfHwgaXRlbTIuZG9uZSkgcmV0dXJuIGZhbHNlO1xuICAgIGlmICghY29tcGFyYXRvcihpdGVtMS52YWx1ZSwgaXRlbTIudmFsdWUpKSByZXR1cm4gZmFsc2U7XG4gIH1cbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGl0ZXJhdGVMaXN0TGlrZShvYmo6IGFueSwgZm46IEZ1bmN0aW9uKSB7XG4gIGlmIChpc0FycmF5KG9iaikpIHtcbiAgICBmb3IgKHZhciBpID0gMDsgaSA8IG9iai5sZW5ndGg7IGkrKykge1xuICAgICAgZm4ob2JqW2ldKTtcbiAgICB9XG4gIH0gZWxzZSB7XG4gICAgdmFyIGl0ZXJhdG9yID0gb2JqW2dldFN5bWJvbEl0ZXJhdG9yKCldKCk7XG4gICAgdmFyIGl0ZW07XG4gICAgd2hpbGUgKCEoKGl0ZW0gPSBpdGVyYXRvci5uZXh0KCkpLmRvbmUpKSB7XG4gICAgICBmbihpdGVtLnZhbHVlKTtcbiAgICB9XG4gIH1cbn1cblxuLy8gU2FmYXJpIGFuZCBJbnRlcm5ldCBFeHBsb3JlciBkbyBub3Qgc3VwcG9ydCB0aGUgaXRlcmFibGUgcGFyYW1ldGVyIHRvIHRoZVxuLy8gU2V0IGNvbnN0cnVjdG9yLiAgV2Ugd29yayBhcm91bmQgdGhhdCBieSBtYW51YWxseSBhZGRpbmcgdGhlIGl0ZW1zLlxudmFyIGNyZWF0ZVNldEZyb21MaXN0OiB7KGxzdDogYW55W10pOiBTZXQ8YW55Pn0gPSAoZnVuY3Rpb24oKSB7XG4gIHZhciB0ZXN0ID0gbmV3IFNldChbMSwgMiwgM10pO1xuICBpZiAodGVzdC5zaXplID09PSAzKSB7XG4gICAgcmV0dXJuIGZ1bmN0aW9uIGNyZWF0ZVNldEZyb21MaXN0KGxzdDogYW55W10pOiBTZXQ8YW55PiB7IHJldHVybiBuZXcgU2V0KGxzdCk7IH07XG4gIH0gZWxzZSB7XG4gICAgcmV0dXJuIGZ1bmN0aW9uIGNyZWF0ZVNldEFuZFBvcHVsYXRlRnJvbUxpc3QobHN0OiBhbnlbXSk6IFNldDxhbnk+IHtcbiAgICAgIHZhciByZXMgPSBuZXcgU2V0KGxzdCk7XG4gICAgICBpZiAocmVzLnNpemUgIT09IGxzdC5sZW5ndGgpIHtcbiAgICAgICAgZm9yICh2YXIgaSA9IDA7IGkgPCBsc3QubGVuZ3RoOyBpKyspIHtcbiAgICAgICAgICByZXMuYWRkKGxzdFtpXSk7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICAgIHJldHVybiByZXM7XG4gICAgfTtcbiAgfVxufSkoKTtcbmV4cG9ydCBjbGFzcyBTZXRXcmFwcGVyIHtcbiAgc3RhdGljIGNyZWF0ZUZyb21MaXN0PFQ+KGxzdDogVFtdKTogU2V0PFQ+IHsgcmV0dXJuIGNyZWF0ZVNldEZyb21MaXN0KGxzdCk7IH1cbiAgc3RhdGljIGhhczxUPihzOiBTZXQ8VD4sIGtleTogVCk6IGJvb2xlYW4geyByZXR1cm4gcy5oYXMoa2V5KTsgfVxuICBzdGF0aWMgZGVsZXRlPEs+KG06IFNldDxLPiwgazogSykgeyBtLmRlbGV0ZShrKTsgfVxufVxuIl19