/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
var ELAPSED_TIME_MAX_DECIMAL_PLACES = 3;
var ANIMATION_PROP = 'animation';
var ANIMATIONEND_EVENT = 'animationend';
var ONE_SECOND = 1000;
var ElementAnimationStyleHandler = /** @class */ (function () {
    function ElementAnimationStyleHandler(_element, _name, _duration, _delay, _easing, _fillMode, _onDoneFn) {
        var _this = this;
        this._element = _element;
        this._name = _name;
        this._duration = _duration;
        this._delay = _delay;
        this._easing = _easing;
        this._fillMode = _fillMode;
        this._onDoneFn = _onDoneFn;
        this._finished = false;
        this._destroyed = false;
        this._startTime = 0;
        this._position = 0;
        this._eventFn = function (e) { return _this._handleCallback(e); };
    }
    ElementAnimationStyleHandler.prototype.apply = function () {
        applyKeyframeAnimation(this._element, this._duration + "ms " + this._easing + " " + this._delay + "ms 1 normal " + this._fillMode + " " + this._name);
        addRemoveAnimationEvent(this._element, this._eventFn, false);
        this._startTime = Date.now();
    };
    ElementAnimationStyleHandler.prototype.pause = function () { playPauseAnimation(this._element, this._name, 'paused'); };
    ElementAnimationStyleHandler.prototype.resume = function () { playPauseAnimation(this._element, this._name, 'running'); };
    ElementAnimationStyleHandler.prototype.setPosition = function (position) {
        var index = findIndexForAnimation(this._element, this._name);
        this._position = position * this._duration;
        setAnimationStyle(this._element, 'Delay', "-" + this._position + "ms", index);
    };
    ElementAnimationStyleHandler.prototype.getPosition = function () { return this._position; };
    ElementAnimationStyleHandler.prototype._handleCallback = function (event) {
        var timestamp = event._ngTestManualTimestamp || Date.now();
        var elapsedTime = parseFloat(event.elapsedTime.toFixed(ELAPSED_TIME_MAX_DECIMAL_PLACES)) * ONE_SECOND;
        if (event.animationName == this._name &&
            Math.max(timestamp - this._startTime, 0) >= this._delay && elapsedTime >= this._duration) {
            this.finish();
        }
    };
    ElementAnimationStyleHandler.prototype.finish = function () {
        if (this._finished)
            return;
        this._finished = true;
        this._onDoneFn();
        addRemoveAnimationEvent(this._element, this._eventFn, true);
    };
    ElementAnimationStyleHandler.prototype.destroy = function () {
        if (this._destroyed)
            return;
        this._destroyed = true;
        this.finish();
        removeKeyframeAnimation(this._element, this._name);
    };
    return ElementAnimationStyleHandler;
}());
export { ElementAnimationStyleHandler };
function playPauseAnimation(element, name, status) {
    var index = findIndexForAnimation(element, name);
    setAnimationStyle(element, 'PlayState', status, index);
}
function applyKeyframeAnimation(element, value) {
    var anim = getAnimationStyle(element, '').trim();
    var index = 0;
    if (anim.length) {
        index = countChars(anim, ',') + 1;
        value = anim + ", " + value;
    }
    setAnimationStyle(element, '', value);
    return index;
}
function removeKeyframeAnimation(element, name) {
    var anim = getAnimationStyle(element, '');
    var tokens = anim.split(',');
    var index = findMatchingTokenIndex(tokens, name);
    if (index >= 0) {
        tokens.splice(index, 1);
        var newValue = tokens.join(',');
        setAnimationStyle(element, '', newValue);
    }
}
function findIndexForAnimation(element, value) {
    var anim = getAnimationStyle(element, '');
    if (anim.indexOf(',') > 0) {
        var tokens = anim.split(',');
        return findMatchingTokenIndex(tokens, value);
    }
    return findMatchingTokenIndex([anim], value);
}
function findMatchingTokenIndex(tokens, searchToken) {
    for (var i = 0; i < tokens.length; i++) {
        if (tokens[i].indexOf(searchToken) >= 0) {
            return i;
        }
    }
    return -1;
}
function addRemoveAnimationEvent(element, fn, doRemove) {
    doRemove ? element.removeEventListener(ANIMATIONEND_EVENT, fn) :
        element.addEventListener(ANIMATIONEND_EVENT, fn);
}
function setAnimationStyle(element, name, value, index) {
    var prop = ANIMATION_PROP + name;
    if (index != null) {
        var oldValue = element.style[prop];
        if (oldValue.length) {
            var tokens = oldValue.split(',');
            tokens[index] = value;
            value = tokens.join(',');
        }
    }
    element.style[prop] = value;
}
function getAnimationStyle(element, name) {
    return element.style[ANIMATION_PROP + name];
}
function countChars(value, char) {
    var count = 0;
    for (var i = 0; i < value.length; i++) {
        var c = value.charAt(i);
        if (c === char)
            count++;
    }
    return count;
}

//# sourceMappingURL=data:application/json;base64,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