/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
/** Used to generate unique IDs for events. */
let uniqueIds = 0;
/**
 * Creates a browser MouseEvent with the specified options.
 * @docs-private
 */
export function createMouseEvent(type, clientX = 0, clientY = 0, button = 0, modifiers = {}) {
    const event = document.createEvent('MouseEvent');
    const originalPreventDefault = event.preventDefault.bind(event);
    // Note: We cannot determine the position of the mouse event based on the screen
    // because the dimensions and position of the browser window are not available
    // To provide reasonable `screenX` and `screenY` coordinates, we simply use the
    // client coordinates as if the browser is opened in fullscreen.
    const screenX = clientX;
    const screenY = clientY;
    event.initMouseEvent(type, 
    /* canBubble */ true, 
    /* cancelable */ true, 
    /* view */ window, 
    /* detail */ 0, 
    /* screenX */ screenX, 
    /* screenY */ screenY, 
    /* clientX */ clientX, 
    /* clientY */ clientY, 
    /* ctrlKey */ !!modifiers.control, 
    /* altKey */ !!modifiers.alt, 
    /* shiftKey */ !!modifiers.shift, 
    /* metaKey */ !!modifiers.meta, 
    /* button */ button, 
    /* relatedTarget */ null);
    // `initMouseEvent` doesn't allow us to pass these properties into the constructor.
    // Override them to 1, because they're used for fake screen reader event detection.
    defineReadonlyEventProperty(event, 'buttons', 1);
    defineReadonlyEventProperty(event, 'offsetX', 1);
    defineReadonlyEventProperty(event, 'offsetY', 1);
    // IE won't set `defaultPrevented` on synthetic events so we need to do it manually.
    event.preventDefault = function () {
        defineReadonlyEventProperty(event, 'defaultPrevented', true);
        return originalPreventDefault();
    };
    return event;
}
/**
 * Creates a browser `PointerEvent` with the specified options. Pointer events
 * by default will appear as if they are the primary pointer of their type.
 * https://www.w3.org/TR/pointerevents2/#dom-pointerevent-isprimary.
 *
 * For example, if pointer events for a multi-touch interaction are created, the non-primary
 * pointer touches would need to be represented by non-primary pointer events.
 *
 * @docs-private
 */
export function createPointerEvent(type, clientX = 0, clientY = 0, options = { isPrimary: true }) {
    return new PointerEvent(type, Object.assign({ bubbles: true, cancelable: true, view: window, clientX,
        clientY }, options));
}
/**
 * Creates a browser TouchEvent with the specified pointer coordinates.
 * @docs-private
 */
export function createTouchEvent(type, pageX = 0, pageY = 0, clientX = 0, clientY = 0) {
    // In favor of creating events that work for most of the browsers, the event is created
    // as a basic UI Event. The necessary details for the event will be set manually.
    const event = document.createEvent('UIEvent');
    const touchDetails = { pageX, pageY, clientX, clientY, id: uniqueIds++ };
    // TS3.6 removes the initUIEvent method and suggests porting to "new UIEvent()".
    event.initUIEvent(type, true, true, window, 0);
    // Most of the browsers don't have a "initTouchEvent" method that can be used to define
    // the touch details.
    defineReadonlyEventProperty(event, 'touches', [touchDetails]);
    defineReadonlyEventProperty(event, 'targetTouches', [touchDetails]);
    defineReadonlyEventProperty(event, 'changedTouches', [touchDetails]);
    return event;
}
/**
 * Creates a keyboard event with the specified key and modifiers.
 * @docs-private
 */
export function createKeyboardEvent(type, keyCode = 0, key = '', modifiers = {}) {
    const event = document.createEvent('KeyboardEvent');
    const originalPreventDefault = event.preventDefault.bind(event);
    // Firefox does not support `initKeyboardEvent`, but supports `initKeyEvent`.
    // https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/initKeyEvent.
    if (event.initKeyEvent !== undefined) {
        event.initKeyEvent(type, true, true, window, modifiers.control, modifiers.alt, modifiers.shift, modifiers.meta, keyCode);
    }
    else {
        // `initKeyboardEvent` expects to receive modifiers as a whitespace-delimited string
        // See https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/initKeyboardEvent
        let modifiersList = '';
        if (modifiers.control) {
            modifiersList += 'Control ';
        }
        if (modifiers.alt) {
            modifiersList += 'Alt ';
        }
        if (modifiers.shift) {
            modifiersList += 'Shift ';
        }
        if (modifiers.meta) {
            modifiersList += 'Meta ';
        }
        // TS3.6 removed the `initKeyboardEvent` method and suggested porting to
        // `new KeyboardEvent()` constructor. We cannot use that as we support IE11.
        // https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/initKeyboardEvent.
        event.initKeyboardEvent(type, true, /* canBubble */ true, /* cancelable */ window, /* view */ 0, /* char */ key, /* key */ 0, /* location */ modifiersList.trim(), /* modifiersList */ false /* repeat */);
    }
    // Webkit Browsers don't set the keyCode when calling the init function.
    // See related bug https://bugs.webkit.org/show_bug.cgi?id=16735
    defineReadonlyEventProperty(event, 'keyCode', keyCode);
    defineReadonlyEventProperty(event, 'key', key);
    defineReadonlyEventProperty(event, 'ctrlKey', !!modifiers.control);
    defineReadonlyEventProperty(event, 'altKey', !!modifiers.alt);
    defineReadonlyEventProperty(event, 'shiftKey', !!modifiers.shift);
    defineReadonlyEventProperty(event, 'metaKey', !!modifiers.meta);
    // IE won't set `defaultPrevented` on synthetic events so we need to do it manually.
    event.preventDefault = function () {
        defineReadonlyEventProperty(event, 'defaultPrevented', true);
        return originalPreventDefault();
    };
    return event;
}
/**
 * Creates a fake event object with any desired event type.
 * @docs-private
 */
export function createFakeEvent(type, canBubble = false, cancelable = true) {
    const event = document.createEvent('Event');
    event.initEvent(type, canBubble, cancelable);
    return event;
}
/**
 * Defines a readonly property on the given event object. Readonly properties on an event object
 * are always set as configurable as that matches default readonly properties for DOM event objects.
 */
function defineReadonlyEventProperty(event, propertyName, value) {
    Object.defineProperty(event, propertyName, { get: () => value, configurable: true });
}
//# sourceMappingURL=data:application/json;base64,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