/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { SelectionModel } from '@angular/cdk/collections';
/** Base tree control. It has basic toggle/expand/collapse operations on a single data node. */
export class BaseTreeControl {
    constructor() {
        /** A selection model with multi-selection to track expansion status. */
        this.expansionModel = new SelectionModel(true);
    }
    /** Toggles one single data node's expanded/collapsed state. */
    toggle(dataNode) {
        this.expansionModel.toggle(this._trackByValue(dataNode));
    }
    /** Expands one single data node. */
    expand(dataNode) {
        this.expansionModel.select(this._trackByValue(dataNode));
    }
    /** Collapses one single data node. */
    collapse(dataNode) {
        this.expansionModel.deselect(this._trackByValue(dataNode));
    }
    /** Whether a given data node is expanded or not. Returns true if the data node is expanded. */
    isExpanded(dataNode) {
        return this.expansionModel.isSelected(this._trackByValue(dataNode));
    }
    /** Toggles a subtree rooted at `node` recursively. */
    toggleDescendants(dataNode) {
        this.expansionModel.isSelected(this._trackByValue(dataNode)) ?
            this.collapseDescendants(dataNode) :
            this.expandDescendants(dataNode);
    }
    /** Collapse all dataNodes in the tree. */
    collapseAll() {
        this.expansionModel.clear();
    }
    /** Expands a subtree rooted at given data node recursively. */
    expandDescendants(dataNode) {
        let toBeProcessed = [dataNode];
        toBeProcessed.push(...this.getDescendants(dataNode));
        this.expansionModel.select(...toBeProcessed.map(value => this._trackByValue(value)));
    }
    /** Collapses a subtree rooted at given data node recursively. */
    collapseDescendants(dataNode) {
        let toBeProcessed = [dataNode];
        toBeProcessed.push(...this.getDescendants(dataNode));
        this.expansionModel.deselect(...toBeProcessed.map(value => this._trackByValue(value)));
    }
    _trackByValue(value) {
        return this.trackBy ? this.trackBy(value) : value;
    }
}
//# sourceMappingURL=data:application/json;base64,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