"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getStartOffsetOfAttribute = exports.findAttributeOnElementWithAttrs = exports.findAttributeOnElementWithTag = exports.findElementsWithAttribute = void 0;
const parse5_1 = require("parse5");
/**
 * Parses a HTML fragment and traverses all AST nodes in order find elements that
 * include the specified attribute.
 */
function findElementsWithAttribute(html, attributeName) {
    const document = parse5_1.parseFragment(html, { sourceCodeLocationInfo: true });
    const elements = [];
    const visitNodes = (nodes) => {
        nodes.forEach(n => {
            var _a;
            const node = n;
            if (node.childNodes) {
                visitNodes(node.childNodes);
            }
            if ((_a = node.attrs) === null || _a === void 0 ? void 0 : _a.some(attr => attr.name === attributeName.toLowerCase())) {
                elements.push(node);
            }
        });
    };
    visitNodes(document.childNodes);
    return elements;
}
exports.findElementsWithAttribute = findElementsWithAttribute;
/**
 * Finds elements with explicit tag names that also contain the specified attribute. Returns the
 * attribute start offset based on the specified HTML.
 */
function findAttributeOnElementWithTag(html, name, tagNames) {
    return findElementsWithAttribute(html, name)
        .filter(element => tagNames.includes(element.tagName))
        .map(element => getStartOffsetOfAttribute(element, name));
}
exports.findAttributeOnElementWithTag = findAttributeOnElementWithTag;
/**
 * Finds elements that contain the given attribute and contain at least one of the other
 * specified attributes. Returns the primary attribute's start offset based on the specified HTML.
 */
function findAttributeOnElementWithAttrs(html, name, attrs) {
    return findElementsWithAttribute(html, name)
        .filter(element => attrs.some(attr => hasElementAttribute(element, attr)))
        .map(element => getStartOffsetOfAttribute(element, name));
}
exports.findAttributeOnElementWithAttrs = findAttributeOnElementWithAttrs;
/** Shorthand function that checks if the specified element contains the given attribute. */
function hasElementAttribute(element, attributeName) {
    return element.attrs && element.attrs.some(attr => attr.name === attributeName.toLowerCase());
}
/** Gets the start offset of the given attribute from a Parse5 element. */
function getStartOffsetOfAttribute(element, attributeName) {
    return element.sourceCodeLocation.attrs[attributeName.toLowerCase()].startOffset;
}
exports.getStartOffsetOfAttribute = getStartOffsetOfAttribute;
//# sourceMappingURL=data:application/json;base64,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