"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getAllChanges = exports.getChangesForTarget = void 0;
const target_version_1 = require("./target-version");
/**
 * Gets the changes for a given target version from the specified version changes object.
 *
 * For readability and a good overview of breaking changes, the version change data always
 * includes the related Pull Request link. Since this data is not needed when performing the
 * upgrade, this unused data can be removed and the changes data can be flattened into an
 * easy iterable array.
 */
function getChangesForTarget(target, data) {
    if (!data) {
        const version = target_version_1.TargetVersion[target];
        throw new Error(`No data could be found for target version: ${version}`);
    }
    return (data[target] || []).reduce((result, prData) => result.concat(prData.changes), []);
}
exports.getChangesForTarget = getChangesForTarget;
/**
 * Gets all changes from the specified version changes object. This is helpful in case a migration
 * rule does not distinguish data based on the target version, but for readability the
 * upgrade data is separated for each target version.
 */
function getAllChanges(data) {
    return Object.keys(data)
        .map(targetVersion => getChangesForTarget(targetVersion, data))
        .reduce((result, versionData) => result.concat(versionData), []);
}
exports.getAllChanges = getAllChanges;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidmVyc2lvbi1jaGFuZ2VzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vLi4vLi4vLi4vLi4vc3JjL2Nkay9zY2hlbWF0aWNzL3VwZGF0ZS10b29sL3ZlcnNpb24tY2hhbmdlcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7OztHQU1HOzs7QUFFSCxxREFBK0M7QUFhL0M7Ozs7Ozs7R0FPRztBQUNILFNBQWdCLG1CQUFtQixDQUFJLE1BQXFCLEVBQUUsSUFBdUI7SUFDbkYsSUFBSSxDQUFDLElBQUksRUFBRTtRQUNULE1BQU0sT0FBTyxHQUFJLDhCQUF3QyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ2xFLE1BQU0sSUFBSSxLQUFLLENBQUMsOENBQThDLE9BQU8sRUFBRSxDQUFDLENBQUM7S0FDMUU7SUFFRCxPQUFPLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLE1BQU0sRUFBRSxNQUFNLEVBQUUsRUFBRSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxFQUFFLEVBQVMsQ0FBQyxDQUFDO0FBQ25HLENBQUM7QUFQRCxrREFPQztBQUVEOzs7O0dBSUc7QUFDSCxTQUFnQixhQUFhLENBQUksSUFBdUI7SUFDdEQsT0FBTyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQztTQUNuQixHQUFHLENBQUMsYUFBYSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxhQUE4QixFQUFFLElBQUksQ0FBQyxDQUFDO1NBQy9FLE1BQU0sQ0FBQyxDQUFDLE1BQU0sRUFBRSxXQUFXLEVBQUUsRUFBRSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsV0FBVyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUM7QUFDdkUsQ0FBQztBQUpELHNDQUlDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBAbGljZW5zZVxuICogQ29weXJpZ2h0IEdvb2dsZSBMTEMgQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiBVc2Ugb2YgdGhpcyBzb3VyY2UgY29kZSBpcyBnb3Zlcm5lZCBieSBhbiBNSVQtc3R5bGUgbGljZW5zZSB0aGF0IGNhbiBiZVxuICogZm91bmQgaW4gdGhlIExJQ0VOU0UgZmlsZSBhdCBodHRwczovL2FuZ3VsYXIuaW8vbGljZW5zZVxuICovXG5cbmltcG9ydCB7VGFyZ2V0VmVyc2lvbn0gZnJvbSAnLi90YXJnZXQtdmVyc2lvbic7XG5cbmV4cG9ydCB0eXBlIFZlcnNpb25DaGFuZ2VzPFQ+ID0ge1xuICBbdGFyZ2V0IGluIFRhcmdldFZlcnNpb25dPzogUmVhZGFibGVDaGFuZ2U8VD5bXTtcbn07XG5cbmV4cG9ydCB0eXBlIFJlYWRhYmxlQ2hhbmdlPFQ+ID0ge1xuICBwcjogc3RyaW5nOyBjaGFuZ2VzOiBUW11cbn07XG5cbi8qKiBDb25kaXRpb25hbCB0eXBlIHRoYXQgdW53cmFwcyB0aGUgdmFsdWUgb2YgYSB2ZXJzaW9uIGNoYW5nZXMgdHlwZS4gKi9cbmV4cG9ydCB0eXBlIFZhbHVlT2ZDaGFuZ2VzPFQ+ID0gVCBleHRlbmRzIFZlcnNpb25DaGFuZ2VzPGluZmVyIFg+PyBYIDogbnVsbDtcblxuLyoqXG4gKiBHZXRzIHRoZSBjaGFuZ2VzIGZvciBhIGdpdmVuIHRhcmdldCB2ZXJzaW9uIGZyb20gdGhlIHNwZWNpZmllZCB2ZXJzaW9uIGNoYW5nZXMgb2JqZWN0LlxuICpcbiAqIEZvciByZWFkYWJpbGl0eSBhbmQgYSBnb29kIG92ZXJ2aWV3IG9mIGJyZWFraW5nIGNoYW5nZXMsIHRoZSB2ZXJzaW9uIGNoYW5nZSBkYXRhIGFsd2F5c1xuICogaW5jbHVkZXMgdGhlIHJlbGF0ZWQgUHVsbCBSZXF1ZXN0IGxpbmsuIFNpbmNlIHRoaXMgZGF0YSBpcyBub3QgbmVlZGVkIHdoZW4gcGVyZm9ybWluZyB0aGVcbiAqIHVwZ3JhZGUsIHRoaXMgdW51c2VkIGRhdGEgY2FuIGJlIHJlbW92ZWQgYW5kIHRoZSBjaGFuZ2VzIGRhdGEgY2FuIGJlIGZsYXR0ZW5lZCBpbnRvIGFuXG4gKiBlYXN5IGl0ZXJhYmxlIGFycmF5LlxuICovXG5leHBvcnQgZnVuY3Rpb24gZ2V0Q2hhbmdlc0ZvclRhcmdldDxUPih0YXJnZXQ6IFRhcmdldFZlcnNpb24sIGRhdGE6IFZlcnNpb25DaGFuZ2VzPFQ+KTogVFtdIHtcbiAgaWYgKCFkYXRhKSB7XG4gICAgY29uc3QgdmVyc2lvbiA9IChUYXJnZXRWZXJzaW9uIGFzIFJlY29yZDxzdHJpbmcsIHN0cmluZz4pW3RhcmdldF07XG4gICAgdGhyb3cgbmV3IEVycm9yKGBObyBkYXRhIGNvdWxkIGJlIGZvdW5kIGZvciB0YXJnZXQgdmVyc2lvbjogJHt2ZXJzaW9ufWApO1xuICB9XG5cbiAgcmV0dXJuIChkYXRhW3RhcmdldF0gfHwgW10pLnJlZHVjZSgocmVzdWx0LCBwckRhdGEpID0+IHJlc3VsdC5jb25jYXQocHJEYXRhLmNoYW5nZXMpLCBbXSBhcyBUW10pO1xufVxuXG4vKipcbiAqIEdldHMgYWxsIGNoYW5nZXMgZnJvbSB0aGUgc3BlY2lmaWVkIHZlcnNpb24gY2hhbmdlcyBvYmplY3QuIFRoaXMgaXMgaGVscGZ1bCBpbiBjYXNlIGEgbWlncmF0aW9uXG4gKiBydWxlIGRvZXMgbm90IGRpc3Rpbmd1aXNoIGRhdGEgYmFzZWQgb24gdGhlIHRhcmdldCB2ZXJzaW9uLCBidXQgZm9yIHJlYWRhYmlsaXR5IHRoZVxuICogdXBncmFkZSBkYXRhIGlzIHNlcGFyYXRlZCBmb3IgZWFjaCB0YXJnZXQgdmVyc2lvbi5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGdldEFsbENoYW5nZXM8VD4oZGF0YTogVmVyc2lvbkNoYW5nZXM8VD4pOiBUW10ge1xuICByZXR1cm4gT2JqZWN0LmtleXMoZGF0YSlcbiAgICAgIC5tYXAodGFyZ2V0VmVyc2lvbiA9PiBnZXRDaGFuZ2VzRm9yVGFyZ2V0KHRhcmdldFZlcnNpb24gYXMgVGFyZ2V0VmVyc2lvbiwgZGF0YSkpXG4gICAgICAucmVkdWNlKChyZXN1bHQsIHZlcnNpb25EYXRhKSA9PiByZXN1bHQuY29uY2F0KHZlcnNpb25EYXRhKSwgW10pO1xufVxuIl19