/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
/** Used to generate unique IDs for events. */
let uniqueIds = 0;
/**
 * Creates a browser MouseEvent with the specified options.
 * @docs-private
 */
export function createMouseEvent(type, clientX = 0, clientY = 0, button = 0, modifiers = {}) {
    // Note: We cannot determine the position of the mouse event based on the screen
    // because the dimensions and position of the browser window are not available
    // To provide reasonable `screenX` and `screenY` coordinates, we simply use the
    // client coordinates as if the browser is opened in fullscreen.
    const screenX = clientX;
    const screenY = clientY;
    const event = new MouseEvent(type, {
        bubbles: true,
        cancelable: true,
        view: window,
        detail: 0,
        relatedTarget: null,
        screenX,
        screenY,
        clientX,
        clientY,
        ctrlKey: modifiers.control,
        altKey: modifiers.alt,
        shiftKey: modifiers.shift,
        metaKey: modifiers.meta,
        button: button,
        buttons: 1,
    });
    // The `MouseEvent` constructor doesn't allow us to pass these properties into the constructor.
    // Override them to `1`, because they're used for fake screen reader event detection.
    defineReadonlyEventProperty(event, 'offsetX', 1);
    defineReadonlyEventProperty(event, 'offsetY', 1);
    return event;
}
/**
 * Creates a browser `PointerEvent` with the specified options. Pointer events
 * by default will appear as if they are the primary pointer of their type.
 * https://www.w3.org/TR/pointerevents2/#dom-pointerevent-isprimary.
 *
 * For example, if pointer events for a multi-touch interaction are created, the non-primary
 * pointer touches would need to be represented by non-primary pointer events.
 *
 * @docs-private
 */
export function createPointerEvent(type, clientX = 0, clientY = 0, options = { isPrimary: true }) {
    return new PointerEvent(type, Object.assign({ bubbles: true, cancelable: true, view: window, clientX,
        clientY }, options));
}
/**
 * Creates a browser TouchEvent with the specified pointer coordinates.
 * @docs-private
 */
export function createTouchEvent(type, pageX = 0, pageY = 0, clientX = 0, clientY = 0) {
    // We cannot use the `TouchEvent` or `Touch` because Firefox and Safari lack support.
    // TODO: Switch to the constructor API when it is available for Firefox and Safari.
    const event = document.createEvent('UIEvent');
    const touchDetails = { pageX, pageY, clientX, clientY, identifier: uniqueIds++ };
    // TS3.6 removes the initUIEvent method and suggests porting to "new UIEvent()".
    event.initUIEvent(type, true, true, window, 0);
    // Most of the browsers don't have a "initTouchEvent" method that can be used to define
    // the touch details.
    defineReadonlyEventProperty(event, 'touches', [touchDetails]);
    defineReadonlyEventProperty(event, 'targetTouches', [touchDetails]);
    defineReadonlyEventProperty(event, 'changedTouches', [touchDetails]);
    return event;
}
/**
 * Creates a keyboard event with the specified key and modifiers.
 * @docs-private
 */
export function createKeyboardEvent(type, keyCode = 0, key = '', modifiers = {}) {
    return new KeyboardEvent(type, {
        bubbles: true,
        cancelable: true,
        view: window,
        keyCode: keyCode,
        key: key,
        shiftKey: modifiers.shift,
        metaKey: modifiers.meta,
        altKey: modifiers.alt,
        ctrlKey: modifiers.control,
    });
}
/**
 * Creates a fake event object with any desired event type.
 * @docs-private
 */
export function createFakeEvent(type, bubbles = false, cancelable = true) {
    return new Event(type, { bubbles, cancelable });
}
/**
 * Defines a readonly property on the given event object. Readonly properties on an event object
 * are always set as configurable as that matches default readonly properties for DOM event objects.
 */
function defineReadonlyEventProperty(event, propertyName, value) {
    Object.defineProperty(event, propertyName, { get: () => value, configurable: true });
}
//# sourceMappingURL=data:application/json;base64,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