"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getDefaultComponentOptions = void 0;
const core_1 = require("@angular-devkit/core");
const schema_1 = require("@schematics/angular/component/schema");
/**
 * Returns the default options for the `@schematics/angular:component` schematic which would
 * have been specified at project initialization (ng new or ng init).
 *
 * This is necessary because the Angular CLI only exposes the default values for the "--style",
 * "--inlineStyle", "--skipTests" and "--inlineTemplate" options to the "component" schematic.
 */
function getDefaultComponentOptions(project) {
    // Note: Not all options which are available when running "ng new" will be stored in the
    // workspace config. List of options which will be available in the configuration:
    // angular/angular-cli/blob/master/packages/schematics/angular/application/index.ts#L109-L131
    let skipTests = getDefaultComponentOption(project, ['skipTests'], null);
    // In case "skipTests" is not set explicitly, also look for the "spec" option. The "spec"
    // option has been deprecated but can be still used in older Angular CLI projects.
    // See: https://github.com/angular/angular-cli/commit/a12a4e02a4689b5bdbc6e740c0d9865afb55671a
    if (skipTests === null) {
        skipTests = !getDefaultComponentOption(project, ['spec'], true);
    }
    return {
        style: getDefaultComponentOption(project, ['style', 'styleext'], schema_1.Style.Css),
        inlineStyle: getDefaultComponentOption(project, ['inlineStyle'], false),
        inlineTemplate: getDefaultComponentOption(project, ['inlineTemplate'], false),
        skipTests: skipTests,
    };
}
exports.getDefaultComponentOptions = getDefaultComponentOptions;
/**
 * Gets the default value for the specified option. The default options will be determined
 * by looking at the stored schematic options for `@schematics/angular:component` in the
 * CLI workspace configuration.
 */
function getDefaultComponentOption(project, optionNames, fallbackValue) {
    const schematicOptions = core_1.isJsonObject(project.extensions.schematics || null) ?
        project.extensions.schematics : null;
    const defaultSchematic = schematicOptions ?
        schematicOptions['@schematics/angular:component'] : null;
    for (const optionName of optionNames) {
        if (defaultSchematic && defaultSchematic[optionName] != null) {
            return defaultSchematic[optionName];
        }
    }
    return fallbackValue;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2NoZW1hdGljLW9wdGlvbnMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi8uLi8uLi8uLi9zcmMvY2RrL3NjaGVtYXRpY3MvdXRpbHMvc2NoZW1hdGljLW9wdGlvbnMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Ozs7R0FNRzs7O0FBR0gsK0NBQThEO0FBQzlELGlFQUFtRTtBQUVuRTs7Ozs7O0dBTUc7QUFDSCxTQUFnQiwwQkFBMEIsQ0FBQyxPQUEwQjtJQUNuRSx3RkFBd0Y7SUFDeEYsa0ZBQWtGO0lBQ2xGLDZGQUE2RjtJQUM3RixJQUFJLFNBQVMsR0FBRyx5QkFBeUIsQ0FBZSxPQUFPLEVBQUUsQ0FBQyxXQUFXLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztJQUV0Rix5RkFBeUY7SUFDekYsa0ZBQWtGO0lBQ2xGLDhGQUE4RjtJQUM5RixJQUFJLFNBQVMsS0FBSyxJQUFJLEVBQUU7UUFDdEIsU0FBUyxHQUFHLENBQUMseUJBQXlCLENBQUMsT0FBTyxFQUFFLENBQUMsTUFBTSxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUM7S0FDakU7SUFFRCxPQUFPO1FBQ0wsS0FBSyxFQUFFLHlCQUF5QixDQUFRLE9BQU8sRUFBRSxDQUFDLE9BQU8sRUFBRSxVQUFVLENBQUMsRUFBRSxjQUFLLENBQUMsR0FBRyxDQUFDO1FBQ2xGLFdBQVcsRUFBRSx5QkFBeUIsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxhQUFhLENBQUMsRUFBRSxLQUFLLENBQUM7UUFDdkUsY0FBYyxFQUFFLHlCQUF5QixDQUFDLE9BQU8sRUFBRSxDQUFDLGdCQUFnQixDQUFDLEVBQUUsS0FBSyxDQUFDO1FBQzdFLFNBQVMsRUFBRSxTQUFTO0tBQ3JCLENBQUM7QUFDSixDQUFDO0FBbkJELGdFQW1CQztBQUVEOzs7O0dBSUc7QUFDSCxTQUFTLHlCQUF5QixDQUFJLE9BQTBCLEVBQUUsV0FBcUIsRUFDakQsYUFBZ0I7SUFDcEQsTUFBTSxnQkFBZ0IsR0FBRyxtQkFBWSxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsVUFBVSxJQUFJLElBQUksQ0FBQyxDQUFDLENBQUM7UUFDMUUsT0FBTyxDQUFDLFVBQVUsQ0FBQyxVQUF3QixDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUM7SUFDdkQsTUFBTSxnQkFBZ0IsR0FBRyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQ3ZDLGdCQUFnQixDQUFDLCtCQUErQixDQUFzQixDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUM7SUFFbEYsS0FBSyxNQUFNLFVBQVUsSUFBSSxXQUFXLEVBQUU7UUFDcEMsSUFBSSxnQkFBZ0IsSUFBSSxnQkFBZ0IsQ0FBQyxVQUFVLENBQUMsSUFBSSxJQUFJLEVBQUU7WUFDNUQsT0FBTyxnQkFBZ0IsQ0FBQyxVQUFVLENBQWlCLENBQUM7U0FDckQ7S0FDRjtJQUVELE9BQU8sYUFBYSxDQUFDO0FBQ3ZCLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIEBsaWNlbnNlXG4gKiBDb3B5cmlnaHQgR29vZ2xlIExMQyBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqIFVzZSBvZiB0aGlzIHNvdXJjZSBjb2RlIGlzIGdvdmVybmVkIGJ5IGFuIE1JVC1zdHlsZSBsaWNlbnNlIHRoYXQgY2FuIGJlXG4gKiBmb3VuZCBpbiB0aGUgTElDRU5TRSBmaWxlIGF0IGh0dHBzOi8vYW5ndWxhci5pby9saWNlbnNlXG4gKi9cblxuaW1wb3J0IHtQcm9qZWN0RGVmaW5pdGlvbn0gZnJvbSAnQGFuZ3VsYXItZGV2a2l0L2NvcmUvc3JjL3dvcmtzcGFjZSc7XG5pbXBvcnQge2lzSnNvbk9iamVjdCwgSnNvbk9iamVjdH0gZnJvbSAnQGFuZ3VsYXItZGV2a2l0L2NvcmUnO1xuaW1wb3J0IHtTY2hlbWEsIFN0eWxlfSBmcm9tICdAc2NoZW1hdGljcy9hbmd1bGFyL2NvbXBvbmVudC9zY2hlbWEnO1xuXG4vKipcbiAqIFJldHVybnMgdGhlIGRlZmF1bHQgb3B0aW9ucyBmb3IgdGhlIGBAc2NoZW1hdGljcy9hbmd1bGFyOmNvbXBvbmVudGAgc2NoZW1hdGljIHdoaWNoIHdvdWxkXG4gKiBoYXZlIGJlZW4gc3BlY2lmaWVkIGF0IHByb2plY3QgaW5pdGlhbGl6YXRpb24gKG5nIG5ldyBvciBuZyBpbml0KS5cbiAqXG4gKiBUaGlzIGlzIG5lY2Vzc2FyeSBiZWNhdXNlIHRoZSBBbmd1bGFyIENMSSBvbmx5IGV4cG9zZXMgdGhlIGRlZmF1bHQgdmFsdWVzIGZvciB0aGUgXCItLXN0eWxlXCIsXG4gKiBcIi0taW5saW5lU3R5bGVcIiwgXCItLXNraXBUZXN0c1wiIGFuZCBcIi0taW5saW5lVGVtcGxhdGVcIiBvcHRpb25zIHRvIHRoZSBcImNvbXBvbmVudFwiIHNjaGVtYXRpYy5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGdldERlZmF1bHRDb21wb25lbnRPcHRpb25zKHByb2plY3Q6IFByb2plY3REZWZpbml0aW9uKTogUGFydGlhbDxTY2hlbWE+IHtcbiAgLy8gTm90ZTogTm90IGFsbCBvcHRpb25zIHdoaWNoIGFyZSBhdmFpbGFibGUgd2hlbiBydW5uaW5nIFwibmcgbmV3XCIgd2lsbCBiZSBzdG9yZWQgaW4gdGhlXG4gIC8vIHdvcmtzcGFjZSBjb25maWcuIExpc3Qgb2Ygb3B0aW9ucyB3aGljaCB3aWxsIGJlIGF2YWlsYWJsZSBpbiB0aGUgY29uZmlndXJhdGlvbjpcbiAgLy8gYW5ndWxhci9hbmd1bGFyLWNsaS9ibG9iL21hc3Rlci9wYWNrYWdlcy9zY2hlbWF0aWNzL2FuZ3VsYXIvYXBwbGljYXRpb24vaW5kZXgudHMjTDEwOS1MMTMxXG4gIGxldCBza2lwVGVzdHMgPSBnZXREZWZhdWx0Q29tcG9uZW50T3B0aW9uPGJvb2xlYW58bnVsbD4ocHJvamVjdCwgWydza2lwVGVzdHMnXSwgbnVsbCk7XG5cbiAgLy8gSW4gY2FzZSBcInNraXBUZXN0c1wiIGlzIG5vdCBzZXQgZXhwbGljaXRseSwgYWxzbyBsb29rIGZvciB0aGUgXCJzcGVjXCIgb3B0aW9uLiBUaGUgXCJzcGVjXCJcbiAgLy8gb3B0aW9uIGhhcyBiZWVuIGRlcHJlY2F0ZWQgYnV0IGNhbiBiZSBzdGlsbCB1c2VkIGluIG9sZGVyIEFuZ3VsYXIgQ0xJIHByb2plY3RzLlxuICAvLyBTZWU6IGh0dHBzOi8vZ2l0aHViLmNvbS9hbmd1bGFyL2FuZ3VsYXItY2xpL2NvbW1pdC9hMTJhNGUwMmE0Njg5YjViZGJjNmU3NDBjMGQ5ODY1YWZiNTU2NzFhXG4gIGlmIChza2lwVGVzdHMgPT09IG51bGwpIHtcbiAgICBza2lwVGVzdHMgPSAhZ2V0RGVmYXVsdENvbXBvbmVudE9wdGlvbihwcm9qZWN0LCBbJ3NwZWMnXSwgdHJ1ZSk7XG4gIH1cblxuICByZXR1cm4ge1xuICAgIHN0eWxlOiBnZXREZWZhdWx0Q29tcG9uZW50T3B0aW9uPFN0eWxlPihwcm9qZWN0LCBbJ3N0eWxlJywgJ3N0eWxlZXh0J10sIFN0eWxlLkNzcyksXG4gICAgaW5saW5lU3R5bGU6IGdldERlZmF1bHRDb21wb25lbnRPcHRpb24ocHJvamVjdCwgWydpbmxpbmVTdHlsZSddLCBmYWxzZSksXG4gICAgaW5saW5lVGVtcGxhdGU6IGdldERlZmF1bHRDb21wb25lbnRPcHRpb24ocHJvamVjdCwgWydpbmxpbmVUZW1wbGF0ZSddLCBmYWxzZSksXG4gICAgc2tpcFRlc3RzOiBza2lwVGVzdHMsXG4gIH07XG59XG5cbi8qKlxuICogR2V0cyB0aGUgZGVmYXVsdCB2YWx1ZSBmb3IgdGhlIHNwZWNpZmllZCBvcHRpb24uIFRoZSBkZWZhdWx0IG9wdGlvbnMgd2lsbCBiZSBkZXRlcm1pbmVkXG4gKiBieSBsb29raW5nIGF0IHRoZSBzdG9yZWQgc2NoZW1hdGljIG9wdGlvbnMgZm9yIGBAc2NoZW1hdGljcy9hbmd1bGFyOmNvbXBvbmVudGAgaW4gdGhlXG4gKiBDTEkgd29ya3NwYWNlIGNvbmZpZ3VyYXRpb24uXG4gKi9cbmZ1bmN0aW9uIGdldERlZmF1bHRDb21wb25lbnRPcHRpb248VD4ocHJvamVjdDogUHJvamVjdERlZmluaXRpb24sIG9wdGlvbk5hbWVzOiBzdHJpbmdbXSxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgZmFsbGJhY2tWYWx1ZTogVCk6IFQge1xuICBjb25zdCBzY2hlbWF0aWNPcHRpb25zID0gaXNKc29uT2JqZWN0KHByb2plY3QuZXh0ZW5zaW9ucy5zY2hlbWF0aWNzIHx8IG51bGwpID9cbiAgICAgIHByb2plY3QuZXh0ZW5zaW9ucy5zY2hlbWF0aWNzIGFzIEpzb25PYmplY3QgOiBudWxsO1xuICBjb25zdCBkZWZhdWx0U2NoZW1hdGljID0gc2NoZW1hdGljT3B0aW9ucyA/XG4gICAgICBzY2hlbWF0aWNPcHRpb25zWydAc2NoZW1hdGljcy9hbmd1bGFyOmNvbXBvbmVudCddIGFzIEpzb25PYmplY3QgfCBudWxsIDogbnVsbDtcblxuICBmb3IgKGNvbnN0IG9wdGlvbk5hbWUgb2Ygb3B0aW9uTmFtZXMpIHtcbiAgICBpZiAoZGVmYXVsdFNjaGVtYXRpYyAmJiBkZWZhdWx0U2NoZW1hdGljW29wdGlvbk5hbWVdICE9IG51bGwpIHtcbiAgICAgIHJldHVybiBkZWZhdWx0U2NoZW1hdGljW29wdGlvbk5hbWVdIGFzIHVua25vd24gYXMgVDtcbiAgICB9XG4gIH1cblxuICByZXR1cm4gZmFsbGJhY2tWYWx1ZTtcbn1cbiJdfQ==