/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { Directive, ElementRef, EventEmitter, Injectable, Input, NgModule, NgZone, Output } from '@angular/core';
import { Subject } from 'rxjs/Subject';
import { RxChain, debounceTime } from '@angular/cdk/rxjs';

/**
 * Factory that creates a new MutationObserver and allows us to stub it out in unit tests.
 * \@docs-private
 */
var MatMutationObserverFactory = (function () {
    function MatMutationObserverFactory() {
    }
    /**
     * @param {?} callback
     * @return {?}
     */
    MatMutationObserverFactory.prototype.create = function (callback) {
        return typeof MutationObserver === 'undefined' ? null : new MutationObserver(callback);
    };
    MatMutationObserverFactory.decorators = [
        { type: Injectable },
    ];
    /**
     * @nocollapse
     */
    MatMutationObserverFactory.ctorParameters = function () { return []; };
    return MatMutationObserverFactory;
}());
/**
 * Directive that triggers a callback whenever the content of
 * its associated element has changed.
 */
var ObserveContent = (function () {
    /**
     * @param {?} _mutationObserverFactory
     * @param {?} _elementRef
     * @param {?} _ngZone
     */
    function ObserveContent(_mutationObserverFactory, _elementRef, _ngZone) {
        this._mutationObserverFactory = _mutationObserverFactory;
        this._elementRef = _elementRef;
        this._ngZone = _ngZone;
        /**
         * Event emitted for each change in the element's content.
         */
        this.event = new EventEmitter();
        /**
         * Used for debouncing the emitted values to the observeContent event.
         */
        this._debouncer = new Subject();
    }
    /**
     * @return {?}
     */
    ObserveContent.prototype.ngAfterContentInit = function () {
        var _this = this;
        if (this.debounce > 0) {
            this._ngZone.runOutsideAngular(function () {
                RxChain.from(_this._debouncer)
                    .call(debounceTime, _this.debounce)
                    .subscribe(function (mutations) { return _this.event.emit(mutations); });
            });
        }
        else {
            this._debouncer.subscribe(function (mutations) { return _this.event.emit(mutations); });
        }
        this._observer = this._ngZone.runOutsideAngular(function () {
            return _this._mutationObserverFactory.create(function (mutations) {
                _this._debouncer.next(mutations);
            });
        });
        if (this._observer) {
            this._observer.observe(this._elementRef.nativeElement, {
                characterData: true,
                childList: true,
                subtree: true
            });
        }
    };
    /**
     * @return {?}
     */
    ObserveContent.prototype.ngOnDestroy = function () {
        if (this._observer) {
            this._observer.disconnect();
        }
        this._debouncer.complete();
    };
    ObserveContent.decorators = [
        { type: Directive, args: [{
                    selector: '[cdkObserveContent]',
                    exportAs: 'cdkObserveContent',
                },] },
    ];
    /**
     * @nocollapse
     */
    ObserveContent.ctorParameters = function () { return [
        { type: MatMutationObserverFactory, },
        { type: ElementRef, },
        { type: NgZone, },
    ]; };
    ObserveContent.propDecorators = {
        'event': [{ type: Output, args: ['cdkObserveContent',] },],
        'debounce': [{ type: Input },],
    };
    return ObserveContent;
}());
var ObserversModule = (function () {
    function ObserversModule() {
    }
    ObserversModule.decorators = [
        { type: NgModule, args: [{
                    exports: [ObserveContent],
                    declarations: [ObserveContent],
                    providers: [MatMutationObserverFactory]
                },] },
    ];
    /**
     * @nocollapse
     */
    ObserversModule.ctorParameters = function () { return []; };
    return ObserversModule;
}());

/**
 * Generated bundle index. Do not edit.
 */

export { MatMutationObserverFactory, ObserveContent, ObserversModule };
//# sourceMappingURL=observers.es5.js.map
