/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes,extraRequire,uselessCode} checked by tsc
 */
import { NumberFormatStyle } from '../../i18n/locale_data_api';
export class NumberFormatter {
    /**
     * @param {?} num
     * @param {?} locale
     * @param {?} style
     * @param {?=} opts
     * @return {?}
     */
    static format(num, locale, style, opts = {}) {
        const { minimumIntegerDigits, minimumFractionDigits, maximumFractionDigits, currency, currencyAsSymbol = false } = opts;
        /** @type {?} */
        const options = {
            minimumIntegerDigits,
            minimumFractionDigits,
            maximumFractionDigits,
            style: NumberFormatStyle[style].toLowerCase()
        };
        if (style == NumberFormatStyle.Currency) {
            options.currency = typeof currency == 'string' ? currency : undefined;
            options.currencyDisplay = currencyAsSymbol ? 'symbol' : 'code';
        }
        return new Intl.NumberFormat(locale, options).format(num);
    }
}
/** @typedef {?} */
var DateFormatterFn;
/** @type {?} */
const DATE_FORMATS_SPLIT = /((?:[^yMLdHhmsazZEwGjJ']+)|(?:'(?:[^']|'')*')|(?:E+|y+|M+|L+|d+|H+|h+|J+|j+|m+|s+|a|z|Z|G+|w+))(.*)/;
/** @type {?} */
const PATTERN_ALIASES = {
    // Keys are quoted so they do not get renamed during closure compilation.
    'yMMMdjms': datePartGetterFactory(combine([
        digitCondition('year', 1),
        nameCondition('month', 3),
        digitCondition('day', 1),
        digitCondition('hour', 1),
        digitCondition('minute', 1),
        digitCondition('second', 1),
    ])),
    'yMdjm': datePartGetterFactory(combine([
        digitCondition('year', 1), digitCondition('month', 1), digitCondition('day', 1),
        digitCondition('hour', 1), digitCondition('minute', 1)
    ])),
    'yMMMMEEEEd': datePartGetterFactory(combine([
        digitCondition('year', 1), nameCondition('month', 4), nameCondition('weekday', 4),
        digitCondition('day', 1)
    ])),
    'yMMMMd': datePartGetterFactory(combine([digitCondition('year', 1), nameCondition('month', 4), digitCondition('day', 1)])),
    'yMMMd': datePartGetterFactory(combine([digitCondition('year', 1), nameCondition('month', 3), digitCondition('day', 1)])),
    'yMd': datePartGetterFactory(combine([digitCondition('year', 1), digitCondition('month', 1), digitCondition('day', 1)])),
    'jms': datePartGetterFactory(combine([digitCondition('hour', 1), digitCondition('second', 1), digitCondition('minute', 1)])),
    'jm': datePartGetterFactory(combine([digitCondition('hour', 1), digitCondition('minute', 1)]))
};
/** @type {?} */
const DATE_FORMATS = {
    // Keys are quoted so they do not get renamed.
    'yyyy': datePartGetterFactory(digitCondition('year', 4)),
    'yy': datePartGetterFactory(digitCondition('year', 2)),
    'y': datePartGetterFactory(digitCondition('year', 1)),
    'MMMM': datePartGetterFactory(nameCondition('month', 4)),
    'MMM': datePartGetterFactory(nameCondition('month', 3)),
    'MM': datePartGetterFactory(digitCondition('month', 2)),
    'M': datePartGetterFactory(digitCondition('month', 1)),
    'LLLL': datePartGetterFactory(nameCondition('month', 4)),
    'L': datePartGetterFactory(nameCondition('month', 1)),
    'dd': datePartGetterFactory(digitCondition('day', 2)),
    'd': datePartGetterFactory(digitCondition('day', 1)),
    'HH': digitModifier(hourExtractor(datePartGetterFactory(hour12Modify(digitCondition('hour', 2), false)))),
    'H': hourExtractor(datePartGetterFactory(hour12Modify(digitCondition('hour', 1), false))),
    'hh': digitModifier(hourExtractor(datePartGetterFactory(hour12Modify(digitCondition('hour', 2), true)))),
    'h': hourExtractor(datePartGetterFactory(hour12Modify(digitCondition('hour', 1), true))),
    'jj': datePartGetterFactory(digitCondition('hour', 2)),
    'j': datePartGetterFactory(digitCondition('hour', 1)),
    'mm': digitModifier(datePartGetterFactory(digitCondition('minute', 2))),
    'm': datePartGetterFactory(digitCondition('minute', 1)),
    'ss': digitModifier(datePartGetterFactory(digitCondition('second', 2))),
    's': datePartGetterFactory(digitCondition('second', 1)),
    // while ISO 8601 requires fractions to be prefixed with `.` or `,`
    // we can be just safely rely on using `sss` since we currently don't support single or two digit
    // fractions
    'sss': datePartGetterFactory(digitCondition('second', 3)),
    'EEEE': datePartGetterFactory(nameCondition('weekday', 4)),
    'EEE': datePartGetterFactory(nameCondition('weekday', 3)),
    'EE': datePartGetterFactory(nameCondition('weekday', 2)),
    'E': datePartGetterFactory(nameCondition('weekday', 1)),
    'a': hourClockExtractor(datePartGetterFactory(hour12Modify(digitCondition('hour', 1), true))),
    'Z': timeZoneGetter('short'),
    'z': timeZoneGetter('long'),
    'ww': datePartGetterFactory({}),
    // Week of year, padded (00-53). Week 01 is the week with the
    // first Thursday of the year. not support ?
    'w': datePartGetterFactory({}),
    // Week of year (0-53). Week 1 is the week with the first Thursday
    // of the year not support ?
    'G': datePartGetterFactory(nameCondition('era', 1)),
    'GG': datePartGetterFactory(nameCondition('era', 2)),
    'GGG': datePartGetterFactory(nameCondition('era', 3)),
    'GGGG': datePartGetterFactory(nameCondition('era', 4))
};
/**
 * @param {?} inner
 * @return {?}
 */
function digitModifier(inner) {
    return function (date, locale) {
        /** @type {?} */
        const result = inner(date, locale);
        return result.length == 1 ? '0' + result : result;
    };
}
/**
 * @param {?} inner
 * @return {?}
 */
function hourClockExtractor(inner) {
    return function (date, locale) { return inner(date, locale).split(' ')[1]; };
}
/**
 * @param {?} inner
 * @return {?}
 */
function hourExtractor(inner) {
    return function (date, locale) { return inner(date, locale).split(' ')[0]; };
}
/**
 * @param {?} date
 * @param {?} locale
 * @param {?} options
 * @return {?}
 */
function intlDateFormat(date, locale, options) {
    return new Intl.DateTimeFormat(locale, options).format(date).replace(/[\u200e\u200f]/g, '');
}
/**
 * @param {?} timezone
 * @return {?}
 */
function timeZoneGetter(timezone) {
    /** @type {?} */
    const options = { hour: '2-digit', hour12: false, timeZoneName: timezone };
    return function (date, locale) {
        /** @type {?} */
        const result = intlDateFormat(date, locale, options);
        // Then extract first 3 letters that related to hours
        return result ? result.substring(3) : '';
    };
}
/**
 * @param {?} options
 * @param {?} value
 * @return {?}
 */
function hour12Modify(options, value) {
    options.hour12 = value;
    return options;
}
/**
 * @param {?} prop
 * @param {?} len
 * @return {?}
 */
function digitCondition(prop, len) {
    /** @type {?} */
    const result = {};
    result[prop] = len === 2 ? '2-digit' : 'numeric';
    return result;
}
/**
 * @param {?} prop
 * @param {?} len
 * @return {?}
 */
function nameCondition(prop, len) {
    /** @type {?} */
    const result = {};
    if (len < 4) {
        result[prop] = len > 1 ? 'short' : 'narrow';
    }
    else {
        result[prop] = 'long';
    }
    return result;
}
/**
 * @param {?} options
 * @return {?}
 */
function combine(options) {
    return options.reduce((merged, opt) => (Object.assign({}, merged, opt)), {});
}
/**
 * @param {?} ret
 * @return {?}
 */
function datePartGetterFactory(ret) {
    return (date, locale) => intlDateFormat(date, locale, ret);
}
/** @type {?} */
const DATE_FORMATTER_CACHE = new Map();
/**
 * @param {?} format
 * @param {?} date
 * @param {?} locale
 * @return {?}
 */
function dateFormatter(format, date, locale) {
    /** @type {?} */
    const fn = PATTERN_ALIASES[format];
    if (fn)
        return fn(date, locale);
    /** @type {?} */
    const cacheKey = format;
    /** @type {?} */
    let parts = DATE_FORMATTER_CACHE.get(cacheKey);
    if (!parts) {
        parts = [];
        /** @type {?} */
        let match;
        DATE_FORMATS_SPLIT.exec(format);
        /** @type {?} */
        let _format = format;
        while (_format) {
            match = DATE_FORMATS_SPLIT.exec(_format);
            if (match) {
                parts = parts.concat(match.slice(1));
                _format = /** @type {?} */ ((parts.pop()));
            }
            else {
                parts.push(_format);
                _format = null;
            }
        }
        DATE_FORMATTER_CACHE.set(cacheKey, parts);
    }
    return parts.reduce((text, part) => {
        /** @type {?} */
        const fn = DATE_FORMATS[part];
        return text + (fn ? fn(date, locale) : partToTime(part));
    }, '');
}
/**
 * @param {?} part
 * @return {?}
 */
function partToTime(part) {
    return part === '\'\'' ? '\'' : part.replace(/(^'|'$)/g, '').replace(/''/g, '\'');
}
export class DateFormatter {
    /**
     * @param {?} date
     * @param {?} locale
     * @param {?} pattern
     * @return {?}
     */
    static format(date, locale, pattern) {
        return dateFormatter(pattern, date, locale);
    }
}

//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW50bC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uLy4uLy4uLy4uLy4uL3BhY2thZ2VzL2NvbW1vbi9zcmMvcGlwZXMvZGVwcmVjYXRlZC9pbnRsLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7QUFPQSxPQUFPLEVBQUMsaUJBQWlCLEVBQUMsTUFBTSw0QkFBNEIsQ0FBQztBQUU3RCxNQUFNLE9BQU8sZUFBZTs7Ozs7Ozs7SUFDMUIsTUFBTSxDQUFDLE1BQU0sQ0FBQyxHQUFXLEVBQUUsTUFBYyxFQUFFLEtBQXdCLEVBQUUsT0FNakUsRUFBRTtRQUNKLE1BQU0sRUFBQyxvQkFBb0IsRUFBRSxxQkFBcUIsRUFBRSxxQkFBcUIsRUFBRSxRQUFRLEVBQzVFLGdCQUFnQixHQUFHLEtBQUssRUFBQyxHQUFHLElBQUksQ0FBQzs7UUFDeEMsTUFBTSxPQUFPLEdBQTZCO1lBQ3hDLG9CQUFvQjtZQUNwQixxQkFBcUI7WUFDckIscUJBQXFCO1lBQ3JCLEtBQUssRUFBRSxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsQ0FBQyxXQUFXLEVBQUU7U0FDOUMsQ0FBQztRQUVGLElBQUksS0FBSyxJQUFJLGlCQUFpQixDQUFDLFFBQVEsRUFBRTtZQUN2QyxPQUFPLENBQUMsUUFBUSxHQUFHLE9BQU8sUUFBUSxJQUFJLFFBQVEsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7WUFDdEUsT0FBTyxDQUFDLGVBQWUsR0FBRyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUM7U0FDaEU7UUFDRCxPQUFPLElBQUksSUFBSSxDQUFDLFlBQVksQ0FBQyxNQUFNLEVBQUUsT0FBTyxDQUFDLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0tBQzNEO0NBQ0Y7Ozs7QUFJRCxNQUFNLGtCQUFrQixHQUNwQixxR0FBcUcsQ0FBQzs7QUFFMUcsTUFBTSxlQUFlLEdBQXdDOztJQUUzRCxVQUFVLEVBQUUscUJBQXFCLENBQUMsT0FBTyxDQUFDO1FBQ3hDLGNBQWMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO1FBQ3pCLGFBQWEsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO1FBQ3pCLGNBQWMsQ0FBQyxLQUFLLEVBQUUsQ0FBQyxDQUFDO1FBQ3hCLGNBQWMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO1FBQ3pCLGNBQWMsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO1FBQzNCLGNBQWMsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO0tBQzVCLENBQUMsQ0FBQztJQUNILE9BQU8sRUFBRSxxQkFBcUIsQ0FBQyxPQUFPLENBQUM7UUFDckMsY0FBYyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsRUFBRSxjQUFjLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQyxFQUFFLGNBQWMsQ0FBQyxLQUFLLEVBQUUsQ0FBQyxDQUFDO1FBQy9FLGNBQWMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLEVBQUUsY0FBYyxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUM7S0FDdkQsQ0FBQyxDQUFDO0lBQ0gsWUFBWSxFQUFFLHFCQUFxQixDQUFDLE9BQU8sQ0FBQztRQUMxQyxjQUFjLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxFQUFFLGFBQWEsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDLEVBQUUsYUFBYSxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUM7UUFDakYsY0FBYyxDQUFDLEtBQUssRUFBRSxDQUFDLENBQUM7S0FDekIsQ0FBQyxDQUFDO0lBQ0gsUUFBUSxFQUFFLHFCQUFxQixDQUMzQixPQUFPLENBQUMsQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxFQUFFLGFBQWEsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDLEVBQUUsY0FBYyxDQUFDLEtBQUssRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDOUYsT0FBTyxFQUFFLHFCQUFxQixDQUMxQixPQUFPLENBQUMsQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxFQUFFLGFBQWEsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDLEVBQUUsY0FBYyxDQUFDLEtBQUssRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDOUYsS0FBSyxFQUFFLHFCQUFxQixDQUN4QixPQUFPLENBQUMsQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxFQUFFLGNBQWMsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDLEVBQUUsY0FBYyxDQUFDLEtBQUssRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDL0YsS0FBSyxFQUFFLHFCQUFxQixDQUFDLE9BQU8sQ0FDaEMsQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxFQUFFLGNBQWMsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLEVBQUUsY0FBYyxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDM0YsSUFBSSxFQUFFLHFCQUFxQixDQUFDLE9BQU8sQ0FBQyxDQUFDLGNBQWMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLEVBQUUsY0FBYyxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7Q0FDL0YsQ0FBQzs7QUFFRixNQUFNLFlBQVksR0FBd0M7O0lBRXhELE1BQU0sRUFBRSxxQkFBcUIsQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ3hELElBQUksRUFBRSxxQkFBcUIsQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ3RELEdBQUcsRUFBRSxxQkFBcUIsQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ3JELE1BQU0sRUFBRSxxQkFBcUIsQ0FBQyxhQUFhLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ3hELEtBQUssRUFBRSxxQkFBcUIsQ0FBQyxhQUFhLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ3ZELElBQUksRUFBRSxxQkFBcUIsQ0FBQyxjQUFjLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ3ZELEdBQUcsRUFBRSxxQkFBcUIsQ0FBQyxjQUFjLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ3RELE1BQU0sRUFBRSxxQkFBcUIsQ0FBQyxhQUFhLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ3hELEdBQUcsRUFBRSxxQkFBcUIsQ0FBQyxhQUFhLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ3JELElBQUksRUFBRSxxQkFBcUIsQ0FBQyxjQUFjLENBQUMsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ3JELEdBQUcsRUFBRSxxQkFBcUIsQ0FBQyxjQUFjLENBQUMsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ3BELElBQUksRUFBRSxhQUFhLENBQ2YsYUFBYSxDQUFDLHFCQUFxQixDQUFDLFlBQVksQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxFQUFFLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUN6RixHQUFHLEVBQUUsYUFBYSxDQUFDLHFCQUFxQixDQUFDLFlBQVksQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxFQUFFLEtBQUssQ0FBQyxDQUFDLENBQUM7SUFDekYsSUFBSSxFQUFFLGFBQWEsQ0FDZixhQUFhLENBQUMscUJBQXFCLENBQUMsWUFBWSxDQUFDLGNBQWMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ3hGLEdBQUcsRUFBRSxhQUFhLENBQUMscUJBQXFCLENBQUMsWUFBWSxDQUFDLGNBQWMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQztJQUN4RixJQUFJLEVBQUUscUJBQXFCLENBQUMsY0FBYyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsQ0FBQztJQUN0RCxHQUFHLEVBQUUscUJBQXFCLENBQUMsY0FBYyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsQ0FBQztJQUNyRCxJQUFJLEVBQUUsYUFBYSxDQUFDLHFCQUFxQixDQUFDLGNBQWMsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUN2RSxHQUFHLEVBQUUscUJBQXFCLENBQUMsY0FBYyxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUN2RCxJQUFJLEVBQUUsYUFBYSxDQUFDLHFCQUFxQixDQUFDLGNBQWMsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUN2RSxHQUFHLEVBQUUscUJBQXFCLENBQUMsY0FBYyxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUMsQ0FBQzs7OztJQUl2RCxLQUFLLEVBQUUscUJBQXFCLENBQUMsY0FBYyxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUN6RCxNQUFNLEVBQUUscUJBQXFCLENBQUMsYUFBYSxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUMxRCxLQUFLLEVBQUUscUJBQXFCLENBQUMsYUFBYSxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUN6RCxJQUFJLEVBQUUscUJBQXFCLENBQUMsYUFBYSxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUN4RCxHQUFHLEVBQUUscUJBQXFCLENBQUMsYUFBYSxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUN2RCxHQUFHLEVBQUUsa0JBQWtCLENBQUMscUJBQXFCLENBQUMsWUFBWSxDQUFDLGNBQWMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQztJQUM3RixHQUFHLEVBQUUsY0FBYyxDQUFDLE9BQU8sQ0FBQztJQUM1QixHQUFHLEVBQUUsY0FBYyxDQUFDLE1BQU0sQ0FBQztJQUMzQixJQUFJLEVBQUUscUJBQXFCLENBQUMsRUFBRSxDQUFDOzs7SUFFL0IsR0FBRyxFQUNDLHFCQUFxQixDQUFDLEVBQUUsQ0FBQzs7O0lBRTdCLEdBQUcsRUFBRSxxQkFBcUIsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ25ELElBQUksRUFBRSxxQkFBcUIsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ3BELEtBQUssRUFBRSxxQkFBcUIsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ3JELE1BQU0sRUFBRSxxQkFBcUIsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDO0NBQ3ZELENBQUM7Ozs7O0FBR0YsU0FBUyxhQUFhLENBQUMsS0FBc0I7SUFDM0MsT0FBTyxVQUFTLElBQVUsRUFBRSxNQUFjOztRQUN4QyxNQUFNLE1BQU0sR0FBRyxLQUFLLENBQUMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ25DLE9BQU8sTUFBTSxDQUFDLE1BQU0sSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLEdBQUcsR0FBRyxNQUFNLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQztLQUNuRCxDQUFDO0NBQ0g7Ozs7O0FBRUQsU0FBUyxrQkFBa0IsQ0FBQyxLQUFzQjtJQUNoRCxPQUFPLFVBQVMsSUFBVSxFQUFFLE1BQWMsSUFBWSxPQUFPLEtBQUssQ0FBQyxJQUFJLEVBQUUsTUFBTSxDQUFDLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztDQUNuRzs7Ozs7QUFFRCxTQUFTLGFBQWEsQ0FBQyxLQUFzQjtJQUMzQyxPQUFPLFVBQVMsSUFBVSxFQUFFLE1BQWMsSUFBWSxPQUFPLEtBQUssQ0FBQyxJQUFJLEVBQUUsTUFBTSxDQUFDLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztDQUNuRzs7Ozs7OztBQUVELFNBQVMsY0FBYyxDQUFDLElBQVUsRUFBRSxNQUFjLEVBQUUsT0FBbUM7SUFDckYsT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFLE9BQU8sQ0FBQyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsaUJBQWlCLEVBQUUsRUFBRSxDQUFDLENBQUM7Q0FDN0Y7Ozs7O0FBRUQsU0FBUyxjQUFjLENBQUMsUUFBZ0I7O0lBRXRDLE1BQU0sT0FBTyxHQUFHLEVBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRSxNQUFNLEVBQUUsS0FBSyxFQUFFLFlBQVksRUFBRSxRQUFRLEVBQUMsQ0FBQztJQUN6RSxPQUFPLFVBQVMsSUFBVSxFQUFFLE1BQWM7O1FBQ3hDLE1BQU0sTUFBTSxHQUFHLGNBQWMsQ0FBQyxJQUFJLEVBQUUsTUFBTSxFQUFFLE9BQU8sQ0FBQyxDQUFDOztRQUVyRCxPQUFPLE1BQU0sQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO0tBQzFDLENBQUM7Q0FDSDs7Ozs7O0FBRUQsU0FBUyxZQUFZLENBQ2pCLE9BQW1DLEVBQUUsS0FBYztJQUNyRCxPQUFPLENBQUMsTUFBTSxHQUFHLEtBQUssQ0FBQztJQUN2QixPQUFPLE9BQU8sQ0FBQztDQUNoQjs7Ozs7O0FBRUQsU0FBUyxjQUFjLENBQUMsSUFBWSxFQUFFLEdBQVc7O0lBQy9DLE1BQU0sTUFBTSxHQUEwQixFQUFFLENBQUM7SUFDekMsTUFBTSxDQUFDLElBQUksQ0FBQyxHQUFHLEdBQUcsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO0lBQ2pELE9BQU8sTUFBTSxDQUFDO0NBQ2Y7Ozs7OztBQUVELFNBQVMsYUFBYSxDQUFDLElBQVksRUFBRSxHQUFXOztJQUM5QyxNQUFNLE1BQU0sR0FBMEIsRUFBRSxDQUFDO0lBQ3pDLElBQUksR0FBRyxHQUFHLENBQUMsRUFBRTtRQUNYLE1BQU0sQ0FBQyxJQUFJLENBQUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQztLQUM3QztTQUFNO1FBQ0wsTUFBTSxDQUFDLElBQUksQ0FBQyxHQUFHLE1BQU0sQ0FBQztLQUN2QjtJQUVELE9BQU8sTUFBTSxDQUFDO0NBQ2Y7Ozs7O0FBRUQsU0FBUyxPQUFPLENBQUMsT0FBcUM7SUFDcEQsT0FBTyxPQUFPLENBQUMsTUFBTSxDQUFDLENBQUMsTUFBTSxFQUFFLEdBQUcsRUFBRSxFQUFFLENBQUMsbUJBQUssTUFBTSxFQUFLLEdBQUcsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDO0NBQ25FOzs7OztBQUVELFNBQVMscUJBQXFCLENBQUMsR0FBK0I7SUFDNUQsT0FBTyxDQUFDLElBQVUsRUFBRSxNQUFjLEVBQVUsRUFBRSxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsTUFBTSxFQUFFLEdBQUcsQ0FBQyxDQUFDO0NBQ2xGOztBQUVELE1BQU0sb0JBQW9CLEdBQUcsSUFBSSxHQUFHLEVBQW9CLENBQUM7Ozs7Ozs7QUFFekQsU0FBUyxhQUFhLENBQUMsTUFBYyxFQUFFLElBQVUsRUFBRSxNQUFjOztJQUMvRCxNQUFNLEVBQUUsR0FBRyxlQUFlLENBQUMsTUFBTSxDQUFDLENBQUM7SUFFbkMsSUFBSSxFQUFFO1FBQUUsT0FBTyxFQUFFLENBQUMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxDQUFDOztJQUVoQyxNQUFNLFFBQVEsR0FBRyxNQUFNLENBQUM7O0lBQ3hCLElBQUksS0FBSyxHQUFHLG9CQUFvQixDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUUvQyxJQUFJLENBQUMsS0FBSyxFQUFFO1FBQ1YsS0FBSyxHQUFHLEVBQUUsQ0FBQzs7UUFDWCxJQUFJLEtBQUssQ0FBdUI7UUFDaEMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDOztRQUVoQyxJQUFJLE9BQU8sR0FBZ0IsTUFBTSxDQUFDO1FBQ2xDLE9BQU8sT0FBTyxFQUFFO1lBQ2QsS0FBSyxHQUFHLGtCQUFrQixDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQztZQUN6QyxJQUFJLEtBQUssRUFBRTtnQkFDVCxLQUFLLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ3JDLE9BQU8sc0JBQUcsS0FBSyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUM7YUFDekI7aUJBQU07Z0JBQ0wsS0FBSyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQztnQkFDcEIsT0FBTyxHQUFHLElBQUksQ0FBQzthQUNoQjtTQUNGO1FBRUQsb0JBQW9CLENBQUMsR0FBRyxDQUFDLFFBQVEsRUFBRSxLQUFLLENBQUMsQ0FBQztLQUMzQztJQUVELE9BQU8sS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLElBQUksRUFBRSxJQUFJLEVBQUUsRUFBRTs7UUFDakMsTUFBTSxFQUFFLEdBQUcsWUFBWSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzlCLE9BQU8sSUFBSSxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztLQUMxRCxFQUFFLEVBQUUsQ0FBQyxDQUFDO0NBQ1I7Ozs7O0FBRUQsU0FBUyxVQUFVLENBQUMsSUFBWTtJQUM5QixPQUFPLElBQUksS0FBSyxNQUFNLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxVQUFVLEVBQUUsRUFBRSxDQUFDLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsQ0FBQztDQUNuRjtBQUVELE1BQU0sT0FBTyxhQUFhOzs7Ozs7O0lBQ3hCLE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBVSxFQUFFLE1BQWMsRUFBRSxPQUFlO1FBQ3ZELE9BQU8sYUFBYSxDQUFDLE9BQU8sRUFBRSxJQUFJLEVBQUUsTUFBTSxDQUFDLENBQUM7S0FDN0M7Q0FDRiIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQGxpY2Vuc2VcbiAqIENvcHlyaWdodCBHb29nbGUgSW5jLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqIFVzZSBvZiB0aGlzIHNvdXJjZSBjb2RlIGlzIGdvdmVybmVkIGJ5IGFuIE1JVC1zdHlsZSBsaWNlbnNlIHRoYXQgY2FuIGJlXG4gKiBmb3VuZCBpbiB0aGUgTElDRU5TRSBmaWxlIGF0IGh0dHBzOi8vYW5ndWxhci5pby9saWNlbnNlXG4gKi9cbmltcG9ydCB7TnVtYmVyRm9ybWF0U3R5bGV9IGZyb20gJy4uLy4uL2kxOG4vbG9jYWxlX2RhdGFfYXBpJztcblxuZXhwb3J0IGNsYXNzIE51bWJlckZvcm1hdHRlciB7XG4gIHN0YXRpYyBmb3JtYXQobnVtOiBudW1iZXIsIGxvY2FsZTogc3RyaW5nLCBzdHlsZTogTnVtYmVyRm9ybWF0U3R5bGUsIG9wdHM6IHtcbiAgICBtaW5pbXVtSW50ZWdlckRpZ2l0cz86IG51bWJlcixcbiAgICBtaW5pbXVtRnJhY3Rpb25EaWdpdHM/OiBudW1iZXIsXG4gICAgbWF4aW11bUZyYWN0aW9uRGlnaXRzPzogbnVtYmVyLFxuICAgIGN1cnJlbmN5Pzogc3RyaW5nfG51bGwsXG4gICAgY3VycmVuY3lBc1N5bWJvbD86IGJvb2xlYW5cbiAgfSA9IHt9KTogc3RyaW5nIHtcbiAgICBjb25zdCB7bWluaW11bUludGVnZXJEaWdpdHMsIG1pbmltdW1GcmFjdGlvbkRpZ2l0cywgbWF4aW11bUZyYWN0aW9uRGlnaXRzLCBjdXJyZW5jeSxcbiAgICAgICAgICAgY3VycmVuY3lBc1N5bWJvbCA9IGZhbHNlfSA9IG9wdHM7XG4gICAgY29uc3Qgb3B0aW9uczogSW50bC5OdW1iZXJGb3JtYXRPcHRpb25zID0ge1xuICAgICAgbWluaW11bUludGVnZXJEaWdpdHMsXG4gICAgICBtaW5pbXVtRnJhY3Rpb25EaWdpdHMsXG4gICAgICBtYXhpbXVtRnJhY3Rpb25EaWdpdHMsXG4gICAgICBzdHlsZTogTnVtYmVyRm9ybWF0U3R5bGVbc3R5bGVdLnRvTG93ZXJDYXNlKClcbiAgICB9O1xuXG4gICAgaWYgKHN0eWxlID09IE51bWJlckZvcm1hdFN0eWxlLkN1cnJlbmN5KSB7XG4gICAgICBvcHRpb25zLmN1cnJlbmN5ID0gdHlwZW9mIGN1cnJlbmN5ID09ICdzdHJpbmcnID8gY3VycmVuY3kgOiB1bmRlZmluZWQ7XG4gICAgICBvcHRpb25zLmN1cnJlbmN5RGlzcGxheSA9IGN1cnJlbmN5QXNTeW1ib2wgPyAnc3ltYm9sJyA6ICdjb2RlJztcbiAgICB9XG4gICAgcmV0dXJuIG5ldyBJbnRsLk51bWJlckZvcm1hdChsb2NhbGUsIG9wdGlvbnMpLmZvcm1hdChudW0pO1xuICB9XG59XG5cbnR5cGUgRGF0ZUZvcm1hdHRlckZuID0gKGRhdGU6IERhdGUsIGxvY2FsZTogc3RyaW5nKSA9PiBzdHJpbmc7XG5cbmNvbnN0IERBVEVfRk9STUFUU19TUExJVCA9XG4gICAgLygoPzpbXnlNTGRIaG1zYXpaRXdHakonXSspfCg/OicoPzpbXiddfCcnKSonKXwoPzpFK3x5K3xNK3xMK3xkK3xIK3xoK3xKK3xqK3xtK3xzK3xhfHp8WnxHK3x3KykpKC4qKS87XG5cbmNvbnN0IFBBVFRFUk5fQUxJQVNFUzoge1tmb3JtYXQ6IHN0cmluZ106IERhdGVGb3JtYXR0ZXJGbn0gPSB7XG4gIC8vIEtleXMgYXJlIHF1b3RlZCBzbyB0aGV5IGRvIG5vdCBnZXQgcmVuYW1lZCBkdXJpbmcgY2xvc3VyZSBjb21waWxhdGlvbi5cbiAgJ3lNTU1kam1zJzogZGF0ZVBhcnRHZXR0ZXJGYWN0b3J5KGNvbWJpbmUoW1xuICAgIGRpZ2l0Q29uZGl0aW9uKCd5ZWFyJywgMSksXG4gICAgbmFtZUNvbmRpdGlvbignbW9udGgnLCAzKSxcbiAgICBkaWdpdENvbmRpdGlvbignZGF5JywgMSksXG4gICAgZGlnaXRDb25kaXRpb24oJ2hvdXInLCAxKSxcbiAgICBkaWdpdENvbmRpdGlvbignbWludXRlJywgMSksXG4gICAgZGlnaXRDb25kaXRpb24oJ3NlY29uZCcsIDEpLFxuICBdKSksXG4gICd5TWRqbSc6IGRhdGVQYXJ0R2V0dGVyRmFjdG9yeShjb21iaW5lKFtcbiAgICBkaWdpdENvbmRpdGlvbigneWVhcicsIDEpLCBkaWdpdENvbmRpdGlvbignbW9udGgnLCAxKSwgZGlnaXRDb25kaXRpb24oJ2RheScsIDEpLFxuICAgIGRpZ2l0Q29uZGl0aW9uKCdob3VyJywgMSksIGRpZ2l0Q29uZGl0aW9uKCdtaW51dGUnLCAxKVxuICBdKSksXG4gICd5TU1NTUVFRUVkJzogZGF0ZVBhcnRHZXR0ZXJGYWN0b3J5KGNvbWJpbmUoW1xuICAgIGRpZ2l0Q29uZGl0aW9uKCd5ZWFyJywgMSksIG5hbWVDb25kaXRpb24oJ21vbnRoJywgNCksIG5hbWVDb25kaXRpb24oJ3dlZWtkYXknLCA0KSxcbiAgICBkaWdpdENvbmRpdGlvbignZGF5JywgMSlcbiAgXSkpLFxuICAneU1NTU1kJzogZGF0ZVBhcnRHZXR0ZXJGYWN0b3J5KFxuICAgICAgY29tYmluZShbZGlnaXRDb25kaXRpb24oJ3llYXInLCAxKSwgbmFtZUNvbmRpdGlvbignbW9udGgnLCA0KSwgZGlnaXRDb25kaXRpb24oJ2RheScsIDEpXSkpLFxuICAneU1NTWQnOiBkYXRlUGFydEdldHRlckZhY3RvcnkoXG4gICAgICBjb21iaW5lKFtkaWdpdENvbmRpdGlvbigneWVhcicsIDEpLCBuYW1lQ29uZGl0aW9uKCdtb250aCcsIDMpLCBkaWdpdENvbmRpdGlvbignZGF5JywgMSldKSksXG4gICd5TWQnOiBkYXRlUGFydEdldHRlckZhY3RvcnkoXG4gICAgICBjb21iaW5lKFtkaWdpdENvbmRpdGlvbigneWVhcicsIDEpLCBkaWdpdENvbmRpdGlvbignbW9udGgnLCAxKSwgZGlnaXRDb25kaXRpb24oJ2RheScsIDEpXSkpLFxuICAnam1zJzogZGF0ZVBhcnRHZXR0ZXJGYWN0b3J5KGNvbWJpbmUoXG4gICAgICBbZGlnaXRDb25kaXRpb24oJ2hvdXInLCAxKSwgZGlnaXRDb25kaXRpb24oJ3NlY29uZCcsIDEpLCBkaWdpdENvbmRpdGlvbignbWludXRlJywgMSldKSksXG4gICdqbSc6IGRhdGVQYXJ0R2V0dGVyRmFjdG9yeShjb21iaW5lKFtkaWdpdENvbmRpdGlvbignaG91cicsIDEpLCBkaWdpdENvbmRpdGlvbignbWludXRlJywgMSldKSlcbn07XG5cbmNvbnN0IERBVEVfRk9STUFUUzoge1tmb3JtYXQ6IHN0cmluZ106IERhdGVGb3JtYXR0ZXJGbn0gPSB7XG4gIC8vIEtleXMgYXJlIHF1b3RlZCBzbyB0aGV5IGRvIG5vdCBnZXQgcmVuYW1lZC5cbiAgJ3l5eXknOiBkYXRlUGFydEdldHRlckZhY3RvcnkoZGlnaXRDb25kaXRpb24oJ3llYXInLCA0KSksXG4gICd5eSc6IGRhdGVQYXJ0R2V0dGVyRmFjdG9yeShkaWdpdENvbmRpdGlvbigneWVhcicsIDIpKSxcbiAgJ3knOiBkYXRlUGFydEdldHRlckZhY3RvcnkoZGlnaXRDb25kaXRpb24oJ3llYXInLCAxKSksXG4gICdNTU1NJzogZGF0ZVBhcnRHZXR0ZXJGYWN0b3J5KG5hbWVDb25kaXRpb24oJ21vbnRoJywgNCkpLFxuICAnTU1NJzogZGF0ZVBhcnRHZXR0ZXJGYWN0b3J5KG5hbWVDb25kaXRpb24oJ21vbnRoJywgMykpLFxuICAnTU0nOiBkYXRlUGFydEdldHRlckZhY3RvcnkoZGlnaXRDb25kaXRpb24oJ21vbnRoJywgMikpLFxuICAnTSc6IGRhdGVQYXJ0R2V0dGVyRmFjdG9yeShkaWdpdENvbmRpdGlvbignbW9udGgnLCAxKSksXG4gICdMTExMJzogZGF0ZVBhcnRHZXR0ZXJGYWN0b3J5KG5hbWVDb25kaXRpb24oJ21vbnRoJywgNCkpLFxuICAnTCc6IGRhdGVQYXJ0R2V0dGVyRmFjdG9yeShuYW1lQ29uZGl0aW9uKCdtb250aCcsIDEpKSxcbiAgJ2RkJzogZGF0ZVBhcnRHZXR0ZXJGYWN0b3J5KGRpZ2l0Q29uZGl0aW9uKCdkYXknLCAyKSksXG4gICdkJzogZGF0ZVBhcnRHZXR0ZXJGYWN0b3J5KGRpZ2l0Q29uZGl0aW9uKCdkYXknLCAxKSksXG4gICdISCc6IGRpZ2l0TW9kaWZpZXIoXG4gICAgICBob3VyRXh0cmFjdG9yKGRhdGVQYXJ0R2V0dGVyRmFjdG9yeShob3VyMTJNb2RpZnkoZGlnaXRDb25kaXRpb24oJ2hvdXInLCAyKSwgZmFsc2UpKSkpLFxuICAnSCc6IGhvdXJFeHRyYWN0b3IoZGF0ZVBhcnRHZXR0ZXJGYWN0b3J5KGhvdXIxMk1vZGlmeShkaWdpdENvbmRpdGlvbignaG91cicsIDEpLCBmYWxzZSkpKSxcbiAgJ2hoJzogZGlnaXRNb2RpZmllcihcbiAgICAgIGhvdXJFeHRyYWN0b3IoZGF0ZVBhcnRHZXR0ZXJGYWN0b3J5KGhvdXIxMk1vZGlmeShkaWdpdENvbmRpdGlvbignaG91cicsIDIpLCB0cnVlKSkpKSxcbiAgJ2gnOiBob3VyRXh0cmFjdG9yKGRhdGVQYXJ0R2V0dGVyRmFjdG9yeShob3VyMTJNb2RpZnkoZGlnaXRDb25kaXRpb24oJ2hvdXInLCAxKSwgdHJ1ZSkpKSxcbiAgJ2pqJzogZGF0ZVBhcnRHZXR0ZXJGYWN0b3J5KGRpZ2l0Q29uZGl0aW9uKCdob3VyJywgMikpLFxuICAnaic6IGRhdGVQYXJ0R2V0dGVyRmFjdG9yeShkaWdpdENvbmRpdGlvbignaG91cicsIDEpKSxcbiAgJ21tJzogZGlnaXRNb2RpZmllcihkYXRlUGFydEdldHRlckZhY3RvcnkoZGlnaXRDb25kaXRpb24oJ21pbnV0ZScsIDIpKSksXG4gICdtJzogZGF0ZVBhcnRHZXR0ZXJGYWN0b3J5KGRpZ2l0Q29uZGl0aW9uKCdtaW51dGUnLCAxKSksXG4gICdzcyc6IGRpZ2l0TW9kaWZpZXIoZGF0ZVBhcnRHZXR0ZXJGYWN0b3J5KGRpZ2l0Q29uZGl0aW9uKCdzZWNvbmQnLCAyKSkpLFxuICAncyc6IGRhdGVQYXJ0R2V0dGVyRmFjdG9yeShkaWdpdENvbmRpdGlvbignc2Vjb25kJywgMSkpLFxuICAvLyB3aGlsZSBJU08gODYwMSByZXF1aXJlcyBmcmFjdGlvbnMgdG8gYmUgcHJlZml4ZWQgd2l0aCBgLmAgb3IgYCxgXG4gIC8vIHdlIGNhbiBiZSBqdXN0IHNhZmVseSByZWx5IG9uIHVzaW5nIGBzc3NgIHNpbmNlIHdlIGN1cnJlbnRseSBkb24ndCBzdXBwb3J0IHNpbmdsZSBvciB0d28gZGlnaXRcbiAgLy8gZnJhY3Rpb25zXG4gICdzc3MnOiBkYXRlUGFydEdldHRlckZhY3RvcnkoZGlnaXRDb25kaXRpb24oJ3NlY29uZCcsIDMpKSxcbiAgJ0VFRUUnOiBkYXRlUGFydEdldHRlckZhY3RvcnkobmFtZUNvbmRpdGlvbignd2Vla2RheScsIDQpKSxcbiAgJ0VFRSc6IGRhdGVQYXJ0R2V0dGVyRmFjdG9yeShuYW1lQ29uZGl0aW9uKCd3ZWVrZGF5JywgMykpLFxuICAnRUUnOiBkYXRlUGFydEdldHRlckZhY3RvcnkobmFtZUNvbmRpdGlvbignd2Vla2RheScsIDIpKSxcbiAgJ0UnOiBkYXRlUGFydEdldHRlckZhY3RvcnkobmFtZUNvbmRpdGlvbignd2Vla2RheScsIDEpKSxcbiAgJ2EnOiBob3VyQ2xvY2tFeHRyYWN0b3IoZGF0ZVBhcnRHZXR0ZXJGYWN0b3J5KGhvdXIxMk1vZGlmeShkaWdpdENvbmRpdGlvbignaG91cicsIDEpLCB0cnVlKSkpLFxuICAnWic6IHRpbWVab25lR2V0dGVyKCdzaG9ydCcpLFxuICAneic6IHRpbWVab25lR2V0dGVyKCdsb25nJyksXG4gICd3dyc6IGRhdGVQYXJ0R2V0dGVyRmFjdG9yeSh7fSksICAvLyBXZWVrIG9mIHllYXIsIHBhZGRlZCAoMDAtNTMpLiBXZWVrIDAxIGlzIHRoZSB3ZWVrIHdpdGggdGhlXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAvLyBmaXJzdCBUaHVyc2RheSBvZiB0aGUgeWVhci4gbm90IHN1cHBvcnQgP1xuICAndyc6XG4gICAgICBkYXRlUGFydEdldHRlckZhY3Rvcnkoe30pLCAgLy8gV2VlayBvZiB5ZWFyICgwLTUzKS4gV2VlayAxIGlzIHRoZSB3ZWVrIHdpdGggdGhlIGZpcnN0IFRodXJzZGF5XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgLy8gb2YgdGhlIHllYXIgbm90IHN1cHBvcnQgP1xuICAnRyc6IGRhdGVQYXJ0R2V0dGVyRmFjdG9yeShuYW1lQ29uZGl0aW9uKCdlcmEnLCAxKSksXG4gICdHRyc6IGRhdGVQYXJ0R2V0dGVyRmFjdG9yeShuYW1lQ29uZGl0aW9uKCdlcmEnLCAyKSksXG4gICdHR0cnOiBkYXRlUGFydEdldHRlckZhY3RvcnkobmFtZUNvbmRpdGlvbignZXJhJywgMykpLFxuICAnR0dHRyc6IGRhdGVQYXJ0R2V0dGVyRmFjdG9yeShuYW1lQ29uZGl0aW9uKCdlcmEnLCA0KSlcbn07XG5cblxuZnVuY3Rpb24gZGlnaXRNb2RpZmllcihpbm5lcjogRGF0ZUZvcm1hdHRlckZuKTogRGF0ZUZvcm1hdHRlckZuIHtcbiAgcmV0dXJuIGZ1bmN0aW9uKGRhdGU6IERhdGUsIGxvY2FsZTogc3RyaW5nKTogc3RyaW5nIHtcbiAgICBjb25zdCByZXN1bHQgPSBpbm5lcihkYXRlLCBsb2NhbGUpO1xuICAgIHJldHVybiByZXN1bHQubGVuZ3RoID09IDEgPyAnMCcgKyByZXN1bHQgOiByZXN1bHQ7XG4gIH07XG59XG5cbmZ1bmN0aW9uIGhvdXJDbG9ja0V4dHJhY3Rvcihpbm5lcjogRGF0ZUZvcm1hdHRlckZuKTogRGF0ZUZvcm1hdHRlckZuIHtcbiAgcmV0dXJuIGZ1bmN0aW9uKGRhdGU6IERhdGUsIGxvY2FsZTogc3RyaW5nKTogc3RyaW5nIHsgcmV0dXJuIGlubmVyKGRhdGUsIGxvY2FsZSkuc3BsaXQoJyAnKVsxXTsgfTtcbn1cblxuZnVuY3Rpb24gaG91ckV4dHJhY3Rvcihpbm5lcjogRGF0ZUZvcm1hdHRlckZuKTogRGF0ZUZvcm1hdHRlckZuIHtcbiAgcmV0dXJuIGZ1bmN0aW9uKGRhdGU6IERhdGUsIGxvY2FsZTogc3RyaW5nKTogc3RyaW5nIHsgcmV0dXJuIGlubmVyKGRhdGUsIGxvY2FsZSkuc3BsaXQoJyAnKVswXTsgfTtcbn1cblxuZnVuY3Rpb24gaW50bERhdGVGb3JtYXQoZGF0ZTogRGF0ZSwgbG9jYWxlOiBzdHJpbmcsIG9wdGlvbnM6IEludGwuRGF0ZVRpbWVGb3JtYXRPcHRpb25zKTogc3RyaW5nIHtcbiAgcmV0dXJuIG5ldyBJbnRsLkRhdGVUaW1lRm9ybWF0KGxvY2FsZSwgb3B0aW9ucykuZm9ybWF0KGRhdGUpLnJlcGxhY2UoL1tcXHUyMDBlXFx1MjAwZl0vZywgJycpO1xufVxuXG5mdW5jdGlvbiB0aW1lWm9uZUdldHRlcih0aW1lem9uZTogc3RyaW5nKTogRGF0ZUZvcm1hdHRlckZuIHtcbiAgLy8gVG8gd29ya2Fyb3VuZCBgSW50bGAgQVBJIHJlc3RyaWN0aW9uIGZvciBzaW5nbGUgdGltZXpvbmUgbGV0IGZvcm1hdCB3aXRoIDI0IGhvdXJzXG4gIGNvbnN0IG9wdGlvbnMgPSB7aG91cjogJzItZGlnaXQnLCBob3VyMTI6IGZhbHNlLCB0aW1lWm9uZU5hbWU6IHRpbWV6b25lfTtcbiAgcmV0dXJuIGZ1bmN0aW9uKGRhdGU6IERhdGUsIGxvY2FsZTogc3RyaW5nKTogc3RyaW5nIHtcbiAgICBjb25zdCByZXN1bHQgPSBpbnRsRGF0ZUZvcm1hdChkYXRlLCBsb2NhbGUsIG9wdGlvbnMpO1xuICAgIC8vIFRoZW4gZXh0cmFjdCBmaXJzdCAzIGxldHRlcnMgdGhhdCByZWxhdGVkIHRvIGhvdXJzXG4gICAgcmV0dXJuIHJlc3VsdCA/IHJlc3VsdC5zdWJzdHJpbmcoMykgOiAnJztcbiAgfTtcbn1cblxuZnVuY3Rpb24gaG91cjEyTW9kaWZ5KFxuICAgIG9wdGlvbnM6IEludGwuRGF0ZVRpbWVGb3JtYXRPcHRpb25zLCB2YWx1ZTogYm9vbGVhbik6IEludGwuRGF0ZVRpbWVGb3JtYXRPcHRpb25zIHtcbiAgb3B0aW9ucy5ob3VyMTIgPSB2YWx1ZTtcbiAgcmV0dXJuIG9wdGlvbnM7XG59XG5cbmZ1bmN0aW9uIGRpZ2l0Q29uZGl0aW9uKHByb3A6IHN0cmluZywgbGVuOiBudW1iZXIpOiBJbnRsLkRhdGVUaW1lRm9ybWF0T3B0aW9ucyB7XG4gIGNvbnN0IHJlc3VsdDoge1trOiBzdHJpbmddOiBzdHJpbmd9ID0ge307XG4gIHJlc3VsdFtwcm9wXSA9IGxlbiA9PT0gMiA/ICcyLWRpZ2l0JyA6ICdudW1lcmljJztcbiAgcmV0dXJuIHJlc3VsdDtcbn1cblxuZnVuY3Rpb24gbmFtZUNvbmRpdGlvbihwcm9wOiBzdHJpbmcsIGxlbjogbnVtYmVyKTogSW50bC5EYXRlVGltZUZvcm1hdE9wdGlvbnMge1xuICBjb25zdCByZXN1bHQ6IHtbazogc3RyaW5nXTogc3RyaW5nfSA9IHt9O1xuICBpZiAobGVuIDwgNCkge1xuICAgIHJlc3VsdFtwcm9wXSA9IGxlbiA+IDEgPyAnc2hvcnQnIDogJ25hcnJvdyc7XG4gIH0gZWxzZSB7XG4gICAgcmVzdWx0W3Byb3BdID0gJ2xvbmcnO1xuICB9XG5cbiAgcmV0dXJuIHJlc3VsdDtcbn1cblxuZnVuY3Rpb24gY29tYmluZShvcHRpb25zOiBJbnRsLkRhdGVUaW1lRm9ybWF0T3B0aW9uc1tdKTogSW50bC5EYXRlVGltZUZvcm1hdE9wdGlvbnMge1xuICByZXR1cm4gb3B0aW9ucy5yZWR1Y2UoKG1lcmdlZCwgb3B0KSA9PiAoey4uLm1lcmdlZCwgLi4ub3B0fSksIHt9KTtcbn1cblxuZnVuY3Rpb24gZGF0ZVBhcnRHZXR0ZXJGYWN0b3J5KHJldDogSW50bC5EYXRlVGltZUZvcm1hdE9wdGlvbnMpOiBEYXRlRm9ybWF0dGVyRm4ge1xuICByZXR1cm4gKGRhdGU6IERhdGUsIGxvY2FsZTogc3RyaW5nKTogc3RyaW5nID0+IGludGxEYXRlRm9ybWF0KGRhdGUsIGxvY2FsZSwgcmV0KTtcbn1cblxuY29uc3QgREFURV9GT1JNQVRURVJfQ0FDSEUgPSBuZXcgTWFwPHN0cmluZywgc3RyaW5nW10+KCk7XG5cbmZ1bmN0aW9uIGRhdGVGb3JtYXR0ZXIoZm9ybWF0OiBzdHJpbmcsIGRhdGU6IERhdGUsIGxvY2FsZTogc3RyaW5nKTogc3RyaW5nIHtcbiAgY29uc3QgZm4gPSBQQVRURVJOX0FMSUFTRVNbZm9ybWF0XTtcblxuICBpZiAoZm4pIHJldHVybiBmbihkYXRlLCBsb2NhbGUpO1xuXG4gIGNvbnN0IGNhY2hlS2V5ID0gZm9ybWF0O1xuICBsZXQgcGFydHMgPSBEQVRFX0ZPUk1BVFRFUl9DQUNIRS5nZXQoY2FjaGVLZXkpO1xuXG4gIGlmICghcGFydHMpIHtcbiAgICBwYXJ0cyA9IFtdO1xuICAgIGxldCBtYXRjaDogUmVnRXhwRXhlY0FycmF5fG51bGw7XG4gICAgREFURV9GT1JNQVRTX1NQTElULmV4ZWMoZm9ybWF0KTtcblxuICAgIGxldCBfZm9ybWF0OiBzdHJpbmd8bnVsbCA9IGZvcm1hdDtcbiAgICB3aGlsZSAoX2Zvcm1hdCkge1xuICAgICAgbWF0Y2ggPSBEQVRFX0ZPUk1BVFNfU1BMSVQuZXhlYyhfZm9ybWF0KTtcbiAgICAgIGlmIChtYXRjaCkge1xuICAgICAgICBwYXJ0cyA9IHBhcnRzLmNvbmNhdChtYXRjaC5zbGljZSgxKSk7XG4gICAgICAgIF9mb3JtYXQgPSBwYXJ0cy5wb3AoKSAhO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgcGFydHMucHVzaChfZm9ybWF0KTtcbiAgICAgICAgX2Zvcm1hdCA9IG51bGw7XG4gICAgICB9XG4gICAgfVxuXG4gICAgREFURV9GT1JNQVRURVJfQ0FDSEUuc2V0KGNhY2hlS2V5LCBwYXJ0cyk7XG4gIH1cblxuICByZXR1cm4gcGFydHMucmVkdWNlKCh0ZXh0LCBwYXJ0KSA9PiB7XG4gICAgY29uc3QgZm4gPSBEQVRFX0ZPUk1BVFNbcGFydF07XG4gICAgcmV0dXJuIHRleHQgKyAoZm4gPyBmbihkYXRlLCBsb2NhbGUpIDogcGFydFRvVGltZShwYXJ0KSk7XG4gIH0sICcnKTtcbn1cblxuZnVuY3Rpb24gcGFydFRvVGltZShwYXJ0OiBzdHJpbmcpOiBzdHJpbmcge1xuICByZXR1cm4gcGFydCA9PT0gJ1xcJ1xcJycgPyAnXFwnJyA6IHBhcnQucmVwbGFjZSgvKF4nfCckKS9nLCAnJykucmVwbGFjZSgvJycvZywgJ1xcJycpO1xufVxuXG5leHBvcnQgY2xhc3MgRGF0ZUZvcm1hdHRlciB7XG4gIHN0YXRpYyBmb3JtYXQoZGF0ZTogRGF0ZSwgbG9jYWxlOiBzdHJpbmcsIHBhdHRlcm46IHN0cmluZyk6IHN0cmluZyB7XG4gICAgcmV0dXJuIGRhdGVGb3JtYXR0ZXIocGF0dGVybiwgZGF0ZSwgbG9jYWxlKTtcbiAgfVxufVxuIl19