/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as tslib_1 from "tslib";
import { Injectable } from '@angular/core';
import { of } from 'rxjs';
import { concatMap, filter, map } from 'rxjs/operators';
import { HttpHandler } from './backend';
import { HttpHeaders } from './headers';
import { HttpParams } from './params';
import { HttpRequest } from './request';
import { HttpResponse } from './response';
/**
 * Construct an instance of `HttpRequestOptions<T>` from a source `HttpMethodOptions` and
 * the given `body`. Basically, this clones the object and adds the body.
 */
function addBody(options, body) {
    return {
        body: body,
        headers: options.headers,
        observe: options.observe,
        params: options.params,
        reportProgress: options.reportProgress,
        responseType: options.responseType,
        withCredentials: options.withCredentials,
    };
}
/**
 * Perform HTTP requests.
 *
 * `HttpClient` is available as an injectable class, with methods to perform HTTP requests.
 * Each request method has multiple signatures, and the return type varies according to which
 * signature is called (mainly the values of `observe` and `responseType`).
 *
 * @publicApi
 */
var HttpClient = /** @class */ (function () {
    function HttpClient(handler) {
        this.handler = handler;
    }
    /**
     * Constructs an `Observable` for a particular HTTP request that, when subscribed,
     * fires the request through the chain of registered interceptors and on to the
     * server.
     *
     * This method can be called in one of two ways. Either an `HttpRequest`
     * instance can be passed directly as the only parameter, or a method can be
     * passed as the first parameter, a string URL as the second, and an
     * options hash as the third.
     *
     * If a `HttpRequest` object is passed directly, an `Observable` of the
     * raw `HttpEvent` stream will be returned.
     *
     * If a request is instead built by providing a URL, the options object
     * determines the return type of `request()`. In addition to configuring
     * request parameters such as the outgoing headers and/or the body, the options
     * hash specifies two key pieces of information about the request: the
     * `responseType` and what to `observe`.
     *
     * The `responseType` value determines how a successful response body will be
     * parsed. If `responseType` is the default `json`, a type interface for the
     * resulting object may be passed as a type parameter to `request()`.
     *
     * The `observe` value determines the return type of `request()`, based on what
     * the consumer is interested in observing. A value of `events` will return an
     * `Observable<HttpEvent>` representing the raw `HttpEvent` stream,
     * including progress events by default. A value of `response` will return an
     * `Observable<HttpResponse<T>>` where the `T` parameter of `HttpResponse`
     * depends on the `responseType` and any optionally provided type parameter.
     * A value of `body` will return an `Observable<T>` with the same `T` body type.
     */
    HttpClient.prototype.request = function (first, url, options) {
        var _this = this;
        if (options === void 0) { options = {}; }
        var req;
        // Firstly, check whether the primary argument is an instance of `HttpRequest`.
        if (first instanceof HttpRequest) {
            // It is. The other arguments must be undefined (per the signatures) and can be
            // ignored.
            req = first;
        }
        else {
            // It's a string, so it represents a URL. Construct a request based on it,
            // and incorporate the remaining arguments (assuming GET unless a method is
            // provided.
            // Figure out the headers.
            var headers = undefined;
            if (options.headers instanceof HttpHeaders) {
                headers = options.headers;
            }
            else {
                headers = new HttpHeaders(options.headers);
            }
            // Sort out parameters.
            var params = undefined;
            if (!!options.params) {
                if (options.params instanceof HttpParams) {
                    params = options.params;
                }
                else {
                    params = new HttpParams({ fromObject: options.params });
                }
            }
            // Construct the request.
            req = new HttpRequest(first, url, (options.body !== undefined ? options.body : null), {
                headers: headers,
                params: params,
                reportProgress: options.reportProgress,
                // By default, JSON is assumed to be returned for all calls.
                responseType: options.responseType || 'json',
                withCredentials: options.withCredentials,
            });
        }
        // Start with an Observable.of() the initial request, and run the handler (which
        // includes all interceptors) inside a concatMap(). This way, the handler runs
        // inside an Observable chain, which causes interceptors to be re-run on every
        // subscription (this also makes retries re-run the handler, including interceptors).
        var events$ = of(req).pipe(concatMap(function (req) { return _this.handler.handle(req); }));
        // If coming via the API signature which accepts a previously constructed HttpRequest,
        // the only option is to get the event stream. Otherwise, return the event stream if
        // that is what was requested.
        if (first instanceof HttpRequest || options.observe === 'events') {
            return events$;
        }
        // The requested stream contains either the full response or the body. In either
        // case, the first step is to filter the event stream to extract a stream of
        // responses(s).
        var res$ = events$.pipe(filter(function (event) { return event instanceof HttpResponse; }));
        // Decide which stream to return.
        switch (options.observe || 'body') {
            case 'body':
                // The requested stream is the body. Map the response stream to the response
                // body. This could be done more simply, but a misbehaving interceptor might
                // transform the response body into a different format and ignore the requested
                // responseType. Guard against this by validating that the response is of the
                // requested type.
                switch (req.responseType) {
                    case 'arraybuffer':
                        return res$.pipe(map(function (res) {
                            // Validate that the body is an ArrayBuffer.
                            if (res.body !== null && !(res.body instanceof ArrayBuffer)) {
                                throw new Error('Response is not an ArrayBuffer.');
                            }
                            return res.body;
                        }));
                    case 'blob':
                        return res$.pipe(map(function (res) {
                            // Validate that the body is a Blob.
                            if (res.body !== null && !(res.body instanceof Blob)) {
                                throw new Error('Response is not a Blob.');
                            }
                            return res.body;
                        }));
                    case 'text':
                        return res$.pipe(map(function (res) {
                            // Validate that the body is a string.
                            if (res.body !== null && typeof res.body !== 'string') {
                                throw new Error('Response is not a string.');
                            }
                            return res.body;
                        }));
                    case 'json':
                    default:
                        // No validation needed for JSON responses, as they can be of any type.
                        return res$.pipe(map(function (res) { return res.body; }));
                }
            case 'response':
                // The response stream was requested directly, so return it.
                return res$;
            default:
                // Guard against new future observe types being added.
                throw new Error("Unreachable: unhandled observe type " + options.observe + "}");
        }
    };
    /**
     * Constructs an `Observable` which, when subscribed, will cause the configured
     * DELETE request to be executed on the server. See the individual overloads for
     * details of `delete()`'s return type based on the provided options.
     */
    HttpClient.prototype.delete = function (url, options) {
        if (options === void 0) { options = {}; }
        return this.request('DELETE', url, options);
    };
    /**
     * Constructs an `Observable` which, when subscribed, will cause the configured
     * GET request to be executed on the server. See the individual overloads for
     * details of `get()`'s return type based on the provided options.
     */
    HttpClient.prototype.get = function (url, options) {
        if (options === void 0) { options = {}; }
        return this.request('GET', url, options);
    };
    /**
     * Constructs an `Observable` which, when subscribed, will cause the configured
     * HEAD request to be executed on the server. See the individual overloads for
     * details of `head()`'s return type based on the provided options.
     */
    HttpClient.prototype.head = function (url, options) {
        if (options === void 0) { options = {}; }
        return this.request('HEAD', url, options);
    };
    /**
     * Constructs an `Observable` which, when subscribed, will cause a request
     * with the special method `JSONP` to be dispatched via the interceptor pipeline.
     *
     * A suitable interceptor must be installed (e.g. via the `HttpClientJsonpModule`).
     * If no such interceptor is reached, then the `JSONP` request will likely be
     * rejected by the configured backend.
     */
    HttpClient.prototype.jsonp = function (url, callbackParam) {
        return this.request('JSONP', url, {
            params: new HttpParams().append(callbackParam, 'JSONP_CALLBACK'),
            observe: 'body',
            responseType: 'json',
        });
    };
    /**
     * Constructs an `Observable` which, when subscribed, will cause the configured
     * OPTIONS request to be executed on the server. See the individual overloads for
     * details of `options()`'s return type based on the provided options.
     */
    HttpClient.prototype.options = function (url, options) {
        if (options === void 0) { options = {}; }
        return this.request('OPTIONS', url, options);
    };
    /**
     * Constructs an `Observable` which, when subscribed, will cause the configured
     * PATCH request to be executed on the server. See the individual overloads for
     * details of `patch()`'s return type based on the provided options.
     */
    HttpClient.prototype.patch = function (url, body, options) {
        if (options === void 0) { options = {}; }
        return this.request('PATCH', url, addBody(options, body));
    };
    /**
     * Constructs an `Observable` which, when subscribed, will cause the configured
     * POST request to be executed on the server. See the individual overloads for
     * details of `post()`'s return type based on the provided options.
     */
    HttpClient.prototype.post = function (url, body, options) {
        if (options === void 0) { options = {}; }
        return this.request('POST', url, addBody(options, body));
    };
    /**
     * Constructs an `Observable` which, when subscribed, will cause the configured
     * PUT request to be executed on the server. See the individual overloads for
     * details of `put()`'s return type based on the provided options.
     */
    HttpClient.prototype.put = function (url, body, options) {
        if (options === void 0) { options = {}; }
        return this.request('PUT', url, addBody(options, body));
    };
    HttpClient = tslib_1.__decorate([
        Injectable(),
        tslib_1.__metadata("design:paramtypes", [HttpHandler])
    ], HttpClient);
    return HttpClient;
}());
export { HttpClient };

//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2xpZW50LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vLi4vLi4vLi4vLi4vLi4vLi4vLi4vLi4vcGFja2FnZXMvY29tbW9uL2h0dHAvc3JjL2NsaWVudC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTs7Ozs7O0dBTUc7O0FBRUgsT0FBTyxFQUFDLFVBQVUsRUFBQyxNQUFNLGVBQWUsQ0FBQztBQUN6QyxPQUFPLEVBQWEsRUFBRSxFQUFFLE1BQU0sTUFBTSxDQUFDO0FBQ3JDLE9BQU8sRUFBQyxTQUFTLEVBQUUsTUFBTSxFQUFFLEdBQUcsRUFBQyxNQUFNLGdCQUFnQixDQUFDO0FBRXRELE9BQU8sRUFBQyxXQUFXLEVBQUMsTUFBTSxXQUFXLENBQUM7QUFDdEMsT0FBTyxFQUFDLFdBQVcsRUFBQyxNQUFNLFdBQVcsQ0FBQztBQUN0QyxPQUFPLEVBQUMsVUFBVSxFQUFvQixNQUFNLFVBQVUsQ0FBQztBQUN2RCxPQUFPLEVBQUMsV0FBVyxFQUFDLE1BQU0sV0FBVyxDQUFDO0FBQ3RDLE9BQU8sRUFBWSxZQUFZLEVBQUMsTUFBTSxZQUFZLENBQUM7QUFHbkQ7OztHQUdHO0FBQ0gsU0FBUyxPQUFPLENBQ1osT0FPQyxFQUNELElBQWM7SUFDaEIsT0FBTztRQUNMLElBQUksTUFBQTtRQUNKLE9BQU8sRUFBRSxPQUFPLENBQUMsT0FBTztRQUN4QixPQUFPLEVBQUUsT0FBTyxDQUFDLE9BQU87UUFDeEIsTUFBTSxFQUFFLE9BQU8sQ0FBQyxNQUFNO1FBQ3RCLGNBQWMsRUFBRSxPQUFPLENBQUMsY0FBYztRQUN0QyxZQUFZLEVBQUUsT0FBTyxDQUFDLFlBQVk7UUFDbEMsZUFBZSxFQUFFLE9BQU8sQ0FBQyxlQUFlO0tBQ3pDLENBQUM7QUFDSixDQUFDO0FBSUQ7Ozs7Ozs7O0dBUUc7QUFFSDtJQUNFLG9CQUFvQixPQUFvQjtRQUFwQixZQUFPLEdBQVAsT0FBTyxDQUFhO0lBQUcsQ0FBQztJQThPNUM7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQThCRztJQUNILDRCQUFPLEdBQVAsVUFBUSxLQUE4QixFQUFFLEdBQVksRUFBRSxPQVFoRDtRQVJOLGlCQWtIQztRQWxIcUQsd0JBQUEsRUFBQSxZQVFoRDtRQUNKLElBQUksR0FBcUIsQ0FBQztRQUMxQiwrRUFBK0U7UUFDL0UsSUFBSSxLQUFLLFlBQVksV0FBVyxFQUFFO1lBQ2hDLCtFQUErRTtZQUMvRSxXQUFXO1lBQ1gsR0FBRyxHQUFHLEtBQXlCLENBQUM7U0FDakM7YUFBTTtZQUNMLDBFQUEwRTtZQUMxRSwyRUFBMkU7WUFDM0UsWUFBWTtZQUVaLDBCQUEwQjtZQUMxQixJQUFJLE9BQU8sR0FBMEIsU0FBUyxDQUFDO1lBQy9DLElBQUksT0FBTyxDQUFDLE9BQU8sWUFBWSxXQUFXLEVBQUU7Z0JBQzFDLE9BQU8sR0FBRyxPQUFPLENBQUMsT0FBTyxDQUFDO2FBQzNCO2lCQUFNO2dCQUNMLE9BQU8sR0FBRyxJQUFJLFdBQVcsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUM7YUFDNUM7WUFFRCx1QkFBdUI7WUFDdkIsSUFBSSxNQUFNLEdBQXlCLFNBQVMsQ0FBQztZQUM3QyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsTUFBTSxFQUFFO2dCQUNwQixJQUFJLE9BQU8sQ0FBQyxNQUFNLFlBQVksVUFBVSxFQUFFO29CQUN4QyxNQUFNLEdBQUcsT0FBTyxDQUFDLE1BQU0sQ0FBQztpQkFDekI7cUJBQU07b0JBQ0wsTUFBTSxHQUFHLElBQUksVUFBVSxDQUFDLEVBQUUsVUFBVSxFQUFFLE9BQU8sQ0FBQyxNQUFNLEVBQXVCLENBQUMsQ0FBQztpQkFDOUU7YUFDRjtZQUVELHlCQUF5QjtZQUN6QixHQUFHLEdBQUcsSUFBSSxXQUFXLENBQUMsS0FBSyxFQUFFLEdBQUssRUFBRSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFBRTtnQkFDdEYsT0FBTyxTQUFBO2dCQUNQLE1BQU0sUUFBQTtnQkFDTixjQUFjLEVBQUUsT0FBTyxDQUFDLGNBQWM7Z0JBQ3RDLDREQUE0RDtnQkFDNUQsWUFBWSxFQUFFLE9BQU8sQ0FBQyxZQUFZLElBQUksTUFBTTtnQkFDNUMsZUFBZSxFQUFFLE9BQU8sQ0FBQyxlQUFlO2FBQ3pDLENBQUMsQ0FBQztTQUNKO1FBRUQsZ0ZBQWdGO1FBQ2hGLDhFQUE4RTtRQUM5RSw4RUFBOEU7UUFDOUUscUZBQXFGO1FBQ3JGLElBQU0sT0FBTyxHQUNULEVBQUUsQ0FBRSxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFVBQUMsR0FBcUIsSUFBSyxPQUFBLEtBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxFQUF4QixDQUF3QixDQUFDLENBQUMsQ0FBQztRQUVsRixzRkFBc0Y7UUFDdEYsb0ZBQW9GO1FBQ3BGLDhCQUE4QjtRQUM5QixJQUFJLEtBQUssWUFBWSxXQUFXLElBQUksT0FBTyxDQUFDLE9BQU8sS0FBSyxRQUFRLEVBQUU7WUFDaEUsT0FBTyxPQUFPLENBQUM7U0FDaEI7UUFFRCxnRkFBZ0Y7UUFDaEYsNEVBQTRFO1FBQzVFLGdCQUFnQjtRQUNoQixJQUFNLElBQUksR0FBaUUsT0FBTyxDQUFDLElBQUksQ0FDbkYsTUFBTSxDQUFDLFVBQUMsS0FBcUIsSUFBSyxPQUFBLEtBQUssWUFBWSxZQUFZLEVBQTdCLENBQTZCLENBQUMsQ0FBQyxDQUFDO1FBRXRFLGlDQUFpQztRQUNqQyxRQUFRLE9BQU8sQ0FBQyxPQUFPLElBQUksTUFBTSxFQUFFO1lBQ2pDLEtBQUssTUFBTTtnQkFDVCw0RUFBNEU7Z0JBQzVFLDRFQUE0RTtnQkFDNUUsK0VBQStFO2dCQUMvRSw2RUFBNkU7Z0JBQzdFLGtCQUFrQjtnQkFDbEIsUUFBUSxHQUFHLENBQUMsWUFBWSxFQUFFO29CQUN4QixLQUFLLGFBQWE7d0JBQ2hCLE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsVUFBQyxHQUFzQjs0QkFDMUMsNENBQTRDOzRCQUM1QyxJQUFJLEdBQUcsQ0FBQyxJQUFJLEtBQUssSUFBSSxJQUFJLENBQUMsQ0FBQyxHQUFHLENBQUMsSUFBSSxZQUFZLFdBQVcsQ0FBQyxFQUFFO2dDQUMzRCxNQUFNLElBQUksS0FBSyxDQUFDLGlDQUFpQyxDQUFDLENBQUM7NkJBQ3BEOzRCQUNELE9BQU8sR0FBRyxDQUFDLElBQUksQ0FBQzt3QkFDbEIsQ0FBQyxDQUFDLENBQUMsQ0FBQztvQkFDTixLQUFLLE1BQU07d0JBQ1QsT0FBTyxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxVQUFDLEdBQXNCOzRCQUMxQyxvQ0FBb0M7NEJBQ3BDLElBQUksR0FBRyxDQUFDLElBQUksS0FBSyxJQUFJLElBQUksQ0FBQyxDQUFDLEdBQUcsQ0FBQyxJQUFJLFlBQVksSUFBSSxDQUFDLEVBQUU7Z0NBQ3BELE1BQU0sSUFBSSxLQUFLLENBQUMseUJBQXlCLENBQUMsQ0FBQzs2QkFDNUM7NEJBQ0QsT0FBTyxHQUFHLENBQUMsSUFBSSxDQUFDO3dCQUNsQixDQUFDLENBQUMsQ0FBQyxDQUFDO29CQUNOLEtBQUssTUFBTTt3QkFDVCxPQUFPLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLFVBQUMsR0FBc0I7NEJBQzFDLHNDQUFzQzs0QkFDdEMsSUFBSSxHQUFHLENBQUMsSUFBSSxLQUFLLElBQUksSUFBSSxPQUFPLEdBQUcsQ0FBQyxJQUFJLEtBQUssUUFBUSxFQUFFO2dDQUNyRCxNQUFNLElBQUksS0FBSyxDQUFDLDJCQUEyQixDQUFDLENBQUM7NkJBQzlDOzRCQUNELE9BQU8sR0FBRyxDQUFDLElBQUksQ0FBQzt3QkFDbEIsQ0FBQyxDQUFDLENBQUMsQ0FBQztvQkFDTixLQUFLLE1BQU0sQ0FBQztvQkFDWjt3QkFDRSx1RUFBdUU7d0JBQ3ZFLE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsVUFBQyxHQUFzQixJQUFLLE9BQUEsR0FBRyxDQUFDLElBQUksRUFBUixDQUFRLENBQUMsQ0FBQyxDQUFDO2lCQUMvRDtZQUNILEtBQUssVUFBVTtnQkFDYiw0REFBNEQ7Z0JBQzVELE9BQU8sSUFBSSxDQUFDO1lBQ2Q7Z0JBQ0Usc0RBQXNEO2dCQUN0RCxNQUFNLElBQUksS0FBSyxDQUFDLHlDQUF1QyxPQUFPLENBQUMsT0FBTyxNQUFHLENBQUMsQ0FBQztTQUM5RTtJQUNILENBQUM7SUE0TUQ7Ozs7T0FJRztJQUNILDJCQUFNLEdBQU4sVUFBUSxHQUFXLEVBQUUsT0FPZjtRQVBlLHdCQUFBLEVBQUEsWUFPZjtRQUNKLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBTSxRQUFRLEVBQUUsR0FBRyxFQUFFLE9BQWMsQ0FBQyxDQUFDO0lBQzFELENBQUM7SUE0TUQ7Ozs7T0FJRztJQUNILHdCQUFHLEdBQUgsVUFBSSxHQUFXLEVBQUUsT0FPWDtRQVBXLHdCQUFBLEVBQUEsWUFPWDtRQUNKLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBTSxLQUFLLEVBQUUsR0FBRyxFQUFFLE9BQWMsQ0FBQyxDQUFDO0lBQ3ZELENBQUM7SUE0TUQ7Ozs7T0FJRztJQUNILHlCQUFJLEdBQUosVUFBSyxHQUFXLEVBQUUsT0FPWjtRQVBZLHdCQUFBLEVBQUEsWUFPWjtRQUNKLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBTSxNQUFNLEVBQUUsR0FBRyxFQUFFLE9BQWMsQ0FBQyxDQUFDO0lBQ3hELENBQUM7SUFnQkQ7Ozs7Ozs7T0FPRztJQUNILDBCQUFLLEdBQUwsVUFBUyxHQUFXLEVBQUUsYUFBcUI7UUFDekMsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFNLE9BQU8sRUFBRSxHQUFHLEVBQUU7WUFDckMsTUFBTSxFQUFFLElBQUksVUFBVSxFQUFFLENBQUMsTUFBTSxDQUFDLGFBQWEsRUFBRSxnQkFBZ0IsQ0FBQztZQUNoRSxPQUFPLEVBQUUsTUFBTTtZQUNmLFlBQVksRUFBRSxNQUFNO1NBQ3JCLENBQUMsQ0FBQztJQUNMLENBQUM7SUEyTUQ7Ozs7T0FJRztJQUNILDRCQUFPLEdBQVAsVUFBUSxHQUFXLEVBQUUsT0FPZjtRQVBlLHdCQUFBLEVBQUEsWUFPZjtRQUNKLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBTSxTQUFTLEVBQUUsR0FBRyxFQUFFLE9BQWMsQ0FBQyxDQUFDO0lBQzNELENBQUM7SUEyTUQ7Ozs7T0FJRztJQUNILDBCQUFLLEdBQUwsVUFBTSxHQUFXLEVBQUUsSUFBYyxFQUFFLE9BTzdCO1FBUDZCLHdCQUFBLEVBQUEsWUFPN0I7UUFDSixPQUFPLElBQUksQ0FBQyxPQUFPLENBQU0sT0FBTyxFQUFFLEdBQUcsRUFBRSxPQUFPLENBQUMsT0FBTyxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUM7SUFDakUsQ0FBQztJQTJNRDs7OztPQUlHO0lBQ0gseUJBQUksR0FBSixVQUFLLEdBQVcsRUFBRSxJQUFjLEVBQUUsT0FPNUI7UUFQNEIsd0JBQUEsRUFBQSxZQU81QjtRQUNKLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBTSxNQUFNLEVBQUUsR0FBRyxFQUFFLE9BQU8sQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQztJQUNoRSxDQUFDO0lBdU1EOzs7O09BSUc7SUFDSCx3QkFBRyxHQUFILFVBQUksR0FBVyxFQUFFLElBQWMsRUFBRSxPQU8zQjtRQVAyQix3QkFBQSxFQUFBLFlBTzNCO1FBQ0osT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFNLEtBQUssRUFBRSxHQUFHLEVBQUUsT0FBTyxDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDO0lBQy9ELENBQUM7SUE1NERVLFVBQVU7UUFEdEIsVUFBVSxFQUFFO2lEQUVrQixXQUFXO09BRDdCLFVBQVUsQ0E2NER0QjtJQUFELGlCQUFDO0NBQUEsQUE3NERELElBNjREQztTQTc0RFksVUFBVSIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQGxpY2Vuc2VcbiAqIENvcHlyaWdodCBHb29nbGUgSW5jLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqIFVzZSBvZiB0aGlzIHNvdXJjZSBjb2RlIGlzIGdvdmVybmVkIGJ5IGFuIE1JVC1zdHlsZSBsaWNlbnNlIHRoYXQgY2FuIGJlXG4gKiBmb3VuZCBpbiB0aGUgTElDRU5TRSBmaWxlIGF0IGh0dHBzOi8vYW5ndWxhci5pby9saWNlbnNlXG4gKi9cblxuaW1wb3J0IHtJbmplY3RhYmxlfSBmcm9tICdAYW5ndWxhci9jb3JlJztcbmltcG9ydCB7T2JzZXJ2YWJsZSwgb2YgfSBmcm9tICdyeGpzJztcbmltcG9ydCB7Y29uY2F0TWFwLCBmaWx0ZXIsIG1hcH0gZnJvbSAncnhqcy9vcGVyYXRvcnMnO1xuXG5pbXBvcnQge0h0dHBIYW5kbGVyfSBmcm9tICcuL2JhY2tlbmQnO1xuaW1wb3J0IHtIdHRwSGVhZGVyc30gZnJvbSAnLi9oZWFkZXJzJztcbmltcG9ydCB7SHR0cFBhcmFtcywgSHR0cFBhcmFtc09wdGlvbnN9IGZyb20gJy4vcGFyYW1zJztcbmltcG9ydCB7SHR0cFJlcXVlc3R9IGZyb20gJy4vcmVxdWVzdCc7XG5pbXBvcnQge0h0dHBFdmVudCwgSHR0cFJlc3BvbnNlfSBmcm9tICcuL3Jlc3BvbnNlJztcblxuXG4vKipcbiAqIENvbnN0cnVjdCBhbiBpbnN0YW5jZSBvZiBgSHR0cFJlcXVlc3RPcHRpb25zPFQ+YCBmcm9tIGEgc291cmNlIGBIdHRwTWV0aG9kT3B0aW9uc2AgYW5kXG4gKiB0aGUgZ2l2ZW4gYGJvZHlgLiBCYXNpY2FsbHksIHRoaXMgY2xvbmVzIHRoZSBvYmplY3QgYW5kIGFkZHMgdGhlIGJvZHkuXG4gKi9cbmZ1bmN0aW9uIGFkZEJvZHk8VD4oXG4gICAgb3B0aW9uczoge1xuICAgICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICAgIG9ic2VydmU/OiBIdHRwT2JzZXJ2ZSxcbiAgICAgIHBhcmFtcz86IEh0dHBQYXJhbXMgfCB7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgICByZXNwb25zZVR5cGU/OiAnYXJyYXlidWZmZXInIHwgJ2Jsb2InIHwgJ2pzb24nIHwgJ3RleHQnLFxuICAgICAgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgICB9LFxuICAgIGJvZHk6IFQgfCBudWxsKTogYW55IHtcbiAgcmV0dXJuIHtcbiAgICBib2R5LFxuICAgIGhlYWRlcnM6IG9wdGlvbnMuaGVhZGVycyxcbiAgICBvYnNlcnZlOiBvcHRpb25zLm9ic2VydmUsXG4gICAgcGFyYW1zOiBvcHRpb25zLnBhcmFtcyxcbiAgICByZXBvcnRQcm9ncmVzczogb3B0aW9ucy5yZXBvcnRQcm9ncmVzcyxcbiAgICByZXNwb25zZVR5cGU6IG9wdGlvbnMucmVzcG9uc2VUeXBlLFxuICAgIHdpdGhDcmVkZW50aWFsczogb3B0aW9ucy53aXRoQ3JlZGVudGlhbHMsXG4gIH07XG59XG5cbmV4cG9ydCB0eXBlIEh0dHBPYnNlcnZlID0gJ2JvZHknIHwgJ2V2ZW50cycgfCAncmVzcG9uc2UnO1xuXG4vKipcbiAqIFBlcmZvcm0gSFRUUCByZXF1ZXN0cy5cbiAqXG4gKiBgSHR0cENsaWVudGAgaXMgYXZhaWxhYmxlIGFzIGFuIGluamVjdGFibGUgY2xhc3MsIHdpdGggbWV0aG9kcyB0byBwZXJmb3JtIEhUVFAgcmVxdWVzdHMuXG4gKiBFYWNoIHJlcXVlc3QgbWV0aG9kIGhhcyBtdWx0aXBsZSBzaWduYXR1cmVzLCBhbmQgdGhlIHJldHVybiB0eXBlIHZhcmllcyBhY2NvcmRpbmcgdG8gd2hpY2hcbiAqIHNpZ25hdHVyZSBpcyBjYWxsZWQgKG1haW5seSB0aGUgdmFsdWVzIG9mIGBvYnNlcnZlYCBhbmQgYHJlc3BvbnNlVHlwZWApLlxuICpcbiAqIEBwdWJsaWNBcGlcbiAqL1xuQEluamVjdGFibGUoKVxuZXhwb3J0IGNsYXNzIEh0dHBDbGllbnQge1xuICBjb25zdHJ1Y3Rvcihwcml2YXRlIGhhbmRsZXI6IEh0dHBIYW5kbGVyKSB7fVxuXG4gIC8qKlxuICAgKiBTZW5kIHRoZSBnaXZlbiBgSHR0cFJlcXVlc3RgIGFuZCByZXR1cm4gYSBzdHJlYW0gb2YgYEh0dHBFdmVudHNgLlxuICAgKi9cbiAgcmVxdWVzdDxSPihyZXE6IEh0dHBSZXF1ZXN0PGFueT4pOiBPYnNlcnZhYmxlPEh0dHBFdmVudDxSPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBhbiBgQXJyYXlCdWZmZXJgIGFuZCByZXR1cm5zIGl0LlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYm9keSBhcyBhbiBgQXJyYXlCdWZmZXJgLlxuICAgKi9cbiAgcmVxdWVzdChtZXRob2Q6IHN0cmluZywgdXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBib2R5PzogYW55LFxuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVyc3x7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU/OiAnYm9keScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ2FycmF5YnVmZmVyJywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8QXJyYXlCdWZmZXI+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgYSBgQmxvYmAgYW5kIHJldHVybnMgaXQuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBib2R5IGFzIGEgYEJsb2JgLlxuICAgKi9cbiAgcmVxdWVzdChtZXRob2Q6IHN0cmluZywgdXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBib2R5PzogYW55LFxuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVyc3x7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU/OiAnYm9keScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ2Jsb2InLCB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxCbG9iPjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIHRleHQgYW5kIHJldHVybnMgaXQuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBib2R5IGFzIGEgYHN0cmluZ2AuXG4gICAqL1xuICByZXF1ZXN0KG1ldGhvZDogc3RyaW5nLCB1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGJvZHk/OiBhbnksXG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzfHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86ICdib2R5JyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAndGV4dCcsIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPHN0cmluZz47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBhbiBgQXJyYXlCdWZmZXJgIGFuZCByZXR1cm5zIHRoZSBmdWxsIGV2ZW50IHN0cmVhbS5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgYWxsIGBIdHRwRXZlbnRgcyBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYEFycmF5QnVmZmVyYC5cbiAgICovXG4gIHJlcXVlc3QobWV0aG9kOiBzdHJpbmcsIHVybDogc3RyaW5nLCBvcHRpb25zOiB7XG4gICAgYm9keT86IGFueSxcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnN8e1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAnZXZlbnRzJywgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ2FycmF5YnVmZmVyJywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cEV2ZW50PEFycmF5QnVmZmVyPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBhbiBgQmxvYmAgYW5kIHJldHVybnMgdGhlIGZ1bGwgZXZlbnQgc3RyZWFtLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiBhbGwgYEh0dHBFdmVudGBzIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgQmxvYmAuXG4gICAqL1xuICByZXF1ZXN0KG1ldGhvZDogc3RyaW5nLCB1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGJvZHk/OiBhbnksXG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzfHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ2V2ZW50cycsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ2Jsb2InLCB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwRXZlbnQ8QmxvYj4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgdGV4dCBhbmQgcmV0dXJucyB0aGUgZnVsbCBldmVudCBzdHJlYW0uXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIGFsbCBgSHR0cEV2ZW50YHMgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBzdHJpbmdgLlxuICAgKi9cbiAgcmVxdWVzdChtZXRob2Q6IHN0cmluZywgdXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBib2R5PzogYW55LFxuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVyc3x7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdldmVudHMnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU6ICd0ZXh0Jywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cEV2ZW50PHN0cmluZz4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgSlNPTiBhbmQgcmV0dXJucyB0aGUgZnVsbCBldmVudCBzdHJlYW0uXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIGFsbCBgSHR0cEV2ZW50YHMgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBPYmplY3RgLlxuICAgKi9cbiAgcmVxdWVzdChtZXRob2Q6IHN0cmluZywgdXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBib2R5PzogYW55LFxuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVyc3x7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICBvYnNlcnZlOiAnZXZlbnRzJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXNwb25zZVR5cGU/OiAnanNvbicsXG4gICAgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cEV2ZW50PGFueT4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgSlNPTiBhbmQgcmV0dXJucyB0aGUgZnVsbCBldmVudCBzdHJlYW0uXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIGFsbCBgSHR0cEV2ZW50YHMgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBSYC5cbiAgICovXG4gIHJlcXVlc3Q8Uj4obWV0aG9kOiBzdHJpbmcsIHVybDogc3RyaW5nLCBvcHRpb25zOiB7XG4gICAgYm9keT86IGFueSxcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnN8e1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgb2JzZXJ2ZTogJ2V2ZW50cycsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVzcG9uc2VUeXBlPzogJ2pzb24nLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBFdmVudDxSPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBhbiBgQXJyYXlCdWZmZXJgIGFuZCByZXR1cm5zIHRoZSBmdWxsIHJlc3BvbnNlLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYEh0dHBSZXNwb25zZWAgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBBcnJheUJ1ZmZlcmAuXG4gICAqL1xuICByZXF1ZXN0KG1ldGhvZDogc3RyaW5nLCB1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGJvZHk/OiBhbnksXG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzfHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ3Jlc3BvbnNlJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAnYXJyYXlidWZmZXInLCB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwUmVzcG9uc2U8QXJyYXlCdWZmZXI+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGEgYEJsb2JgIGFuZCByZXR1cm5zIHRoZSBmdWxsIHJlc3BvbnNlLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYEh0dHBSZXNwb25zZWAgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBCbG9iYC5cbiAgICovXG4gIHJlcXVlc3QobWV0aG9kOiBzdHJpbmcsIHVybDogc3RyaW5nLCBvcHRpb25zOiB7XG4gICAgYm9keT86IGFueSxcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnN8e1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAncmVzcG9uc2UnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU6ICdibG9iJywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cFJlc3BvbnNlPEJsb2I+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIHRleHQgYW5kIHJldHVybnMgdGhlIGZ1bGwgcmVzcG9uc2UuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBgSHR0cFJlc3BvbnNlYCBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYHN0cmluZ2AuXG4gICAqL1xuICByZXF1ZXN0KG1ldGhvZDogc3RyaW5nLCB1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGJvZHk/OiBhbnksXG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzfHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ3Jlc3BvbnNlJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAndGV4dCcsIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBSZXNwb25zZTxzdHJpbmc+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIEpTT04gYW5kIHJldHVybnMgdGhlIGZ1bGwgcmVzcG9uc2UuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBgSHR0cFJlc3BvbnNlYCBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYE9iamVjdGAuXG4gICAqL1xuICByZXF1ZXN0KG1ldGhvZDogc3RyaW5nLCB1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGJvZHk/OiBhbnksXG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzfHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIG9ic2VydmU6ICdyZXNwb25zZScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVzcG9uc2VUeXBlPzogJ2pzb24nLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBSZXNwb25zZTxPYmplY3Q+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIEpTT04gYW5kIHJldHVybnMgdGhlIGZ1bGwgcmVzcG9uc2UuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBgSHR0cFJlc3BvbnNlYCBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYFJgLlxuICAgKi9cbiAgcmVxdWVzdDxSPihtZXRob2Q6IHN0cmluZywgdXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBib2R5PzogYW55LFxuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVyc3x7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICBvYnNlcnZlOiAncmVzcG9uc2UnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlc3BvbnNlVHlwZT86ICdqc29uJyxcbiAgICB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwUmVzcG9uc2U8Uj4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgSlNPTiBhbmQgcmV0dXJucyBpdC5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgdGhlIGBIdHRwUmVzcG9uc2VgIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgT2JqZWN0YC5cbiAgICovXG4gIHJlcXVlc3QobWV0aG9kOiBzdHJpbmcsIHVybDogc3RyaW5nLCBvcHRpb25zPzoge1xuICAgIGJvZHk/OiBhbnksXG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzfHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86ICdib2R5JyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXNwb25zZVR5cGU/OiAnanNvbicsXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPE9iamVjdD47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBKU09OIGFuZCByZXR1cm5zIGl0LlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYEh0dHBSZXNwb25zZWAgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBSYC5cbiAgICovXG4gIHJlcXVlc3Q8Uj4obWV0aG9kOiBzdHJpbmcsIHVybDogc3RyaW5nLCBvcHRpb25zPzoge1xuICAgIGJvZHk/OiBhbnksXG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzfHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86ICdib2R5JyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXNwb25zZVR5cGU/OiAnanNvbicsXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPFI+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSByZXF1ZXN0IGluIGEgbWFubmVyIHdoZXJlIHJlc3BvbnNlIHR5cGUgYW5kIHJlcXVlc3RlZCBgT2JzZXJ2YWJsZWAgYXJlIG5vdCBrbm93blxuICAgKiBzdGF0aWNhbGx5LlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB3aGF0ZXZlciB3YXMgcmVxdWVzdGVkLCB0eXBlZCB0byBgYW55YC5cbiAgICovXG4gIHJlcXVlc3QobWV0aG9kOiBzdHJpbmcsIHVybDogc3RyaW5nLCBvcHRpb25zPzoge1xuICAgIGJvZHk/OiBhbnksXG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzfHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86IEh0dHBPYnNlcnZlLFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU/OiAnYXJyYXlidWZmZXInfCdibG9iJ3wnanNvbid8J3RleHQnLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPGFueT47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdHMgYW4gYE9ic2VydmFibGVgIGZvciBhIHBhcnRpY3VsYXIgSFRUUCByZXF1ZXN0IHRoYXQsIHdoZW4gc3Vic2NyaWJlZCxcbiAgICogZmlyZXMgdGhlIHJlcXVlc3QgdGhyb3VnaCB0aGUgY2hhaW4gb2YgcmVnaXN0ZXJlZCBpbnRlcmNlcHRvcnMgYW5kIG9uIHRvIHRoZVxuICAgKiBzZXJ2ZXIuXG4gICAqXG4gICAqIFRoaXMgbWV0aG9kIGNhbiBiZSBjYWxsZWQgaW4gb25lIG9mIHR3byB3YXlzLiBFaXRoZXIgYW4gYEh0dHBSZXF1ZXN0YFxuICAgKiBpbnN0YW5jZSBjYW4gYmUgcGFzc2VkIGRpcmVjdGx5IGFzIHRoZSBvbmx5IHBhcmFtZXRlciwgb3IgYSBtZXRob2QgY2FuIGJlXG4gICAqIHBhc3NlZCBhcyB0aGUgZmlyc3QgcGFyYW1ldGVyLCBhIHN0cmluZyBVUkwgYXMgdGhlIHNlY29uZCwgYW5kIGFuXG4gICAqIG9wdGlvbnMgaGFzaCBhcyB0aGUgdGhpcmQuXG4gICAqXG4gICAqIElmIGEgYEh0dHBSZXF1ZXN0YCBvYmplY3QgaXMgcGFzc2VkIGRpcmVjdGx5LCBhbiBgT2JzZXJ2YWJsZWAgb2YgdGhlXG4gICAqIHJhdyBgSHR0cEV2ZW50YCBzdHJlYW0gd2lsbCBiZSByZXR1cm5lZC5cbiAgICpcbiAgICogSWYgYSByZXF1ZXN0IGlzIGluc3RlYWQgYnVpbHQgYnkgcHJvdmlkaW5nIGEgVVJMLCB0aGUgb3B0aW9ucyBvYmplY3RcbiAgICogZGV0ZXJtaW5lcyB0aGUgcmV0dXJuIHR5cGUgb2YgYHJlcXVlc3QoKWAuIEluIGFkZGl0aW9uIHRvIGNvbmZpZ3VyaW5nXG4gICAqIHJlcXVlc3QgcGFyYW1ldGVycyBzdWNoIGFzIHRoZSBvdXRnb2luZyBoZWFkZXJzIGFuZC9vciB0aGUgYm9keSwgdGhlIG9wdGlvbnNcbiAgICogaGFzaCBzcGVjaWZpZXMgdHdvIGtleSBwaWVjZXMgb2YgaW5mb3JtYXRpb24gYWJvdXQgdGhlIHJlcXVlc3Q6IHRoZVxuICAgKiBgcmVzcG9uc2VUeXBlYCBhbmQgd2hhdCB0byBgb2JzZXJ2ZWAuXG4gICAqXG4gICAqIFRoZSBgcmVzcG9uc2VUeXBlYCB2YWx1ZSBkZXRlcm1pbmVzIGhvdyBhIHN1Y2Nlc3NmdWwgcmVzcG9uc2UgYm9keSB3aWxsIGJlXG4gICAqIHBhcnNlZC4gSWYgYHJlc3BvbnNlVHlwZWAgaXMgdGhlIGRlZmF1bHQgYGpzb25gLCBhIHR5cGUgaW50ZXJmYWNlIGZvciB0aGVcbiAgICogcmVzdWx0aW5nIG9iamVjdCBtYXkgYmUgcGFzc2VkIGFzIGEgdHlwZSBwYXJhbWV0ZXIgdG8gYHJlcXVlc3QoKWAuXG4gICAqXG4gICAqIFRoZSBgb2JzZXJ2ZWAgdmFsdWUgZGV0ZXJtaW5lcyB0aGUgcmV0dXJuIHR5cGUgb2YgYHJlcXVlc3QoKWAsIGJhc2VkIG9uIHdoYXRcbiAgICogdGhlIGNvbnN1bWVyIGlzIGludGVyZXN0ZWQgaW4gb2JzZXJ2aW5nLiBBIHZhbHVlIG9mIGBldmVudHNgIHdpbGwgcmV0dXJuIGFuXG4gICAqIGBPYnNlcnZhYmxlPEh0dHBFdmVudD5gIHJlcHJlc2VudGluZyB0aGUgcmF3IGBIdHRwRXZlbnRgIHN0cmVhbSxcbiAgICogaW5jbHVkaW5nIHByb2dyZXNzIGV2ZW50cyBieSBkZWZhdWx0LiBBIHZhbHVlIG9mIGByZXNwb25zZWAgd2lsbCByZXR1cm4gYW5cbiAgICogYE9ic2VydmFibGU8SHR0cFJlc3BvbnNlPFQ+PmAgd2hlcmUgdGhlIGBUYCBwYXJhbWV0ZXIgb2YgYEh0dHBSZXNwb25zZWBcbiAgICogZGVwZW5kcyBvbiB0aGUgYHJlc3BvbnNlVHlwZWAgYW5kIGFueSBvcHRpb25hbGx5IHByb3ZpZGVkIHR5cGUgcGFyYW1ldGVyLlxuICAgKiBBIHZhbHVlIG9mIGBib2R5YCB3aWxsIHJldHVybiBhbiBgT2JzZXJ2YWJsZTxUPmAgd2l0aCB0aGUgc2FtZSBgVGAgYm9keSB0eXBlLlxuICAgKi9cbiAgcmVxdWVzdChmaXJzdDogc3RyaW5nfEh0dHBSZXF1ZXN0PGFueT4sIHVybD86IHN0cmluZywgb3B0aW9uczoge1xuICAgIGJvZHk/OiBhbnksXG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzfHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86IEh0dHBPYnNlcnZlLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU/OiAnYXJyYXlidWZmZXInfCdibG9iJ3wnanNvbid8J3RleHQnLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0gPSB7fSk6IE9ic2VydmFibGU8YW55PiB7XG4gICAgbGV0IHJlcTogSHR0cFJlcXVlc3Q8YW55PjtcbiAgICAvLyBGaXJzdGx5LCBjaGVjayB3aGV0aGVyIHRoZSBwcmltYXJ5IGFyZ3VtZW50IGlzIGFuIGluc3RhbmNlIG9mIGBIdHRwUmVxdWVzdGAuXG4gICAgaWYgKGZpcnN0IGluc3RhbmNlb2YgSHR0cFJlcXVlc3QpIHtcbiAgICAgIC8vIEl0IGlzLiBUaGUgb3RoZXIgYXJndW1lbnRzIG11c3QgYmUgdW5kZWZpbmVkIChwZXIgdGhlIHNpZ25hdHVyZXMpIGFuZCBjYW4gYmVcbiAgICAgIC8vIGlnbm9yZWQuXG4gICAgICByZXEgPSBmaXJzdCBhcyBIdHRwUmVxdWVzdDxhbnk+O1xuICAgIH0gZWxzZSB7XG4gICAgICAvLyBJdCdzIGEgc3RyaW5nLCBzbyBpdCByZXByZXNlbnRzIGEgVVJMLiBDb25zdHJ1Y3QgYSByZXF1ZXN0IGJhc2VkIG9uIGl0LFxuICAgICAgLy8gYW5kIGluY29ycG9yYXRlIHRoZSByZW1haW5pbmcgYXJndW1lbnRzIChhc3N1bWluZyBHRVQgdW5sZXNzIGEgbWV0aG9kIGlzXG4gICAgICAvLyBwcm92aWRlZC5cblxuICAgICAgLy8gRmlndXJlIG91dCB0aGUgaGVhZGVycy5cbiAgICAgIGxldCBoZWFkZXJzOiBIdHRwSGVhZGVyc3x1bmRlZmluZWQgPSB1bmRlZmluZWQ7XG4gICAgICBpZiAob3B0aW9ucy5oZWFkZXJzIGluc3RhbmNlb2YgSHR0cEhlYWRlcnMpIHtcbiAgICAgICAgaGVhZGVycyA9IG9wdGlvbnMuaGVhZGVycztcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIGhlYWRlcnMgPSBuZXcgSHR0cEhlYWRlcnMob3B0aW9ucy5oZWFkZXJzKTtcbiAgICAgIH1cblxuICAgICAgLy8gU29ydCBvdXQgcGFyYW1ldGVycy5cbiAgICAgIGxldCBwYXJhbXM6IEh0dHBQYXJhbXN8dW5kZWZpbmVkID0gdW5kZWZpbmVkO1xuICAgICAgaWYgKCEhb3B0aW9ucy5wYXJhbXMpIHtcbiAgICAgICAgaWYgKG9wdGlvbnMucGFyYW1zIGluc3RhbmNlb2YgSHR0cFBhcmFtcykge1xuICAgICAgICAgIHBhcmFtcyA9IG9wdGlvbnMucGFyYW1zO1xuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIHBhcmFtcyA9IG5ldyBIdHRwUGFyYW1zKHsgZnJvbU9iamVjdDogb3B0aW9ucy5wYXJhbXMgfSBhcyBIdHRwUGFyYW1zT3B0aW9ucyk7XG4gICAgICAgIH1cbiAgICAgIH1cblxuICAgICAgLy8gQ29uc3RydWN0IHRoZSByZXF1ZXN0LlxuICAgICAgcmVxID0gbmV3IEh0dHBSZXF1ZXN0KGZpcnN0LCB1cmwgISwgKG9wdGlvbnMuYm9keSAhPT0gdW5kZWZpbmVkID8gb3B0aW9ucy5ib2R5IDogbnVsbCksIHtcbiAgICAgICAgaGVhZGVycyxcbiAgICAgICAgcGFyYW1zLFxuICAgICAgICByZXBvcnRQcm9ncmVzczogb3B0aW9ucy5yZXBvcnRQcm9ncmVzcyxcbiAgICAgICAgLy8gQnkgZGVmYXVsdCwgSlNPTiBpcyBhc3N1bWVkIHRvIGJlIHJldHVybmVkIGZvciBhbGwgY2FsbHMuXG4gICAgICAgIHJlc3BvbnNlVHlwZTogb3B0aW9ucy5yZXNwb25zZVR5cGUgfHwgJ2pzb24nLFxuICAgICAgICB3aXRoQ3JlZGVudGlhbHM6IG9wdGlvbnMud2l0aENyZWRlbnRpYWxzLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgLy8gU3RhcnQgd2l0aCBhbiBPYnNlcnZhYmxlLm9mKCkgdGhlIGluaXRpYWwgcmVxdWVzdCwgYW5kIHJ1biB0aGUgaGFuZGxlciAod2hpY2hcbiAgICAvLyBpbmNsdWRlcyBhbGwgaW50ZXJjZXB0b3JzKSBpbnNpZGUgYSBjb25jYXRNYXAoKS4gVGhpcyB3YXksIHRoZSBoYW5kbGVyIHJ1bnNcbiAgICAvLyBpbnNpZGUgYW4gT2JzZXJ2YWJsZSBjaGFpbiwgd2hpY2ggY2F1c2VzIGludGVyY2VwdG9ycyB0byBiZSByZS1ydW4gb24gZXZlcnlcbiAgICAvLyBzdWJzY3JpcHRpb24gKHRoaXMgYWxzbyBtYWtlcyByZXRyaWVzIHJlLXJ1biB0aGUgaGFuZGxlciwgaW5jbHVkaW5nIGludGVyY2VwdG9ycykuXG4gICAgY29uc3QgZXZlbnRzJDogT2JzZXJ2YWJsZTxIdHRwRXZlbnQ8YW55Pj4gPVxuICAgICAgICBvZiAocmVxKS5waXBlKGNvbmNhdE1hcCgocmVxOiBIdHRwUmVxdWVzdDxhbnk+KSA9PiB0aGlzLmhhbmRsZXIuaGFuZGxlKHJlcSkpKTtcblxuICAgIC8vIElmIGNvbWluZyB2aWEgdGhlIEFQSSBzaWduYXR1cmUgd2hpY2ggYWNjZXB0cyBhIHByZXZpb3VzbHkgY29uc3RydWN0ZWQgSHR0cFJlcXVlc3QsXG4gICAgLy8gdGhlIG9ubHkgb3B0aW9uIGlzIHRvIGdldCB0aGUgZXZlbnQgc3RyZWFtLiBPdGhlcndpc2UsIHJldHVybiB0aGUgZXZlbnQgc3RyZWFtIGlmXG4gICAgLy8gdGhhdCBpcyB3aGF0IHdhcyByZXF1ZXN0ZWQuXG4gICAgaWYgKGZpcnN0IGluc3RhbmNlb2YgSHR0cFJlcXVlc3QgfHwgb3B0aW9ucy5vYnNlcnZlID09PSAnZXZlbnRzJykge1xuICAgICAgcmV0dXJuIGV2ZW50cyQ7XG4gICAgfVxuXG4gICAgLy8gVGhlIHJlcXVlc3RlZCBzdHJlYW0gY29udGFpbnMgZWl0aGVyIHRoZSBmdWxsIHJlc3BvbnNlIG9yIHRoZSBib2R5LiBJbiBlaXRoZXJcbiAgICAvLyBjYXNlLCB0aGUgZmlyc3Qgc3RlcCBpcyB0byBmaWx0ZXIgdGhlIGV2ZW50IHN0cmVhbSB0byBleHRyYWN0IGEgc3RyZWFtIG9mXG4gICAgLy8gcmVzcG9uc2VzKHMpLlxuICAgIGNvbnN0IHJlcyQ6IE9ic2VydmFibGU8SHR0cFJlc3BvbnNlPGFueT4+ID0gPE9ic2VydmFibGU8SHR0cFJlc3BvbnNlPGFueT4+PmV2ZW50cyQucGlwZShcbiAgICAgICAgZmlsdGVyKChldmVudDogSHR0cEV2ZW50PGFueT4pID0+IGV2ZW50IGluc3RhbmNlb2YgSHR0cFJlc3BvbnNlKSk7XG5cbiAgICAvLyBEZWNpZGUgd2hpY2ggc3RyZWFtIHRvIHJldHVybi5cbiAgICBzd2l0Y2ggKG9wdGlvbnMub2JzZXJ2ZSB8fCAnYm9keScpIHtcbiAgICAgIGNhc2UgJ2JvZHknOlxuICAgICAgICAvLyBUaGUgcmVxdWVzdGVkIHN0cmVhbSBpcyB0aGUgYm9keS4gTWFwIHRoZSByZXNwb25zZSBzdHJlYW0gdG8gdGhlIHJlc3BvbnNlXG4gICAgICAgIC8vIGJvZHkuIFRoaXMgY291bGQgYmUgZG9uZSBtb3JlIHNpbXBseSwgYnV0IGEgbWlzYmVoYXZpbmcgaW50ZXJjZXB0b3IgbWlnaHRcbiAgICAgICAgLy8gdHJhbnNmb3JtIHRoZSByZXNwb25zZSBib2R5IGludG8gYSBkaWZmZXJlbnQgZm9ybWF0IGFuZCBpZ25vcmUgdGhlIHJlcXVlc3RlZFxuICAgICAgICAvLyByZXNwb25zZVR5cGUuIEd1YXJkIGFnYWluc3QgdGhpcyBieSB2YWxpZGF0aW5nIHRoYXQgdGhlIHJlc3BvbnNlIGlzIG9mIHRoZVxuICAgICAgICAvLyByZXF1ZXN0ZWQgdHlwZS5cbiAgICAgICAgc3dpdGNoIChyZXEucmVzcG9uc2VUeXBlKSB7XG4gICAgICAgICAgY2FzZSAnYXJyYXlidWZmZXInOlxuICAgICAgICAgICAgcmV0dXJuIHJlcyQucGlwZShtYXAoKHJlczogSHR0cFJlc3BvbnNlPGFueT4pID0+IHtcbiAgICAgICAgICAgICAgLy8gVmFsaWRhdGUgdGhhdCB0aGUgYm9keSBpcyBhbiBBcnJheUJ1ZmZlci5cbiAgICAgICAgICAgICAgaWYgKHJlcy5ib2R5ICE9PSBudWxsICYmICEocmVzLmJvZHkgaW5zdGFuY2VvZiBBcnJheUJ1ZmZlcikpIHtcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1Jlc3BvbnNlIGlzIG5vdCBhbiBBcnJheUJ1ZmZlci4nKTtcbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICByZXR1cm4gcmVzLmJvZHk7XG4gICAgICAgICAgICB9KSk7XG4gICAgICAgICAgY2FzZSAnYmxvYic6XG4gICAgICAgICAgICByZXR1cm4gcmVzJC5waXBlKG1hcCgocmVzOiBIdHRwUmVzcG9uc2U8YW55PikgPT4ge1xuICAgICAgICAgICAgICAvLyBWYWxpZGF0ZSB0aGF0IHRoZSBib2R5IGlzIGEgQmxvYi5cbiAgICAgICAgICAgICAgaWYgKHJlcy5ib2R5ICE9PSBudWxsICYmICEocmVzLmJvZHkgaW5zdGFuY2VvZiBCbG9iKSkge1xuICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignUmVzcG9uc2UgaXMgbm90IGEgQmxvYi4nKTtcbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICByZXR1cm4gcmVzLmJvZHk7XG4gICAgICAgICAgICB9KSk7XG4gICAgICAgICAgY2FzZSAndGV4dCc6XG4gICAgICAgICAgICByZXR1cm4gcmVzJC5waXBlKG1hcCgocmVzOiBIdHRwUmVzcG9uc2U8YW55PikgPT4ge1xuICAgICAgICAgICAgICAvLyBWYWxpZGF0ZSB0aGF0IHRoZSBib2R5IGlzIGEgc3RyaW5nLlxuICAgICAgICAgICAgICBpZiAocmVzLmJvZHkgIT09IG51bGwgJiYgdHlwZW9mIHJlcy5ib2R5ICE9PSAnc3RyaW5nJykge1xuICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignUmVzcG9uc2UgaXMgbm90IGEgc3RyaW5nLicpO1xuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgIHJldHVybiByZXMuYm9keTtcbiAgICAgICAgICAgIH0pKTtcbiAgICAgICAgICBjYXNlICdqc29uJzpcbiAgICAgICAgICBkZWZhdWx0OlxuICAgICAgICAgICAgLy8gTm8gdmFsaWRhdGlvbiBuZWVkZWQgZm9yIEpTT04gcmVzcG9uc2VzLCBhcyB0aGV5IGNhbiBiZSBvZiBhbnkgdHlwZS5cbiAgICAgICAgICAgIHJldHVybiByZXMkLnBpcGUobWFwKChyZXM6IEh0dHBSZXNwb25zZTxhbnk+KSA9PiByZXMuYm9keSkpO1xuICAgICAgICB9XG4gICAgICBjYXNlICdyZXNwb25zZSc6XG4gICAgICAgIC8vIFRoZSByZXNwb25zZSBzdHJlYW0gd2FzIHJlcXVlc3RlZCBkaXJlY3RseSwgc28gcmV0dXJuIGl0LlxuICAgICAgICByZXR1cm4gcmVzJDtcbiAgICAgIGRlZmF1bHQ6XG4gICAgICAgIC8vIEd1YXJkIGFnYWluc3QgbmV3IGZ1dHVyZSBvYnNlcnZlIHR5cGVzIGJlaW5nIGFkZGVkLlxuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFVucmVhY2hhYmxlOiB1bmhhbmRsZWQgb2JzZXJ2ZSB0eXBlICR7b3B0aW9ucy5vYnNlcnZlfX1gKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgREVMRVRFIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBhbiBgQXJyYXlCdWZmZXJgIGFuZCByZXR1cm5zIGl0LlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYm9keSBhcyBhbiBgQXJyYXlCdWZmZXJgLlxuICAgKi9cbiAgZGVsZXRlICh1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86ICdib2R5JyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAnYXJyYXlidWZmZXInLCB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxBcnJheUJ1ZmZlcj47XG5cblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgREVMRVRFIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBhIGBCbG9iYCBhbmQgcmV0dXJucyBpdC5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgdGhlIGJvZHkgYXMgYSBgQmxvYmAuXG4gICAqL1xuICBkZWxldGUgKHVybDogc3RyaW5nLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlPzogJ2JvZHknLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU6ICdibG9iJywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8QmxvYj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIERFTEVURSByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgdGV4dCBhbmQgcmV0dXJucyBpdC5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgdGhlIGJvZHkgYXMgYSBgc3RyaW5nYC5cbiAgICovXG4gIGRlbGV0ZSAodXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU/OiAnYm9keScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ3RleHQnLCB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxzdHJpbmc+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBERUxFVEUgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGFuIGBBcnJheUJ1ZmZlcmAgYW5kIHJldHVybnMgdGhlIGZ1bGwgZXZlbnQgc3RyZWFtLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiBhbGwgYEh0dHBFdmVudGBzIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgQXJyYXlCdWZmZXJgLlxuICAgKi9cbiAgZGVsZXRlICh1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ2V2ZW50cycsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ2FycmF5YnVmZmVyJywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cEV2ZW50PEFycmF5QnVmZmVyPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIERFTEVURSByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgYSBgQmxvYmAgYW5kIHJldHVybnMgdGhlIGZ1bGwgZXZlbnQgc3RyZWFtLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiBhbGwgYEh0dHBFdmVudGBzIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgQmxvYmAuXG4gICAqL1xuICBkZWxldGUgKHVybDogc3RyaW5nLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAnZXZlbnRzJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAnYmxvYicsIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBFdmVudDxCbG9iPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIERFTEVURSByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgdGV4dCBhbmQgcmV0dXJucyB0aGUgZnVsbCBldmVudCBzdHJlYW0uXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIGFsbCBgSHR0cEV2ZW50YHMgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBzdHJpbmdgLlxuICAgKi9cbiAgZGVsZXRlICh1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ2V2ZW50cycsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ3RleHQnLCB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwRXZlbnQ8c3RyaW5nPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIERFTEVURSByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgSlNPTiBhbmQgcmV0dXJucyB0aGUgZnVsbCBldmVudCBzdHJlYW0uXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIGFsbCBgSHR0cEV2ZW50YHMgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBPYmplY3RgLlxuICAgKi9cbiAgZGVsZXRlICh1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ2V2ZW50cycsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZT86ICdqc29uJyxcbiAgICB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwRXZlbnQ8T2JqZWN0Pj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIERFTEVURSByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgSlNPTiBhbmQgcmV0dXJucyB0aGUgZnVsbCBldmVudCBzdHJlYW0uXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIGFsbCBgSHR0cEV2ZW50YHMgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBUYC5cbiAgICovXG4gIGRlbGV0ZTxUPih1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ2V2ZW50cycsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZT86ICdqc29uJyxcbiAgICB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwRXZlbnQ8VD4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBERUxFVEUgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGFuIGBBcnJheUJ1ZmZlcmAgYW5kIHJldHVybnMgdGhlIGZ1bGwgcmVzcG9uc2UuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBgSHR0cFJlc3BvbnNlYCBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYEFycmF5QnVmZmVyYC5cbiAgICovXG4gIGRlbGV0ZSAodXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdyZXNwb25zZScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ2FycmF5YnVmZmVyJywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cFJlc3BvbnNlPEFycmF5QnVmZmVyPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIERFTEVURSByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgYSBgQmxvYmAgYW5kIHJldHVybnMgdGhlIGZ1bGwgcmVzcG9uc2UuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBgSHR0cFJlc3BvbnNlYCBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYEJsb2JgLlxuICAgKi9cbiAgZGVsZXRlICh1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ3Jlc3BvbnNlJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAnYmxvYicsIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBSZXNwb25zZTxCbG9iPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIERFTEVURSByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgdGV4dCBhbmQgcmV0dXJucyB0aGUgZnVsbCByZXNwb25zZS5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgdGhlIGBIdHRwUmVzcG9uc2VgIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgc3RyaW5nYC5cbiAgICovXG4gIGRlbGV0ZSAodXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdyZXNwb25zZScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ3RleHQnLCB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwUmVzcG9uc2U8c3RyaW5nPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIERFTEVURSByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgSlNPTiBhbmQgcmV0dXJucyB0aGUgZnVsbCByZXNwb25zZS5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgdGhlIGBIdHRwUmVzcG9uc2VgIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgT2JqZWN0YC5cbiAgICovXG4gIGRlbGV0ZSAodXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdyZXNwb25zZScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZT86ICdqc29uJyxcbiAgICB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwUmVzcG9uc2U8T2JqZWN0Pj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIERFTEVURSByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgSlNPTiBhbmQgcmV0dXJucyB0aGUgZnVsbCByZXNwb25zZS5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgdGhlIGBIdHRwUmVzcG9uc2VgIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgVGAuXG4gICAqL1xuICBkZWxldGU8VD4odXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdyZXNwb25zZScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZT86ICdqc29uJyxcbiAgICB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwUmVzcG9uc2U8VD4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBERUxFVEUgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIEpTT04gYW5kIHJldHVybnMgaXQuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBib2R5IGFzIGFuIGBPYmplY3RgLlxuICAgKi9cbiAgZGVsZXRlICh1cmw6IHN0cmluZywgb3B0aW9ucz86IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU/OiAnYm9keScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZT86ICdqc29uJyxcbiAgICB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxPYmplY3Q+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBERUxFVEUgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIEpTT04gYW5kIHJldHVybnMgaXQuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBib2R5IGFzIHR5cGUgYFRgLlxuICAgKi9cbiAgZGVsZXRlPFQ+KHVybDogc3RyaW5nLCBvcHRpb25zPzoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86ICdib2R5JyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlPzogJ2pzb24nLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPFQ+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3RzIGFuIGBPYnNlcnZhYmxlYCB3aGljaCwgd2hlbiBzdWJzY3JpYmVkLCB3aWxsIGNhdXNlIHRoZSBjb25maWd1cmVkXG4gICAqIERFTEVURSByZXF1ZXN0IHRvIGJlIGV4ZWN1dGVkIG9uIHRoZSBzZXJ2ZXIuIFNlZSB0aGUgaW5kaXZpZHVhbCBvdmVybG9hZHMgZm9yXG4gICAqIGRldGFpbHMgb2YgYGRlbGV0ZSgpYCdzIHJldHVybiB0eXBlIGJhc2VkIG9uIHRoZSBwcm92aWRlZCBvcHRpb25zLlxuICAgKi9cbiAgZGVsZXRlICh1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86IEh0dHBPYnNlcnZlLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU/OiAnYXJyYXlidWZmZXInfCdibG9iJ3wnanNvbid8J3RleHQnLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0gPSB7fSk6IE9ic2VydmFibGU8YW55PiB7XG4gICAgcmV0dXJuIHRoaXMucmVxdWVzdDxhbnk+KCdERUxFVEUnLCB1cmwsIG9wdGlvbnMgYXMgYW55KTtcbiAgfVxuXG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIEdFVCByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgYW4gYEFycmF5QnVmZmVyYCBhbmQgcmV0dXJucyBpdC5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgdGhlIGJvZHkgYXMgYW4gYEFycmF5QnVmZmVyYC5cbiAgICovXG4gIGdldCh1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86ICdib2R5JyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAnYXJyYXlidWZmZXInLCB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxBcnJheUJ1ZmZlcj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIEdFVCByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgYSBgQmxvYmAgYW5kIHJldHVybnMgaXQuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBib2R5IGFzIGEgYEJsb2JgLlxuICAgKi9cbiAgZ2V0KHVybDogc3RyaW5nLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlPzogJ2JvZHknLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU6ICdibG9iJywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8QmxvYj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIEdFVCByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgdGV4dCBhbmQgcmV0dXJucyBpdC5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgdGhlIGJvZHkgYXMgYSBgc3RyaW5nYC5cbiAgICovXG4gIGdldCh1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86ICdib2R5JyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAndGV4dCcsIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPHN0cmluZz47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIEdFVCByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgYW4gYEFycmF5QnVmZmVyYCBhbmQgcmV0dXJucyB0aGUgZnVsbCBldmVudCBzdHJlYW0uXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIGFsbCBgSHR0cEV2ZW50YHMgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBBcnJheUJ1ZmZlcmAuXG4gICAqL1xuICBnZXQodXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdldmVudHMnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU6ICdhcnJheWJ1ZmZlcicsIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBFdmVudDxBcnJheUJ1ZmZlcj4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBHRVQgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGEgYEJsb2JgIGFuZCByZXR1cm5zIHRoZSBmdWxsIGV2ZW50IHN0cmVhbS5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgYWxsIGBIdHRwRXZlbnRgcyBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYEJsb2JgLlxuICAgKi9cbiAgZ2V0KHVybDogc3RyaW5nLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAnZXZlbnRzJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAnYmxvYicsIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBFdmVudDxCbG9iPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIEdFVCByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgdGV4dCBhbmQgcmV0dXJucyB0aGUgZnVsbCBldmVudCBzdHJlYW0uXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIGFsbCBgSHR0cEV2ZW50YHMgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBzdHJpbmdgLlxuICAgKi9cbiAgZ2V0KHVybDogc3RyaW5nLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAnZXZlbnRzJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAndGV4dCcsIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBFdmVudDxzdHJpbmc+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgR0VUIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBKU09OIGFuZCByZXR1cm5zIHRoZSBmdWxsIGV2ZW50IHN0cmVhbS5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgYWxsIGBIdHRwRXZlbnRgcyBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYE9iamVjdGAuXG4gICAqL1xuICBnZXQodXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdldmVudHMnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU/OiAnanNvbicsXG4gICAgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cEV2ZW50PE9iamVjdD4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBHRVQgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIEpTT04gYW5kIHJldHVybnMgdGhlIGZ1bGwgZXZlbnQgc3RyZWFtLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiBhbGwgYEh0dHBFdmVudGBzIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgVGAuXG4gICAqL1xuICBnZXQ8VD4odXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdldmVudHMnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU/OiAnanNvbicsXG4gICAgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cEV2ZW50PFQ+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgR0VUIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBhbiBgQXJyYXlCdWZmZXJgIGFuZCByZXR1cm5zIHRoZSBmdWxsIHJlc3BvbnNlLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYEh0dHBSZXNwb25zZWAgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBBcnJheUJ1ZmZlcmAuXG4gICAqL1xuICBnZXQodXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdyZXNwb25zZScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ2FycmF5YnVmZmVyJywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cFJlc3BvbnNlPEFycmF5QnVmZmVyPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIEdFVCByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgYSBgQmxvYmAgYW5kIHJldHVybnMgdGhlIGZ1bGwgcmVzcG9uc2UuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBgSHR0cFJlc3BvbnNlYCBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYEJsb2JgLlxuICAgKi9cbiAgZ2V0KHVybDogc3RyaW5nLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAncmVzcG9uc2UnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU6ICdibG9iJywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cFJlc3BvbnNlPEJsb2I+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgR0VUIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyB0ZXh0IGFuZCByZXR1cm5zIHRoZSBmdWxsIHJlc3BvbnNlLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYEh0dHBSZXNwb25zZWAgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBzdHJpbmdgLlxuICAgKi9cbiAgZ2V0KHVybDogc3RyaW5nLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAncmVzcG9uc2UnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU6ICd0ZXh0Jywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cFJlc3BvbnNlPHN0cmluZz4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBHRVQgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIEpTT04gYW5kIHJldHVybnMgdGhlIGZ1bGwgcmVzcG9uc2UuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBgSHR0cFJlc3BvbnNlYCBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYE9iamVjdGAuXG4gICAqL1xuICBnZXQodXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdyZXNwb25zZScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZT86ICdqc29uJyxcbiAgICB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwUmVzcG9uc2U8T2JqZWN0Pj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIEdFVCByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgSlNPTiBhbmQgcmV0dXJucyB0aGUgZnVsbCByZXNwb25zZS5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgdGhlIGBIdHRwUmVzcG9uc2VgIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgVGAuXG4gICAqL1xuICBnZXQ8VD4odXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdyZXNwb25zZScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZT86ICdqc29uJyxcbiAgICB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwUmVzcG9uc2U8VD4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBHRVQgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIEpTT04gYW5kIHJldHVybnMgaXQuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBib2R5IGFzIGFuIGBPYmplY3RgLlxuICAgKi9cbiAgZ2V0KHVybDogc3RyaW5nLCBvcHRpb25zPzoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86ICdib2R5JyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlPzogJ2pzb24nLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPE9iamVjdD47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIEdFVCByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgSlNPTiBhbmQgcmV0dXJucyBpdC5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgdGhlIGJvZHkgYXMgdHlwZSBgVGAuXG4gICAqL1xuICBnZXQ8VD4odXJsOiBzdHJpbmcsIG9wdGlvbnM/OiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlPzogJ2JvZHknLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU/OiAnanNvbicsXG4gICAgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8VD47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdHMgYW4gYE9ic2VydmFibGVgIHdoaWNoLCB3aGVuIHN1YnNjcmliZWQsIHdpbGwgY2F1c2UgdGhlIGNvbmZpZ3VyZWRcbiAgICogR0VUIHJlcXVlc3QgdG8gYmUgZXhlY3V0ZWQgb24gdGhlIHNlcnZlci4gU2VlIHRoZSBpbmRpdmlkdWFsIG92ZXJsb2FkcyBmb3JcbiAgICogZGV0YWlscyBvZiBgZ2V0KClgJ3MgcmV0dXJuIHR5cGUgYmFzZWQgb24gdGhlIHByb3ZpZGVkIG9wdGlvbnMuXG4gICAqL1xuICBnZXQodXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU/OiBIdHRwT2JzZXJ2ZSxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlPzogJ2FycmF5YnVmZmVyJ3wnYmxvYid8J2pzb24nfCd0ZXh0JyxcbiAgICB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9ID0ge30pOiBPYnNlcnZhYmxlPGFueT4ge1xuICAgIHJldHVybiB0aGlzLnJlcXVlc3Q8YW55PignR0VUJywgdXJsLCBvcHRpb25zIGFzIGFueSk7XG4gIH1cblxuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBIRUFEIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBhbiBgQXJyYXlCdWZmZXJgIGFuZCByZXR1cm5zIGl0LlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYm9keSBhcyBhbiBgQXJyYXlCdWZmZXJgLlxuICAgKi9cbiAgaGVhZCh1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86ICdib2R5JyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAnYXJyYXlidWZmZXInLCB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuXG4gICAgLyoqXG4gICAqIENvbnN0cnVjdCBhIEhFQUQgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGEgYEJsb2JgIGFuZCByZXR1cm5zIGl0LlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYm9keSBhcyBhIGBCbG9iYC5cbiAgICovXG4gIH0pOiBPYnNlcnZhYmxlPEFycmF5QnVmZmVyPjtcbiAgaGVhZCh1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86ICdib2R5JyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAnYmxvYicsIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEJsb2I+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBIRUFEIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyB0ZXh0IGFuZCByZXR1cm5zIGl0LlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYm9keSBhcyBhIGBzdHJpbmdgLlxuICAgKi9cbiAgaGVhZCh1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86ICdib2R5JyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAndGV4dCcsIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPHN0cmluZz47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIEhFQUQgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGFuIGBBcnJheUJ1ZmZlcmAgYW5kIHJldHVybnMgdGhlIGZ1bGwgZXZlbnQgc3RyZWFtLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiBhbGwgYEh0dHBFdmVudGBzIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgQXJyYXlCdWZmZXJgLlxuICAgKi9cbiAgaGVhZCh1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ2V2ZW50cycsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ2FycmF5YnVmZmVyJywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cEV2ZW50PEFycmF5QnVmZmVyPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIEhFQUQgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGEgYEJsb2JgIGFuZCByZXR1cm5zIHRoZSBmdWxsIGV2ZW50IHN0cmVhbS5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgYWxsIGBIdHRwRXZlbnRgcyBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYEJsb2JgLlxuICAgKi9cbiAgaGVhZCh1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ2V2ZW50cycsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ2Jsb2InLCB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwRXZlbnQ8QmxvYj4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBIRUFEIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyB0ZXh0IGFuZCByZXR1cm5zIHRoZSBmdWxsIGV2ZW50IHN0cmVhbS5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgYWxsIGBIdHRwRXZlbnRgcyBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYHN0cmluZ2AuXG4gICAqL1xuICBoZWFkKHVybDogc3RyaW5nLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAnZXZlbnRzJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAndGV4dCcsIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBFdmVudDxzdHJpbmc+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgSEVBRCByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgSlNPTiBhbmQgcmV0dXJucyB0aGUgZnVsbCBldmVudCBzdHJlYW0uXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIGFsbCBgSHR0cEV2ZW50YHMgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBPYmplY3RgLlxuICAgKi9cbiAgaGVhZCh1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ2V2ZW50cycsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZT86ICdqc29uJyxcbiAgICB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwRXZlbnQ8T2JqZWN0Pj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIEhFQUQgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIEpTT04gYW5kIHJldHVybnMgdGhlIGZ1bGwgZXZlbnQgc3RyZWFtLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiBhbGwgYEh0dHBFdmVudGBzIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgVGAuXG4gICAqL1xuICBoZWFkPFQ+KHVybDogc3RyaW5nLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAnZXZlbnRzJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlPzogJ2pzb24nLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBFdmVudDxUPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIEhFQUQgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGFuIGBBcnJheUJ1ZmZlcmAgYW5kIHJldHVybnMgdGhlIGZ1bGwgcmVzcG9uc2UuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBgSHR0cFJlc3BvbnNlYCBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYEFycmF5QnVmZmVyYC5cbiAgICovXG4gIGhlYWQodXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdyZXNwb25zZScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ2FycmF5YnVmZmVyJywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cFJlc3BvbnNlPEFycmF5QnVmZmVyPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIEhFQUQgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGEgYEJsb2JgIGFuZCByZXR1cm5zIHRoZSBmdWxsIHJlc3BvbnNlLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYEh0dHBSZXNwb25zZWAgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBCbG9iYC5cbiAgICovXG4gIGhlYWQodXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdyZXNwb25zZScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ2Jsb2InLCB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwUmVzcG9uc2U8QmxvYj4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBIRUFEIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyB0ZXh0IGFuZCByZXR1cm5zIHRoZSBmdWxsIHJlc3BvbnNlLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYEh0dHBSZXNwb25zZWAgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBzdHJpbmdgLlxuICAgKi9cbiAgaGVhZCh1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ3Jlc3BvbnNlJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAndGV4dCcsIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBSZXNwb25zZTxzdHJpbmc+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgSEVBRCByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgSlNPTiBhbmQgcmV0dXJucyB0aGUgZnVsbCByZXNwb25zZS5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgdGhlIGBIdHRwUmVzcG9uc2VgIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgT2JqZWN0YC5cbiAgICovXG4gIGhlYWQodXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdyZXNwb25zZScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZT86ICdqc29uJyxcbiAgICB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwUmVzcG9uc2U8T2JqZWN0Pj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIEhFQUQgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIEpTT04gYW5kIHJldHVybnMgdGhlIGZ1bGwgcmVzcG9uc2UuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBgSHR0cFJlc3BvbnNlYCBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYFRgLlxuICAgKi9cbiAgaGVhZDxUPih1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ3Jlc3BvbnNlJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlPzogJ2pzb24nLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBSZXNwb25zZTxUPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIEhFQUQgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIEpTT04gYW5kIHJldHVybnMgaXQuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBib2R5IGFzIGFuIGBPYmplY3RgLlxuICAgKi9cbiAgaGVhZCh1cmw6IHN0cmluZywgb3B0aW9ucz86IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU/OiAnYm9keScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZT86ICdqc29uJyxcbiAgICB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxPYmplY3Q+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBIRUFEIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBKU09OIGFuZCByZXR1cm5zIGl0LlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYm9keSBhcyB0eXBlIGBUYC5cbiAgICovXG4gIGhlYWQ8VD4odXJsOiBzdHJpbmcsIG9wdGlvbnM/OiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlPzogJ2JvZHknLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU/OiAnanNvbicsXG4gICAgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8VD47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdHMgYW4gYE9ic2VydmFibGVgIHdoaWNoLCB3aGVuIHN1YnNjcmliZWQsIHdpbGwgY2F1c2UgdGhlIGNvbmZpZ3VyZWRcbiAgICogSEVBRCByZXF1ZXN0IHRvIGJlIGV4ZWN1dGVkIG9uIHRoZSBzZXJ2ZXIuIFNlZSB0aGUgaW5kaXZpZHVhbCBvdmVybG9hZHMgZm9yXG4gICAqIGRldGFpbHMgb2YgYGhlYWQoKWAncyByZXR1cm4gdHlwZSBiYXNlZCBvbiB0aGUgcHJvdmlkZWQgb3B0aW9ucy5cbiAgICovXG4gIGhlYWQodXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU/OiBIdHRwT2JzZXJ2ZSxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlPzogJ2FycmF5YnVmZmVyJ3wnYmxvYid8J2pzb24nfCd0ZXh0JyxcbiAgICB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9ID0ge30pOiBPYnNlcnZhYmxlPGFueT4ge1xuICAgIHJldHVybiB0aGlzLnJlcXVlc3Q8YW55PignSEVBRCcsIHVybCwgb3B0aW9ucyBhcyBhbnkpO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIEpTT05QIHJlcXVlc3QgZm9yIHRoZSBnaXZlbiBVUkwgYW5kIG5hbWUgb2YgdGhlIGNhbGxiYWNrIHBhcmFtZXRlci5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgdGhlIHJlc3BvbnNlIG9iamVjdCBhcyBhbiBgT2JqZWN0YFxuICAgKi9cbiAganNvbnAodXJsOiBzdHJpbmcsIGNhbGxiYWNrUGFyYW06IHN0cmluZyk6IE9ic2VydmFibGU8T2JqZWN0PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgSlNPTlAgcmVxdWVzdCBmb3IgdGhlIGdpdmVuIFVSTCBhbmQgbmFtZSBvZiB0aGUgY2FsbGJhY2sgcGFyYW1ldGVyLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgcmVzcG9uc2Ugb2JqZWN0IGFzIHR5cGUgYFRgLlxuICAgKi9cbiAganNvbnA8VD4odXJsOiBzdHJpbmcsIGNhbGxiYWNrUGFyYW06IHN0cmluZyk6IE9ic2VydmFibGU8VD47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdHMgYW4gYE9ic2VydmFibGVgIHdoaWNoLCB3aGVuIHN1YnNjcmliZWQsIHdpbGwgY2F1c2UgYSByZXF1ZXN0XG4gICAqIHdpdGggdGhlIHNwZWNpYWwgbWV0aG9kIGBKU09OUGAgdG8gYmUgZGlzcGF0Y2hlZCB2aWEgdGhlIGludGVyY2VwdG9yIHBpcGVsaW5lLlxuICAgKlxuICAgKiBBIHN1aXRhYmxlIGludGVyY2VwdG9yIG11c3QgYmUgaW5zdGFsbGVkIChlLmcuIHZpYSB0aGUgYEh0dHBDbGllbnRKc29ucE1vZHVsZWApLlxuICAgKiBJZiBubyBzdWNoIGludGVyY2VwdG9yIGlzIHJlYWNoZWQsIHRoZW4gdGhlIGBKU09OUGAgcmVxdWVzdCB3aWxsIGxpa2VseSBiZVxuICAgKiByZWplY3RlZCBieSB0aGUgY29uZmlndXJlZCBiYWNrZW5kLlxuICAgKi9cbiAganNvbnA8VD4odXJsOiBzdHJpbmcsIGNhbGxiYWNrUGFyYW06IHN0cmluZyk6IE9ic2VydmFibGU8VD4ge1xuICAgIHJldHVybiB0aGlzLnJlcXVlc3Q8YW55PignSlNPTlAnLCB1cmwsIHtcbiAgICAgIHBhcmFtczogbmV3IEh0dHBQYXJhbXMoKS5hcHBlbmQoY2FsbGJhY2tQYXJhbSwgJ0pTT05QX0NBTExCQUNLJyksXG4gICAgICBvYnNlcnZlOiAnYm9keScsXG4gICAgICByZXNwb25zZVR5cGU6ICdqc29uJyxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBNYWtlIGFuIE9QVElPTlMgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGFuIGBBcnJheUJ1ZmZlcmAgYW5kIHJldHVybnMgaXQuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBib2R5IGFzIGFuIGBBcnJheUJ1ZmZlcmAuXG4gICAqL1xuICBvcHRpb25zKHVybDogc3RyaW5nLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlPzogJ2JvZHknLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU6ICdhcnJheWJ1ZmZlcicsIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEFycmF5QnVmZmVyPjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGFuIE9QVElPTlMgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGEgYEJsb2JgIGFuZCByZXR1cm5zIGl0LlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYm9keSBhcyBhIGBCbG9iYC5cbiAgICovXG4gIG9wdGlvbnModXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU/OiAnYm9keScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ2Jsb2InLCB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxCbG9iPjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGFuIE9QVElPTlMgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIHRleHQgYW5kIHJldHVybnMgaXQuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBib2R5IGFzIGEgYHN0cmluZ2AuXG4gICAqL1xuICBvcHRpb25zKHVybDogc3RyaW5nLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlPzogJ2JvZHknLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU6ICd0ZXh0Jywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8c3RyaW5nPjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGFuIE9QVElPTlMgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGFuIGBBcnJheUJ1ZmZlcmAgYW5kIHJldHVybnMgdGhlIGZ1bGwgZXZlbnQgc3RyZWFtLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiBhbGwgYEh0dHBFdmVudGBzIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgQXJyYXlCdWZmZXJgLlxuICAgKi9cbiAgb3B0aW9ucyh1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ2V2ZW50cycsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ2FycmF5YnVmZmVyJywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cEV2ZW50PEFycmF5QnVmZmVyPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhbiBPUFRJT05TIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBhIGBCbG9iYCBhbmQgcmV0dXJucyB0aGUgZnVsbCBldmVudCBzdHJlYW0uXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIGFsbCBgSHR0cEV2ZW50YHMgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBCbG9iYC5cbiAgICovXG4gIG9wdGlvbnModXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdldmVudHMnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU6ICdibG9iJywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cEV2ZW50PEJsb2I+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGFuIE9QVElPTlMgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIHRleHQgYW5kIHJldHVybnMgdGhlIGZ1bGwgZXZlbnQgc3RyZWFtLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiBhbGwgYEh0dHBFdmVudGBzIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgc3RyaW5nYC5cbiAgICovXG4gIG9wdGlvbnModXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdldmVudHMnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU6ICd0ZXh0Jywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cEV2ZW50PHN0cmluZz4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYW4gT1BUSU9OUyByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgSlNPTiBhbmQgcmV0dXJucyB0aGUgZnVsbCBldmVudCBzdHJlYW0uXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIGFsbCBgSHR0cEV2ZW50YHMgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBPYmplY3RgLlxuICAgKi9cbiAgb3B0aW9ucyh1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ2V2ZW50cycsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZT86ICdqc29uJyxcbiAgICB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwRXZlbnQ8T2JqZWN0Pj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhbiBPUFRJT05TIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBKU09OIGFuZCByZXR1cm5zIHRoZSBmdWxsIGV2ZW50IHN0cmVhbS5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgYWxsIGBIdHRwRXZlbnRgcyBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYFRgLlxuICAgKi9cbiAgb3B0aW9uczxUPih1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ2V2ZW50cycsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZT86ICdqc29uJyxcbiAgICB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwRXZlbnQ8VD4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYW4gT1BUSU9OUyByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgYW4gYEFycmF5QnVmZmVyYCBhbmQgcmV0dXJucyB0aGUgZnVsbCByZXNwb25zZS5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgdGhlIGBIdHRwUmVzcG9uc2VgIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgQXJyYXlCdWZmZXJgLlxuICAgKi9cbiAgb3B0aW9ucyh1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ3Jlc3BvbnNlJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAnYXJyYXlidWZmZXInLCB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwUmVzcG9uc2U8QXJyYXlCdWZmZXI+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGFuIE9QVElPTlMgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGEgYEJsb2JgIGFuZCByZXR1cm5zIHRoZSBmdWxsIHJlc3BvbnNlLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYEh0dHBSZXNwb25zZWAgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBCbG9iYC5cbiAgICovXG4gIG9wdGlvbnModXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdyZXNwb25zZScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ2Jsb2InLCB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwUmVzcG9uc2U8QmxvYj4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYW4gT1BUSU9OUyByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgdGV4dCBhbmQgcmV0dXJucyB0aGUgZnVsbCByZXNwb25zZS5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgdGhlIGBIdHRwUmVzcG9uc2VgIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgc3RyaW5nYC5cbiAgICovXG4gIG9wdGlvbnModXJsOiBzdHJpbmcsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdyZXNwb25zZScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ3RleHQnLCB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwUmVzcG9uc2U8c3RyaW5nPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhbiBPUFRJT05TIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBKU09OIGFuZCByZXR1cm5zIHRoZSBmdWxsIHJlc3BvbnNlLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYEh0dHBSZXNwb25zZWAgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBPYmplY3RgLlxuICAgKi9cbiAgb3B0aW9ucyh1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ3Jlc3BvbnNlJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlPzogJ2pzb24nLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBSZXNwb25zZTxPYmplY3Q+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGFuIE9QVElPTlMgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIEpTT04gYW5kIHJldHVybnMgdGhlIGZ1bGwgcmVzcG9uc2UuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBgSHR0cFJlc3BvbnNlYCBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYFRgLlxuICAgKi9cbiAgb3B0aW9uczxUPih1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ3Jlc3BvbnNlJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlPzogJ2pzb24nLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBSZXNwb25zZTxUPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhbiBPUFRJT05TIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBKU09OIGFuZCByZXR1cm5zIGl0LlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYm9keSBhcyBhbiBgT2JqZWN0YC5cbiAgICovXG4gIG9wdGlvbnModXJsOiBzdHJpbmcsIG9wdGlvbnM/OiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlPzogJ2JvZHknLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU/OiAnanNvbicsXG4gICAgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8T2JqZWN0PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGFuIE9QVElPTlMgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIEpTT04gYW5kIHJldHVybnMgaXQuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBib2R5IGFzIHR5cGUgYFRgLlxuICAgKi9cbiAgb3B0aW9uczxUPih1cmw6IHN0cmluZywgb3B0aW9ucz86IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU/OiAnYm9keScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZT86ICdqc29uJyxcbiAgICB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxUPjtcblxuICAvKipcbiAgICogQ29uc3RydWN0cyBhbiBgT2JzZXJ2YWJsZWAgd2hpY2gsIHdoZW4gc3Vic2NyaWJlZCwgd2lsbCBjYXVzZSB0aGUgY29uZmlndXJlZFxuICAgKiBPUFRJT05TIHJlcXVlc3QgdG8gYmUgZXhlY3V0ZWQgb24gdGhlIHNlcnZlci4gU2VlIHRoZSBpbmRpdmlkdWFsIG92ZXJsb2FkcyBmb3JcbiAgICogZGV0YWlscyBvZiBgb3B0aW9ucygpYCdzIHJldHVybiB0eXBlIGJhc2VkIG9uIHRoZSBwcm92aWRlZCBvcHRpb25zLlxuICAgKi9cbiAgb3B0aW9ucyh1cmw6IHN0cmluZywgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86IEh0dHBPYnNlcnZlLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU/OiAnYXJyYXlidWZmZXInfCdibG9iJ3wnanNvbid8J3RleHQnLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0gPSB7fSk6IE9ic2VydmFibGU8YW55PiB7XG4gICAgcmV0dXJuIHRoaXMucmVxdWVzdDxhbnk+KCdPUFRJT05TJywgdXJsLCBvcHRpb25zIGFzIGFueSk7XG4gIH1cblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgUEFUQ0ggcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGFuIGBBcnJheUJ1ZmZlcmAgYW5kIHJldHVybnMgaXQuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBib2R5IGFzIGFuIGBBcnJheUJ1ZmZlcmAuXG4gICAqL1xuICBwYXRjaCh1cmw6IHN0cmluZywgYm9keTogYW55fG51bGwsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU/OiAnYm9keScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ2FycmF5YnVmZmVyJywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8QXJyYXlCdWZmZXI+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBQQVRDSCByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgYSBgQmxvYmAgYW5kIHJldHVybnMgaXQuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBib2R5IGFzIGEgYEJsb2JgLlxuICAgKi9cbiAgcGF0Y2godXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlPzogJ2JvZHknLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU6ICdibG9iJywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8QmxvYj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIFBBVENIIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyB0ZXh0IGFuZCByZXR1cm5zIGl0LlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYm9keSBhcyBhIGBzdHJpbmdgLlxuICAgKi9cbiAgcGF0Y2godXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlPzogJ2JvZHknLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU6ICd0ZXh0Jywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8c3RyaW5nPjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgUEFUQ0ggcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGFuIGBBcnJheUJ1ZmZlcmAgYW5kIHJldHVybnMgdGhlIGZ1bGwgZXZlbnQgc3RyZWFtLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiBhbGwgYEh0dHBFdmVudGBzIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgQXJyYXlCdWZmZXJgLlxuICAgKi9cbiAgcGF0Y2godXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAnZXZlbnRzJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAnYXJyYXlidWZmZXInLCB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwRXZlbnQ8QXJyYXlCdWZmZXI+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgUEFUQ0ggcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGEgYEJsb2JgIGFuZCByZXR1cm5zIHRoZSBmdWxsIGV2ZW50IHN0cmVhbS5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgYWxsIGBIdHRwRXZlbnRgcyBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYEJsb2JgLlxuICAgKi9cbiAgcGF0Y2godXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAnZXZlbnRzJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAnYmxvYicsIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBFdmVudDxCbG9iPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIFBBVENIIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyB0ZXh0IGFuZCByZXR1cm5zIHRoZSBmdWxsIGV2ZW50IHN0cmVhbS5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgYWxsIGBIdHRwRXZlbnRgcyBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYHN0cmluZ2AuXG4gICAqL1xuICBwYXRjaCh1cmw6IHN0cmluZywgYm9keTogYW55fG51bGwsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdldmVudHMnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU6ICd0ZXh0Jywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cEV2ZW50PHN0cmluZz4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBQQVRDSCByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgSlNPTiBhbmQgcmV0dXJucyB0aGUgZnVsbCBldmVudCBzdHJlYW0uXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIGFsbCBgSHR0cEV2ZW50YHMgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBPYmplY3RgLlxuICAgKi9cbiAgcGF0Y2godXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAnZXZlbnRzJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlPzogJ2pzb24nLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBFdmVudDxPYmplY3Q+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgUEFUQ0ggcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIEpTT04gYW5kIHJldHVybnMgdGhlIGZ1bGwgZXZlbnQgc3RyZWFtLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiBhbGwgYEh0dHBFdmVudGBzIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgVGAuXG4gICAqL1xuICBwYXRjaDxUPih1cmw6IHN0cmluZywgYm9keTogYW55fG51bGwsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdldmVudHMnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU/OiAnanNvbicsXG4gICAgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cEV2ZW50PFQ+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgUEFUQ0ggcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGFuIGBBcnJheUJ1ZmZlcmAgYW5kIHJldHVybnMgdGhlIGZ1bGwgcmVzcG9uc2UuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBgSHR0cFJlc3BvbnNlYCBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYEFycmF5QnVmZmVyYC5cbiAgICovXG4gIHBhdGNoKHVybDogc3RyaW5nLCBib2R5OiBhbnl8bnVsbCwgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ3Jlc3BvbnNlJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAnYXJyYXlidWZmZXInLCB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwUmVzcG9uc2U8QXJyYXlCdWZmZXI+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgUEFUQ0ggcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGEgYEJsb2JgIGFuZCByZXR1cm5zIHRoZSBmdWxsIHJlc3BvbnNlLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYEh0dHBSZXNwb25zZWAgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBCbG9iYC5cbiAgICovXG4gIHBhdGNoKHVybDogc3RyaW5nLCBib2R5OiBhbnl8bnVsbCwgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ3Jlc3BvbnNlJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAnYmxvYicsIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBSZXNwb25zZTxCbG9iPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIFBBVENIIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyB0ZXh0IGFuZCByZXR1cm5zIHRoZSBmdWxsIHJlc3BvbnNlLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYEh0dHBSZXNwb25zZWAgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBzdHJpbmdgLlxuICAgKi9cbiAgcGF0Y2godXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAncmVzcG9uc2UnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU6ICd0ZXh0Jywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cFJlc3BvbnNlPHN0cmluZz4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBQQVRDSCByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgSlNPTiBhbmQgcmV0dXJucyB0aGUgZnVsbCByZXNwb25zZS5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgdGhlIGBIdHRwUmVzcG9uc2VgIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgT2JqZWN0YC5cbiAgICovXG4gIHBhdGNoKHVybDogc3RyaW5nLCBib2R5OiBhbnl8bnVsbCwgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ3Jlc3BvbnNlJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlPzogJ2pzb24nLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBSZXNwb25zZTxPYmplY3Q+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgUEFUQ0ggcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIEpTT04gYW5kIHJldHVybnMgdGhlIGZ1bGwgcmVzcG9uc2UuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBgSHR0cFJlc3BvbnNlYCBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYFRgLlxuICAgKi9cbiAgcGF0Y2g8VD4odXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAncmVzcG9uc2UnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU/OiAnanNvbicsXG4gICAgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cFJlc3BvbnNlPFQ+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgUEFUQ0ggcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIEpTT04gYW5kIHJldHVybnMgaXQuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBib2R5IGFzIGFuIGBPYmplY3RgLlxuICAgKi9cbiAgcGF0Y2godXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zPzoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86ICdib2R5JyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlPzogJ2pzb24nLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPE9iamVjdD47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIFBBVENIIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBKU09OIGFuZCByZXR1cm5zIGl0LlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYm9keSBhcyB0eXBlIGBUYC5cbiAgICovXG4gIHBhdGNoPFQ+KHVybDogc3RyaW5nLCBib2R5OiBhbnl8bnVsbCwgb3B0aW9ucz86IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU/OiAnYm9keScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZT86ICdqc29uJyxcbiAgICB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxUPjtcblxuICAvKipcbiAgICogQ29uc3RydWN0cyBhbiBgT2JzZXJ2YWJsZWAgd2hpY2gsIHdoZW4gc3Vic2NyaWJlZCwgd2lsbCBjYXVzZSB0aGUgY29uZmlndXJlZFxuICAgKiBQQVRDSCByZXF1ZXN0IHRvIGJlIGV4ZWN1dGVkIG9uIHRoZSBzZXJ2ZXIuIFNlZSB0aGUgaW5kaXZpZHVhbCBvdmVybG9hZHMgZm9yXG4gICAqIGRldGFpbHMgb2YgYHBhdGNoKClgJ3MgcmV0dXJuIHR5cGUgYmFzZWQgb24gdGhlIHByb3ZpZGVkIG9wdGlvbnMuXG4gICAqL1xuICBwYXRjaCh1cmw6IHN0cmluZywgYm9keTogYW55fG51bGwsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU/OiBIdHRwT2JzZXJ2ZSxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlPzogJ2FycmF5YnVmZmVyJ3wnYmxvYid8J2pzb24nfCd0ZXh0JyxcbiAgICB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9ID0ge30pOiBPYnNlcnZhYmxlPGFueT4ge1xuICAgIHJldHVybiB0aGlzLnJlcXVlc3Q8YW55PignUEFUQ0gnLCB1cmwsIGFkZEJvZHkob3B0aW9ucywgYm9keSkpO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIFBPU1QgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGFuIGBBcnJheUJ1ZmZlcmAgYW5kIHJldHVybnMgaXQuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBib2R5IGFzIGFuIGBBcnJheUJ1ZmZlcmAuXG4gICAqL1xuICBwb3N0KHVybDogc3RyaW5nLCBib2R5OiBhbnl8bnVsbCwgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86ICdib2R5JyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAnYXJyYXlidWZmZXInLCB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxBcnJheUJ1ZmZlcj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIFBPU1QgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGEgYEJsb2JgIGFuZCByZXR1cm5zIGl0LlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYm9keSBhcyBhIGBCbG9iYC5cbiAgICovXG4gIHBvc3QodXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlPzogJ2JvZHknLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU6ICdibG9iJywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8QmxvYj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIFBPU1QgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIHRleHQgYW5kIHJldHVybnMgaXQuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBib2R5IGFzIGEgYHN0cmluZ2AuXG4gICAqL1xuICBwb3N0KHVybDogc3RyaW5nLCBib2R5OiBhbnl8bnVsbCwgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86ICdib2R5JyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAndGV4dCcsIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPHN0cmluZz47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIFBPU1QgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGFuIGBBcnJheUJ1ZmZlcmAgYW5kIHJldHVybnMgdGhlIGZ1bGwgZXZlbnQgc3RyZWFtLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiBhbGwgYEh0dHBFdmVudGBzIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgQXJyYXlCdWZmZXJgLlxuICAgKi9cbiAgcG9zdCh1cmw6IHN0cmluZywgYm9keTogYW55fG51bGwsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdldmVudHMnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU6ICdhcnJheWJ1ZmZlcicsIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBFdmVudDxBcnJheUJ1ZmZlcj4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBQT1NUIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBhIGBCbG9iYCBhbmQgcmV0dXJucyB0aGUgZnVsbCBldmVudCBzdHJlYW0uXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIGFsbCBgSHR0cEV2ZW50YHMgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBCbG9iYC5cbiAgICovXG4gIHBvc3QodXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAnZXZlbnRzJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAnYmxvYicsIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBFdmVudDxCbG9iPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIFBPU1QgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIHRleHQgYW5kIHJldHVybnMgdGhlIGZ1bGwgZXZlbnQgc3RyZWFtLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiBhbGwgYEh0dHBFdmVudGBzIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgc3RyaW5nYC5cbiAgICovXG4gIHBvc3QodXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAnZXZlbnRzJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAndGV4dCcsIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBFdmVudDxzdHJpbmc+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgUE9TVCByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgSlNPTiBhbmQgcmV0dXJucyB0aGUgZnVsbCBldmVudCBzdHJlYW0uXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIGFsbCBgSHR0cEV2ZW50YHMgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBPYmplY3RgLlxuICAgKi9cbiAgcG9zdCh1cmw6IHN0cmluZywgYm9keTogYW55fG51bGwsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdldmVudHMnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU/OiAnanNvbicsXG4gICAgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cEV2ZW50PE9iamVjdD4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBQT1NUIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBKU09OIGFuZCByZXR1cm5zIHRoZSBmdWxsIGV2ZW50IHN0cmVhbS5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgYWxsIGBIdHRwRXZlbnRgcyBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYFRgLlxuICAgKi9cbiAgcG9zdDxUPih1cmw6IHN0cmluZywgYm9keTogYW55fG51bGwsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdldmVudHMnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU/OiAnanNvbicsXG4gICAgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cEV2ZW50PFQ+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgUE9TVCByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgYW4gYEFycmF5QnVmZmVyYCBhbmQgcmV0dXJucyB0aGUgZnVsbCByZXNwb25zZS5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgdGhlIGBIdHRwUmVzcG9uc2VgIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgQXJyYXlCdWZmZXJgLlxuICAgKi9cbiAgcG9zdCh1cmw6IHN0cmluZywgYm9keTogYW55fG51bGwsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdyZXNwb25zZScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ2FycmF5YnVmZmVyJywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cFJlc3BvbnNlPEFycmF5QnVmZmVyPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIFBPU1QgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGEgYEJsb2JgIGFuZCByZXR1cm5zIHRoZSBmdWxsIHJlc3BvbnNlLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYEh0dHBSZXNwb25zZWAgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBCbG9iYC5cbiAgICovXG4gIHBvc3QodXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAncmVzcG9uc2UnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU6ICdibG9iJywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cFJlc3BvbnNlPEJsb2I+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgUE9TVCByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgdGV4dCBhbmQgcmV0dXJucyB0aGUgZnVsbCByZXNwb25zZS5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgdGhlIGBIdHRwUmVzcG9uc2VgIGZvciB0aGUgcmVxdWVzdCwgd2l0aCBhIGJvZHkgdHlwZSBvZiBgc3RyaW5nYC5cbiAgICovXG4gIHBvc3QodXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAncmVzcG9uc2UnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU6ICd0ZXh0Jywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cFJlc3BvbnNlPHN0cmluZz4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBQT1NUIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBKU09OIGFuZCByZXR1cm5zIHRoZSBmdWxsIHJlc3BvbnNlLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYEh0dHBSZXNwb25zZWAgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBPYmplY3RgLlxuICAgKi9cbiAgcG9zdCh1cmw6IHN0cmluZywgYm9keTogYW55fG51bGwsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdyZXNwb25zZScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZT86ICdqc29uJyxcbiAgICB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwUmVzcG9uc2U8T2JqZWN0Pj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIFBPU1QgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIEpTT04gYW5kIHJldHVybnMgdGhlIGZ1bGwgcmVzcG9uc2UuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBgSHR0cFJlc3BvbnNlYCBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYFRgLlxuICAgKi9cbiAgcG9zdDxUPih1cmw6IHN0cmluZywgYm9keTogYW55fG51bGwsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdyZXNwb25zZScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZT86ICdqc29uJyxcbiAgICB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwUmVzcG9uc2U8VD4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBQT1NUIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBKU09OIGFuZCByZXR1cm5zIGl0LlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYm9keSBhcyBhbiBgT2JqZWN0YC5cbiAgICovXG4gIHBvc3QodXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zPzoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86ICdib2R5JyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlPzogJ2pzb24nLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPE9iamVjdD47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIFBPU1QgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIEpTT04gYW5kIHJldHVybnMgaXQuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBib2R5IGFzIHR5cGUgYFRgLlxuICAgKi9cbiAgcG9zdDxUPih1cmw6IHN0cmluZywgYm9keTogYW55fG51bGwsIG9wdGlvbnM/OiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlPzogJ2JvZHknLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU/OiAnanNvbicsXG4gICAgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8VD47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdHMgYW4gYE9ic2VydmFibGVgIHdoaWNoLCB3aGVuIHN1YnNjcmliZWQsIHdpbGwgY2F1c2UgdGhlIGNvbmZpZ3VyZWRcbiAgICogUE9TVCByZXF1ZXN0IHRvIGJlIGV4ZWN1dGVkIG9uIHRoZSBzZXJ2ZXIuIFNlZSB0aGUgaW5kaXZpZHVhbCBvdmVybG9hZHMgZm9yXG4gICAqIGRldGFpbHMgb2YgYHBvc3QoKWAncyByZXR1cm4gdHlwZSBiYXNlZCBvbiB0aGUgcHJvdmlkZWQgb3B0aW9ucy5cbiAgICovXG4gIHBvc3QodXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlPzogSHR0cE9ic2VydmUsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZT86ICdhcnJheWJ1ZmZlcid8J2Jsb2InfCdqc29uJ3wndGV4dCcsXG4gICAgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSA9IHt9KTogT2JzZXJ2YWJsZTxhbnk+IHtcbiAgICByZXR1cm4gdGhpcy5yZXF1ZXN0PGFueT4oJ1BPU1QnLCB1cmwsIGFkZEJvZHkob3B0aW9ucywgYm9keSkpO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIFBVVCByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgYW4gYEFycmF5QnVmZmVyYCBhbmQgcmV0dXJucyBpdC5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgdGhlIGJvZHkgYXMgYW4gYEFycmF5QnVmZmVyYC5cbiAgICovXG4gIHB1dCh1cmw6IHN0cmluZywgYm9keTogYW55fG51bGwsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU/OiAnYm9keScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ2FycmF5YnVmZmVyJywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8QXJyYXlCdWZmZXI+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBQVVQgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGEgYEJsb2JgIGFuZCByZXR1cm5zIGl0LlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYm9keSBhcyBhIGBCbG9iYC5cbiAgICovXG4gIHB1dCh1cmw6IHN0cmluZywgYm9keTogYW55fG51bGwsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU/OiAnYm9keScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ2Jsb2InLCB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxCbG9iPjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgUFVUIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyB0ZXh0IGFuZCByZXR1cm5zIGl0LlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYm9keSBhcyBhIGBzdHJpbmdgLlxuICAgKi9cbiAgcHV0KHVybDogc3RyaW5nLCBib2R5OiBhbnl8bnVsbCwgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86ICdib2R5JyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAndGV4dCcsIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPHN0cmluZz47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIFBVVCByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgYW4gYEFycmF5QnVmZmVyYCBhbmQgcmV0dXJucyB0aGUgZnVsbCBldmVudCBzdHJlYW0uXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIGFsbCBgSHR0cEV2ZW50YHMgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBBcnJheUJ1ZmZlcmAuXG4gICAqL1xuICBwdXQodXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAnZXZlbnRzJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAnYXJyYXlidWZmZXInLCB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwRXZlbnQ8QXJyYXlCdWZmZXI+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgUFVUIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBhIGBCbG9iYCBhbmQgcmV0dXJucyB0aGUgZnVsbCBldmVudCBzdHJlYW0uXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIGFsbCBgSHR0cEV2ZW50YHMgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBCbG9iYC5cbiAgICovXG4gIHB1dCh1cmw6IHN0cmluZywgYm9keTogYW55fG51bGwsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdldmVudHMnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU6ICdibG9iJywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cEV2ZW50PEJsb2I+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgUFVUIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyB0ZXh0IGFuZCByZXR1cm5zIHRoZSBmdWxsIGV2ZW50IHN0cmVhbS5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgYWxsIGBIdHRwRXZlbnRgcyBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYHN0cmluZ2AuXG4gICAqL1xuICBwdXQodXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAnZXZlbnRzJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlOiAndGV4dCcsIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBFdmVudDxzdHJpbmc+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgUFVUIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBKU09OIGFuZCByZXR1cm5zIHRoZSBmdWxsIGV2ZW50IHN0cmVhbS5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgYWxsIGBIdHRwRXZlbnRgcyBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYE9iamVjdGAuXG4gICAqL1xuICBwdXQodXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAnZXZlbnRzJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlPzogJ2pzb24nLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBFdmVudDxPYmplY3Q+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgUFVUIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBKU09OIGFuZCByZXR1cm5zIHRoZSBmdWxsIGV2ZW50IHN0cmVhbS5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgYWxsIGBIdHRwRXZlbnRgcyBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYFRgLlxuICAgKi9cbiAgcHV0PFQ+KHVybDogc3RyaW5nLCBib2R5OiBhbnl8bnVsbCwgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ2V2ZW50cycsIHJlc3BvbnNlVHlwZT86ICdqc29uJywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cEV2ZW50PFQ+PjtcblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgUFVUIHJlcXVlc3Qgd2hpY2ggaW50ZXJwcmV0cyB0aGUgYm9keSBhcyBhbiBgQXJyYXlCdWZmZXJgIGFuZCByZXR1cm5zIHRoZSBmdWxsIHJlc3BvbnNlLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYEh0dHBSZXNwb25zZWAgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBBcnJheUJ1ZmZlcmAuXG4gICAqL1xuICBwdXQodXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAncmVzcG9uc2UnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU6ICdhcnJheWJ1ZmZlcicsIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBSZXNwb25zZTxBcnJheUJ1ZmZlcj4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBQVVQgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIGEgYEJsb2JgIGFuZCByZXR1cm5zIHRoZSBmdWxsIHJlc3BvbnNlLlxuICAgKlxuICAgKiBAcmV0dXJuIGFuIGBPYnNlcnZhYmxlYCBvZiB0aGUgYEh0dHBSZXNwb25zZWAgZm9yIHRoZSByZXF1ZXN0LCB3aXRoIGEgYm9keSB0eXBlIG9mIGBCbG9iYC5cbiAgICovXG4gIHB1dCh1cmw6IHN0cmluZywgYm9keTogYW55fG51bGwsIG9wdGlvbnM6IHtcbiAgICBoZWFkZXJzPzogSHR0cEhlYWRlcnMgfCB7W2hlYWRlcjogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIG9ic2VydmU6ICdyZXNwb25zZScsXG4gICAgcGFyYW1zPzogSHR0cFBhcmFtc3x7W3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgcmVwb3J0UHJvZ3Jlc3M/OiBib29sZWFuLFxuICAgIHJlc3BvbnNlVHlwZTogJ2Jsb2InLCB3aXRoQ3JlZGVudGlhbHM/OiBib29sZWFuLFxuICB9KTogT2JzZXJ2YWJsZTxIdHRwUmVzcG9uc2U8QmxvYj4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBQVVQgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIHRleHQgYW5kIHJldHVybnMgdGhlIGZ1bGwgcmVzcG9uc2UuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBgSHR0cFJlc3BvbnNlYCBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYHN0cmluZ2AuXG4gICAqL1xuICBwdXQodXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAncmVzcG9uc2UnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU6ICd0ZXh0Jywgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cFJlc3BvbnNlPHN0cmluZz4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBQVVQgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIEpTT04gYW5kIHJldHVybnMgdGhlIGZ1bGwgcmVzcG9uc2UuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBgSHR0cFJlc3BvbnNlYCBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYE9iamVjdGAuXG4gICAqL1xuICBwdXQodXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zOiB7XG4gICAgaGVhZGVycz86IEh0dHBIZWFkZXJzIHwge1toZWFkZXI6IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICBvYnNlcnZlOiAncmVzcG9uc2UnLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU/OiAnanNvbicsXG4gICAgd2l0aENyZWRlbnRpYWxzPzogYm9vbGVhbixcbiAgfSk6IE9ic2VydmFibGU8SHR0cFJlc3BvbnNlPE9iamVjdD4+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBQVVQgcmVxdWVzdCB3aGljaCBpbnRlcnByZXRzIHRoZSBib2R5IGFzIEpTT04gYW5kIHJldHVybnMgdGhlIGZ1bGwgcmVzcG9uc2UuXG4gICAqXG4gICAqIEByZXR1cm4gYW4gYE9ic2VydmFibGVgIG9mIHRoZSBgSHR0cFJlc3BvbnNlYCBmb3IgdGhlIHJlcXVlc3QsIHdpdGggYSBib2R5IHR5cGUgb2YgYFRgLlxuICAgKi9cbiAgcHV0PFQ+KHVybDogc3RyaW5nLCBib2R5OiBhbnl8bnVsbCwgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZTogJ3Jlc3BvbnNlJyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlPzogJ2pzb24nLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPEh0dHBSZXNwb25zZTxUPj47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIFBVVCByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgSlNPTiBhbmQgcmV0dXJucyBpdC5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgdGhlIGJvZHkgYXMgYW4gYE9iamVjdGAuXG4gICAqL1xuICBwdXQodXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zPzoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86ICdib2R5JyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlPzogJ2pzb24nLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPE9iamVjdD47XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIFBVVCByZXF1ZXN0IHdoaWNoIGludGVycHJldHMgdGhlIGJvZHkgYXMgSlNPTiBhbmQgcmV0dXJucyBpdC5cbiAgICpcbiAgICogQHJldHVybiBhbiBgT2JzZXJ2YWJsZWAgb2YgdGhlIGJvZHkgYXMgdHlwZSBgVGAuXG4gICAqL1xuICBwdXQ8VD4odXJsOiBzdHJpbmcsIGJvZHk6IGFueXxudWxsLCBvcHRpb25zPzoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86ICdib2R5JyxcbiAgICBwYXJhbXM/OiBIdHRwUGFyYW1zfHtbcGFyYW06IHN0cmluZ106IHN0cmluZyB8IHN0cmluZ1tdfSxcbiAgICByZXBvcnRQcm9ncmVzcz86IGJvb2xlYW4sXG4gICAgcmVzcG9uc2VUeXBlPzogJ2pzb24nLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0pOiBPYnNlcnZhYmxlPFQ+O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3RzIGFuIGBPYnNlcnZhYmxlYCB3aGljaCwgd2hlbiBzdWJzY3JpYmVkLCB3aWxsIGNhdXNlIHRoZSBjb25maWd1cmVkXG4gICAqIFBVVCByZXF1ZXN0IHRvIGJlIGV4ZWN1dGVkIG9uIHRoZSBzZXJ2ZXIuIFNlZSB0aGUgaW5kaXZpZHVhbCBvdmVybG9hZHMgZm9yXG4gICAqIGRldGFpbHMgb2YgYHB1dCgpYCdzIHJldHVybiB0eXBlIGJhc2VkIG9uIHRoZSBwcm92aWRlZCBvcHRpb25zLlxuICAgKi9cbiAgcHV0KHVybDogc3RyaW5nLCBib2R5OiBhbnl8bnVsbCwgb3B0aW9uczoge1xuICAgIGhlYWRlcnM/OiBIdHRwSGVhZGVycyB8IHtbaGVhZGVyOiBzdHJpbmddOiBzdHJpbmcgfCBzdHJpbmdbXX0sXG4gICAgb2JzZXJ2ZT86IEh0dHBPYnNlcnZlLFxuICAgIHBhcmFtcz86IEh0dHBQYXJhbXN8e1twYXJhbTogc3RyaW5nXTogc3RyaW5nIHwgc3RyaW5nW119LFxuICAgIHJlcG9ydFByb2dyZXNzPzogYm9vbGVhbixcbiAgICByZXNwb25zZVR5cGU/OiAnYXJyYXlidWZmZXInfCdibG9iJ3wnanNvbid8J3RleHQnLFxuICAgIHdpdGhDcmVkZW50aWFscz86IGJvb2xlYW4sXG4gIH0gPSB7fSk6IE9ic2VydmFibGU8YW55PiB7XG4gICAgcmV0dXJuIHRoaXMucmVxdWVzdDxhbnk+KCdQVVQnLCB1cmwsIGFkZEJvZHkob3B0aW9ucywgYm9keSkpO1xuICB9XG59XG4iXX0=