/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { identifierName } from './compile_metadata';
import { Identifiers } from './identifiers';
import * as o from './output/output_ast';
import { convertValueToOutputAst } from './output/value_util';
import { Identifiers as R3 } from './render3/r3_identifiers';
function mapEntry(key, value) {
    return { key, value, quoted: false };
}
export class InjectableCompiler {
    constructor(reflector, alwaysGenerateDef) {
        this.reflector = reflector;
        this.alwaysGenerateDef = alwaysGenerateDef;
        this.tokenInjector = reflector.resolveExternalReference(Identifiers.Injector);
    }
    depsArray(deps, ctx) {
        return deps.map(dep => {
            let token = dep;
            let args = [token];
            let flags = 0 /* Default */;
            if (Array.isArray(dep)) {
                for (let i = 0; i < dep.length; i++) {
                    const v = dep[i];
                    if (v) {
                        if (v.ngMetadataName === 'Optional') {
                            flags |= 8 /* Optional */;
                        }
                        else if (v.ngMetadataName === 'SkipSelf') {
                            flags |= 4 /* SkipSelf */;
                        }
                        else if (v.ngMetadataName === 'Self') {
                            flags |= 2 /* Self */;
                        }
                        else if (v.ngMetadataName === 'Inject') {
                            token = v.token;
                        }
                        else {
                            token = v;
                        }
                    }
                }
            }
            let tokenExpr;
            if (typeof token === 'string') {
                tokenExpr = o.literal(token);
            }
            else if (token === this.tokenInjector) {
                tokenExpr = o.importExpr(Identifiers.INJECTOR);
            }
            else {
                tokenExpr = ctx.importExpr(token);
            }
            if (flags !== 0 /* Default */) {
                args = [tokenExpr, o.literal(flags)];
            }
            else {
                args = [tokenExpr];
            }
            return o.importExpr(Identifiers.inject).callFn(args);
        });
    }
    factoryFor(injectable, ctx) {
        let retValue;
        if (injectable.useExisting) {
            retValue = o.importExpr(Identifiers.inject).callFn([ctx.importExpr(injectable.useExisting)]);
        }
        else if (injectable.useFactory) {
            const deps = injectable.deps || [];
            if (deps.length > 0) {
                retValue = ctx.importExpr(injectable.useFactory).callFn(this.depsArray(deps, ctx));
            }
            else {
                return ctx.importExpr(injectable.useFactory);
            }
        }
        else if (injectable.useValue) {
            retValue = convertValueToOutputAst(ctx, injectable.useValue);
        }
        else {
            const clazz = injectable.useClass || injectable.symbol;
            const depArgs = this.depsArray(this.reflector.parameters(clazz), ctx);
            retValue = new o.InstantiateExpr(ctx.importExpr(clazz), depArgs);
        }
        return o.fn([], [new o.ReturnStatement(retValue)], undefined, undefined, injectable.symbol.name + '_Factory');
    }
    injectableDef(injectable, ctx) {
        let providedIn = o.NULL_EXPR;
        if (injectable.providedIn !== undefined) {
            if (injectable.providedIn === null) {
                providedIn = o.NULL_EXPR;
            }
            else if (typeof injectable.providedIn === 'string') {
                providedIn = o.literal(injectable.providedIn);
            }
            else {
                providedIn = ctx.importExpr(injectable.providedIn);
            }
        }
        const def = [
            mapEntry('factory', this.factoryFor(injectable, ctx)),
            mapEntry('token', ctx.importExpr(injectable.type.reference)),
            mapEntry('providedIn', providedIn),
        ];
        return o.importExpr(R3.ɵɵdefineInjectable).callFn([o.literalMap(def)], undefined, true);
    }
    compile(injectable, ctx) {
        if (this.alwaysGenerateDef || injectable.providedIn !== undefined) {
            const className = identifierName(injectable.type);
            const clazz = new o.ClassStmt(className, null, [
                new o.ClassField('ɵprov', o.INFERRED_TYPE, [o.StmtModifier.Static], this.injectableDef(injectable, ctx)),
            ], [], new o.ClassMethod(null, [], []), []);
            ctx.statements.push(clazz);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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