/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { escapeIdentifier } from '../output/abstract_emitter';
import * as o from '../output/output_ast';
import { Identifiers } from './r3_identifiers';
export function typeWithParameters(type, numParams) {
    if (numParams === 0) {
        return o.expressionType(type);
    }
    const params = [];
    for (let i = 0; i < numParams; i++) {
        params.push(o.DYNAMIC_TYPE);
    }
    return o.expressionType(type, undefined, params);
}
const ANIMATE_SYMBOL_PREFIX = '@';
export function prepareSyntheticPropertyName(name) {
    return `${ANIMATE_SYMBOL_PREFIX}${name}`;
}
export function prepareSyntheticListenerName(name, phase) {
    return `${ANIMATE_SYMBOL_PREFIX}${name}.${phase}`;
}
export function getSafePropertyAccessString(accessor, name) {
    const escapedName = escapeIdentifier(name, false, false);
    return escapedName !== name ? `${accessor}[${escapedName}]` : `${accessor}.${name}`;
}
export function prepareSyntheticListenerFunctionName(name, phase) {
    return `animation_${name}_${phase}`;
}
export function jitOnlyGuardedExpression(expr) {
    return guardedExpression('ngJitMode', expr);
}
export function devOnlyGuardedExpression(expr) {
    return guardedExpression('ngDevMode', expr);
}
export function guardedExpression(guard, expr) {
    const guardExpr = new o.ExternalExpr({ name: guard, moduleName: null });
    const guardNotDefined = new o.BinaryOperatorExpr(o.BinaryOperator.Identical, new o.TypeofExpr(guardExpr), o.literal('undefined'));
    const guardUndefinedOrTrue = new o.BinaryOperatorExpr(o.BinaryOperator.Or, guardNotDefined, guardExpr, /* type */ undefined, 
    /* sourceSpan */ undefined, true);
    return new o.BinaryOperatorExpr(o.BinaryOperator.And, guardUndefinedOrTrue, expr);
}
export function wrapReference(value) {
    const wrapped = new o.WrappedNodeExpr(value);
    return { value: wrapped, type: wrapped };
}
export function refsToArray(refs, shouldForwardDeclare) {
    const values = o.literalArr(refs.map(ref => ref.value));
    return shouldForwardDeclare ? o.fn([], [new o.ReturnStatement(values)]) : values;
}
export function createMayBeForwardRefExpression(expression, forwardRef) {
    return { expression, forwardRef };
}
/**
 * Convert a `MaybeForwardRefExpression` to an `Expression`, possibly wrapping its expression in a
 * `forwardRef()` call.
 *
 * If `MaybeForwardRefExpression.forwardRef` is `ForwardRefHandling.Unwrapped` then the expression
 * was originally wrapped in a `forwardRef()` call to prevent the value from being eagerly evaluated
 * in the code.
 *
 * See `packages/compiler-cli/src/ngtsc/annotations/src/injectable.ts` and
 * `packages/compiler/src/jit_compiler_facade.ts` for more information.
 */
export function convertFromMaybeForwardRefExpression({ expression, forwardRef }) {
    switch (forwardRef) {
        case 0 /* None */:
        case 1 /* Wrapped */:
            return expression;
        case 2 /* Unwrapped */:
            return generateForwardRef(expression);
    }
}
/**
 * Generate an expression that has the given `expr` wrapped in the following form:
 *
 * ```
 * forwardRef(() => expr)
 * ```
 */
export function generateForwardRef(expr) {
    return o.importExpr(Identifiers.forwardRef).callFn([o.fn([], [new o.ReturnStatement(expr)])]);
}
//# sourceMappingURL=data:application/json;base64,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