/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { CompilerConfig } from '../config';
import { ViewEncapsulation } from '../core';
import { DirectiveNormalizer } from '../directive_normalizer';
import { DirectiveResolver } from '../directive_resolver';
import { Lexer } from '../expression_parser/lexer';
import { Parser } from '../expression_parser/parser';
import { I18NHtmlParser } from '../i18n/i18n_html_parser';
import { InjectableCompiler } from '../injectable_compiler';
import { CompileMetadataResolver } from '../metadata_resolver';
import { HtmlParser } from '../ml_parser/html_parser';
import { NgModuleCompiler } from '../ng_module_compiler';
import { NgModuleResolver } from '../ng_module_resolver';
import { TypeScriptEmitter } from '../output/ts_emitter';
import { PipeResolver } from '../pipe_resolver';
import { DomElementSchemaRegistry } from '../schema/dom_element_schema_registry';
import { StyleCompiler } from '../style_compiler';
import { TemplateParser } from '../template_parser/template_parser';
import { syntaxError } from '../util';
import { TypeCheckCompiler } from '../view_compiler/type_check_compiler';
import { ViewCompiler } from '../view_compiler/view_compiler';
import { AotCompiler } from './compiler';
import { StaticReflector } from './static_reflector';
import { StaticSymbolCache } from './static_symbol';
import { StaticSymbolResolver } from './static_symbol_resolver';
import { AotSummaryResolver } from './summary_resolver';
export function createAotUrlResolver(host) {
    return {
        resolve: function (basePath, url) {
            var filePath = host.resourceNameToFileName(url, basePath);
            if (!filePath) {
                throw syntaxError("Couldn't resolve resource " + url + " from " + basePath);
            }
            return filePath;
        }
    };
}
/**
 * Creates a new AotCompiler based on options and a host.
 */
export function createAotCompiler(compilerHost, options, errorCollector) {
    var translations = options.translations || '';
    var urlResolver = createAotUrlResolver(compilerHost);
    var symbolCache = new StaticSymbolCache();
    var summaryResolver = new AotSummaryResolver(compilerHost, symbolCache);
    var symbolResolver = new StaticSymbolResolver(compilerHost, symbolCache, summaryResolver);
    var staticReflector = new StaticReflector(summaryResolver, symbolResolver, [], [], errorCollector);
    var htmlParser;
    if (!!options.enableIvy) {
        // Ivy handles i18n at the compiler level so we must use a regular parser
        htmlParser = new HtmlParser();
    }
    else {
        htmlParser = new I18NHtmlParser(new HtmlParser(), translations, options.i18nFormat, options.missingTranslation, console);
    }
    var config = new CompilerConfig({
        defaultEncapsulation: ViewEncapsulation.Emulated,
        useJit: false,
        missingTranslation: options.missingTranslation,
        preserveWhitespaces: options.preserveWhitespaces,
        strictInjectionParameters: options.strictInjectionParameters,
    });
    var normalizer = new DirectiveNormalizer({ get: function (url) { return compilerHost.loadResource(url); } }, urlResolver, htmlParser, config);
    var expressionParser = new Parser(new Lexer());
    var elementSchemaRegistry = new DomElementSchemaRegistry();
    var tmplParser = new TemplateParser(config, staticReflector, expressionParser, elementSchemaRegistry, htmlParser, console, []);
    var resolver = new CompileMetadataResolver(config, htmlParser, new NgModuleResolver(staticReflector), new DirectiveResolver(staticReflector), new PipeResolver(staticReflector), summaryResolver, elementSchemaRegistry, normalizer, console, symbolCache, staticReflector, errorCollector);
    // TODO(vicb): do not pass options.i18nFormat here
    var viewCompiler = new ViewCompiler(staticReflector);
    var typeCheckCompiler = new TypeCheckCompiler(options, staticReflector);
    var compiler = new AotCompiler(config, options, compilerHost, staticReflector, resolver, tmplParser, new StyleCompiler(urlResolver), viewCompiler, typeCheckCompiler, new NgModuleCompiler(staticReflector), new InjectableCompiler(staticReflector, !!options.enableIvy), new TypeScriptEmitter(), summaryResolver, symbolResolver);
    return { compiler: compiler, reflector: staticReflector };
}
//# sourceMappingURL=data:application/json;base64,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