/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("@angular/compiler/src/ml_parser/tags", ["require", "exports"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    var TagContentType;
    (function (TagContentType) {
        TagContentType[TagContentType["RAW_TEXT"] = 0] = "RAW_TEXT";
        TagContentType[TagContentType["ESCAPABLE_RAW_TEXT"] = 1] = "ESCAPABLE_RAW_TEXT";
        TagContentType[TagContentType["PARSABLE_DATA"] = 2] = "PARSABLE_DATA";
    })(TagContentType = exports.TagContentType || (exports.TagContentType = {}));
    function splitNsName(elementName) {
        if (elementName[0] != ':') {
            return [null, elementName];
        }
        var colonIndex = elementName.indexOf(':', 1);
        if (colonIndex == -1) {
            throw new Error("Unsupported format \"" + elementName + "\" expecting \":namespace:name\"");
        }
        return [elementName.slice(1, colonIndex), elementName.slice(colonIndex + 1)];
    }
    exports.splitNsName = splitNsName;
    // `<ng-container>` tags work the same regardless the namespace
    function isNgContainer(tagName) {
        return splitNsName(tagName)[1] === 'ng-container';
    }
    exports.isNgContainer = isNgContainer;
    // `<ng-content>` tags work the same regardless the namespace
    function isNgContent(tagName) {
        return splitNsName(tagName)[1] === 'ng-content';
    }
    exports.isNgContent = isNgContent;
    // `<ng-template>` tags work the same regardless the namespace
    function isNgTemplate(tagName) {
        return splitNsName(tagName)[1] === 'ng-template';
    }
    exports.isNgTemplate = isNgTemplate;
    function getNsPrefix(fullName) {
        return fullName === null ? null : splitNsName(fullName)[0];
    }
    exports.getNsPrefix = getNsPrefix;
    function mergeNsAndName(prefix, localName) {
        return prefix ? ":" + prefix + ":" + localName : localName;
    }
    exports.mergeNsAndName = mergeNsAndName;
    // see http://www.w3.org/TR/html51/syntax.html#named-character-references
    // see https://html.spec.whatwg.org/multipage/entities.json
    // This list is not exhaustive to keep the compiler footprint low.
    // The `&#123;` / `&#x1ab;` syntax should be used when the named character reference does not
    // exist.
    exports.NAMED_ENTITIES = {
        'Aacute': '\u00C1',
        'aacute': '\u00E1',
        'Acirc': '\u00C2',
        'acirc': '\u00E2',
        'acute': '\u00B4',
        'AElig': '\u00C6',
        'aelig': '\u00E6',
        'Agrave': '\u00C0',
        'agrave': '\u00E0',
        'alefsym': '\u2135',
        'Alpha': '\u0391',
        'alpha': '\u03B1',
        'amp': '&',
        'and': '\u2227',
        'ang': '\u2220',
        'apos': '\u0027',
        'Aring': '\u00C5',
        'aring': '\u00E5',
        'asymp': '\u2248',
        'Atilde': '\u00C3',
        'atilde': '\u00E3',
        'Auml': '\u00C4',
        'auml': '\u00E4',
        'bdquo': '\u201E',
        'Beta': '\u0392',
        'beta': '\u03B2',
        'brvbar': '\u00A6',
        'bull': '\u2022',
        'cap': '\u2229',
        'Ccedil': '\u00C7',
        'ccedil': '\u00E7',
        'cedil': '\u00B8',
        'cent': '\u00A2',
        'Chi': '\u03A7',
        'chi': '\u03C7',
        'circ': '\u02C6',
        'clubs': '\u2663',
        'cong': '\u2245',
        'copy': '\u00A9',
        'crarr': '\u21B5',
        'cup': '\u222A',
        'curren': '\u00A4',
        'dagger': '\u2020',
        'Dagger': '\u2021',
        'darr': '\u2193',
        'dArr': '\u21D3',
        'deg': '\u00B0',
        'Delta': '\u0394',
        'delta': '\u03B4',
        'diams': '\u2666',
        'divide': '\u00F7',
        'Eacute': '\u00C9',
        'eacute': '\u00E9',
        'Ecirc': '\u00CA',
        'ecirc': '\u00EA',
        'Egrave': '\u00C8',
        'egrave': '\u00E8',
        'empty': '\u2205',
        'emsp': '\u2003',
        'ensp': '\u2002',
        'Epsilon': '\u0395',
        'epsilon': '\u03B5',
        'equiv': '\u2261',
        'Eta': '\u0397',
        'eta': '\u03B7',
        'ETH': '\u00D0',
        'eth': '\u00F0',
        'Euml': '\u00CB',
        'euml': '\u00EB',
        'euro': '\u20AC',
        'exist': '\u2203',
        'fnof': '\u0192',
        'forall': '\u2200',
        'frac12': '\u00BD',
        'frac14': '\u00BC',
        'frac34': '\u00BE',
        'frasl': '\u2044',
        'Gamma': '\u0393',
        'gamma': '\u03B3',
        'ge': '\u2265',
        'gt': '>',
        'harr': '\u2194',
        'hArr': '\u21D4',
        'hearts': '\u2665',
        'hellip': '\u2026',
        'Iacute': '\u00CD',
        'iacute': '\u00ED',
        'Icirc': '\u00CE',
        'icirc': '\u00EE',
        'iexcl': '\u00A1',
        'Igrave': '\u00CC',
        'igrave': '\u00EC',
        'image': '\u2111',
        'infin': '\u221E',
        'int': '\u222B',
        'Iota': '\u0399',
        'iota': '\u03B9',
        'iquest': '\u00BF',
        'isin': '\u2208',
        'Iuml': '\u00CF',
        'iuml': '\u00EF',
        'Kappa': '\u039A',
        'kappa': '\u03BA',
        'Lambda': '\u039B',
        'lambda': '\u03BB',
        'lang': '\u27E8',
        'laquo': '\u00AB',
        'larr': '\u2190',
        'lArr': '\u21D0',
        'lceil': '\u2308',
        'ldquo': '\u201C',
        'le': '\u2264',
        'lfloor': '\u230A',
        'lowast': '\u2217',
        'loz': '\u25CA',
        'lrm': '\u200E',
        'lsaquo': '\u2039',
        'lsquo': '\u2018',
        'lt': '<',
        'macr': '\u00AF',
        'mdash': '\u2014',
        'micro': '\u00B5',
        'middot': '\u00B7',
        'minus': '\u2212',
        'Mu': '\u039C',
        'mu': '\u03BC',
        'nabla': '\u2207',
        'nbsp': '\u00A0',
        'ndash': '\u2013',
        'ne': '\u2260',
        'ni': '\u220B',
        'not': '\u00AC',
        'notin': '\u2209',
        'nsub': '\u2284',
        'Ntilde': '\u00D1',
        'ntilde': '\u00F1',
        'Nu': '\u039D',
        'nu': '\u03BD',
        'Oacute': '\u00D3',
        'oacute': '\u00F3',
        'Ocirc': '\u00D4',
        'ocirc': '\u00F4',
        'OElig': '\u0152',
        'oelig': '\u0153',
        'Ograve': '\u00D2',
        'ograve': '\u00F2',
        'oline': '\u203E',
        'Omega': '\u03A9',
        'omega': '\u03C9',
        'Omicron': '\u039F',
        'omicron': '\u03BF',
        'oplus': '\u2295',
        'or': '\u2228',
        'ordf': '\u00AA',
        'ordm': '\u00BA',
        'Oslash': '\u00D8',
        'oslash': '\u00F8',
        'Otilde': '\u00D5',
        'otilde': '\u00F5',
        'otimes': '\u2297',
        'Ouml': '\u00D6',
        'ouml': '\u00F6',
        'para': '\u00B6',
        'permil': '\u2030',
        'perp': '\u22A5',
        'Phi': '\u03A6',
        'phi': '\u03C6',
        'Pi': '\u03A0',
        'pi': '\u03C0',
        'piv': '\u03D6',
        'plusmn': '\u00B1',
        'pound': '\u00A3',
        'prime': '\u2032',
        'Prime': '\u2033',
        'prod': '\u220F',
        'prop': '\u221D',
        'Psi': '\u03A8',
        'psi': '\u03C8',
        'quot': '\u0022',
        'radic': '\u221A',
        'rang': '\u27E9',
        'raquo': '\u00BB',
        'rarr': '\u2192',
        'rArr': '\u21D2',
        'rceil': '\u2309',
        'rdquo': '\u201D',
        'real': '\u211C',
        'reg': '\u00AE',
        'rfloor': '\u230B',
        'Rho': '\u03A1',
        'rho': '\u03C1',
        'rlm': '\u200F',
        'rsaquo': '\u203A',
        'rsquo': '\u2019',
        'sbquo': '\u201A',
        'Scaron': '\u0160',
        'scaron': '\u0161',
        'sdot': '\u22C5',
        'sect': '\u00A7',
        'shy': '\u00AD',
        'Sigma': '\u03A3',
        'sigma': '\u03C3',
        'sigmaf': '\u03C2',
        'sim': '\u223C',
        'spades': '\u2660',
        'sub': '\u2282',
        'sube': '\u2286',
        'sum': '\u2211',
        'sup': '\u2283',
        'sup1': '\u00B9',
        'sup2': '\u00B2',
        'sup3': '\u00B3',
        'supe': '\u2287',
        'szlig': '\u00DF',
        'Tau': '\u03A4',
        'tau': '\u03C4',
        'there4': '\u2234',
        'Theta': '\u0398',
        'theta': '\u03B8',
        'thetasym': '\u03D1',
        'thinsp': '\u2009',
        'THORN': '\u00DE',
        'thorn': '\u00FE',
        'tilde': '\u02DC',
        'times': '\u00D7',
        'trade': '\u2122',
        'Uacute': '\u00DA',
        'uacute': '\u00FA',
        'uarr': '\u2191',
        'uArr': '\u21D1',
        'Ucirc': '\u00DB',
        'ucirc': '\u00FB',
        'Ugrave': '\u00D9',
        'ugrave': '\u00F9',
        'uml': '\u00A8',
        'upsih': '\u03D2',
        'Upsilon': '\u03A5',
        'upsilon': '\u03C5',
        'Uuml': '\u00DC',
        'uuml': '\u00FC',
        'weierp': '\u2118',
        'Xi': '\u039E',
        'xi': '\u03BE',
        'Yacute': '\u00DD',
        'yacute': '\u00FD',
        'yen': '\u00A5',
        'yuml': '\u00FF',
        'Yuml': '\u0178',
        'Zeta': '\u0396',
        'zeta': '\u03B6',
        'zwj': '\u200D',
        'zwnj': '\u200C',
    };
    // The &ngsp; pseudo-entity is denoting a space. see:
    // https://github.com/dart-lang/angular/blob/0bb611387d29d65b5af7f9d2515ab571fd3fbee4/_tests/test/compiler/preserve_whitespace_test.dart
    exports.NGSP_UNICODE = '\uE500';
    exports.NAMED_ENTITIES['ngsp'] = exports.NGSP_UNICODE;
});
//# sourceMappingURL=data:application/json;base64,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