/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { Observable } from 'rxjs';
import { AbstractControl } from '../model';
import { ValidationErrors } from './validators';
/**
 * Base class for control directives.
 *
 * Only used internally in the forms module.
 *
 *
 */
export declare abstract class AbstractControlDirective {
    /**
     * The `FormControl`, `FormGroup`, or `FormArray`
     * that backs this directive. Most properties fall through to that
     * instance.
     */
    abstract readonly control: AbstractControl | null;
    /** The value of the control. */
    readonly value: any;
    /**
     * A control is `valid` when its `status === VALID`.
     *
     * In order to have this status, the control must have passed all its
     * validation checks.
     */
    readonly valid: boolean | null;
    /**
     * A control is `invalid` when its `status === INVALID`.
     *
     * In order to have this status, the control must have failed
     * at least one of its validation checks.
     */
    readonly invalid: boolean | null;
    /**
     * A control is `pending` when its `status === PENDING`.
     *
     * In order to have this status, the control must be in the
     * middle of conducting a validation check.
     */
    readonly pending: boolean | null;
    /**
     * A control is `disabled` when its `status === DISABLED`.
     *
     * Disabled controls are exempt from validation checks and
     * are not included in the aggregate value of their ancestor
     * controls.
     */
    readonly disabled: boolean | null;
    /**
     * A control is `enabled` as long as its `status !== DISABLED`.
     *
     * In other words, it has a status of `VALID`, `INVALID`, or
     * `PENDING`.
     */
    readonly enabled: boolean | null;
    /**
     * Returns any errors generated by failing validation. If there
     * are no errors, it will return null.
     */
    readonly errors: ValidationErrors | null;
    /**
     * A control is `pristine` if the user has not yet changed
     * the value in the UI.
     *
     * Note that programmatic changes to a control's value will
     * *not* mark it dirty.
     */
    readonly pristine: boolean | null;
    /**
     * A control is `dirty` if the user has changed the value
     * in the UI.
     *
     * Note that programmatic changes to a control's value will
     * *not* mark it dirty.
     */
    readonly dirty: boolean | null;
    /**
     * A control is marked `touched` once the user has triggered
     * a `blur` event on it.
     */
    readonly touched: boolean | null;
    readonly status: string | null;
    /**
     * A control is `untouched` if the user has not yet triggered
     * a `blur` event on it.
     */
    readonly untouched: boolean | null;
    /**
     * Emits an event every time the validation status of the control
     * is re-calculated.
     */
    readonly statusChanges: Observable<any> | null;
    /**
     * Emits an event every time the value of the control changes, in
     * the UI or programmatically.
     */
    readonly valueChanges: Observable<any> | null;
    /**
     * Returns an array that represents the path from the top-level form
     * to this control. Each index is the string name of the control on
     * that level.
     */
    readonly path: string[] | null;
    /**
     * Resets the form control. This means by default:
     *
     * * it is marked as `pristine`
     * * it is marked as `untouched`
     * * value is set to null
     *
     * For more information, see `AbstractControl`.
     */
    reset(value?: any): void;
    /**
     * Returns true if the control with the given path has the error specified. Otherwise
     * returns false.
     *
     * If no path is given, it checks for the error on the present control.
     */
    hasError(errorCode: string, path?: string[]): boolean;
    /**
     * Returns error data if the control with the given path has the error specified. Otherwise
     * returns null or undefined.
     *
     * If no path is given, it checks for the error on the present control.
     */
    getError(errorCode: string, path?: string[]): any;
}
