/**
 * @license Angular v12.2.9
 * (c) 2010-2021 Google LLC. https://angular.io/
 * License: MIT
 */

(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define('@angular/localize/init', ['exports'], factory) :
    (global = global || self, factory((global.ng = global.ng || {}, global.ng.localize = global.ng.localize || {}, global.ng.localize.init = {})));
}(this, (function (exports) { 'use strict';

    /**
     * @license
     * Copyright Google LLC All Rights Reserved.
     *
     * Use of this source code is governed by an MIT-style license that can be
     * found in the LICENSE file at https://angular.io/license
     */
    var __globalThis = typeof globalThis !== 'undefined' && globalThis;
    var __window = typeof window !== 'undefined' && window;
    var __self = typeof self !== 'undefined' && typeof WorkerGlobalScope !== 'undefined' &&
        self instanceof WorkerGlobalScope && self;
    var __global = typeof global !== 'undefined' && global;
    // Always use __globalThis if available; this is the spec-defined global variable across all
    // environments.
    // Then fallback to __global first; in Node tests both __global and __window may be defined.
    var _global = __globalThis || __global || __window || __self;

    /**
     * @license
     * Copyright Google LLC All Rights Reserved.
     *
     * Use of this source code is governed by an MIT-style license that can be
     * found in the LICENSE file at https://angular.io/license
     */
    /**
     * Tag a template literal string for localization.
     *
     * For example:
     *
     * ```ts
     * $localize `some string to localize`
     * ```
     *
     * **Providing meaning, description and id**
     *
     * You can optionally specify one or more of `meaning`, `description` and `id` for a localized
     * string by pre-pending it with a colon delimited block of the form:
     *
     * ```ts
     * $localize`:meaning|description@@id:source message text`;
     *
     * $localize`:meaning|:source message text`;
     * $localize`:description:source message text`;
     * $localize`:@@id:source message text`;
     * ```
     *
     * This format is the same as that used for `i18n` markers in Angular templates. See the
     * [Angular 18n guide](guide/i18n-common-prepare#mark-text-for-translations).
     *
     * **Naming placeholders**
     *
     * If the template literal string contains expressions, then the expressions will be automatically
     * associated with placeholder names for you.
     *
     * For example:
     *
     * ```ts
     * $localize `Hi ${name}! There are ${items.length} items.`;
     * ```
     *
     * will generate a message-source of `Hi {$PH}! There are {$PH_1} items`.
     *
     * The recommended practice is to name the placeholder associated with each expression though.
     *
     * Do this by providing the placeholder name wrapped in `:` characters directly after the
     * expression. These placeholder names are stripped out of the rendered localized string.
     *
     * For example, to name the `items.length` expression placeholder `itemCount` you write:
     *
     * ```ts
     * $localize `There are ${items.length}:itemCount: items`;
     * ```
     *
     * **Escaping colon markers**
     *
     * If you need to use a `:` character directly at the start of a tagged string that has no
     * metadata block, or directly after a substitution expression that has no name you must escape
     * the `:` by preceding it with a backslash:
     *
     * For example:
     *
     * ```ts
     * // message has a metadata block so no need to escape colon
     * $localize `:some description::this message starts with a colon (:)`;
     * // no metadata block so the colon must be escaped
     * $localize `\:this message starts with a colon (:)`;
     * ```
     *
     * ```ts
     * // named substitution so no need to escape colon
     * $localize `${label}:label:: ${}`
     * // anonymous substitution so colon must be escaped
     * $localize `${label}\: ${}`
     * ```
     *
     * **Processing localized strings:**
     *
     * There are three scenarios:
     *
     * * **compile-time inlining**: the `$localize` tag is transformed at compile time by a
     * transpiler, removing the tag and replacing the template literal string with a translated
     * literal string from a collection of translations provided to the transpilation tool.
     *
     * * **run-time evaluation**: the `$localize` tag is a run-time function that replaces and
     * reorders the parts (static strings and expressions) of the template literal string with strings
     * from a collection of translations loaded at run-time.
     *
     * * **pass-through evaluation**: the `$localize` tag is a run-time function that simply evaluates
     * the original template literal string without applying any translations to the parts. This
     * version is used during development or where there is no need to translate the localized
     * template literals.
     *
     * @param messageParts a collection of the static parts of the template string.
     * @param expressions a collection of the values of each placeholder in the template string.
     * @returns the translated string, with the `messageParts` and `expressions` interleaved together.
     *
     * @globalApi
     * @publicApi
     */
    var $localize = function (messageParts) {
        var expressions = [];
        for (var _i = 1; _i < arguments.length; _i++) {
            expressions[_i - 1] = arguments[_i];
        }
        if ($localize.translate) {
            // Don't use array expansion here to avoid the compiler adding `__read()` helper unnecessarily.
            var translation = $localize.translate(messageParts, expressions);
            messageParts = translation[0];
            expressions = translation[1];
        }
        var message = stripBlock(messageParts[0], messageParts.raw[0]);
        for (var i = 1; i < messageParts.length; i++) {
            message += expressions[i - 1] + stripBlock(messageParts[i], messageParts.raw[i]);
        }
        return message;
    };
    var BLOCK_MARKER = ':';
    /**
     * Strip a delimited "block" from the start of the `messagePart`, if it is found.
     *
     * If a marker character (:) actually appears in the content at the start of a tagged string or
     * after a substitution expression, where a block has not been provided the character must be
     * escaped with a backslash, `\:`. This function checks for this by looking at the `raw`
     * messagePart, which should still contain the backslash.
     *
     * @param messagePart The cooked message part to process.
     * @param rawMessagePart The raw message part to check.
     * @returns the message part with the placeholder name stripped, if found.
     * @throws an error if the block is unterminated
     */
    function stripBlock(messagePart, rawMessagePart) {
        return rawMessagePart.charAt(0) === BLOCK_MARKER ?
            messagePart.substring(findEndOfBlock(messagePart, rawMessagePart) + 1) :
            messagePart;
    }
    /**
     * Find the end of a "marked block" indicated by the first non-escaped colon.
     *
     * @param cooked The cooked string (where escaped chars have been processed)
     * @param raw The raw string (where escape sequences are still in place)
     *
     * @returns the index of the end of block marker
     * @throws an error if the block is unterminated
     */
    function findEndOfBlock(cooked, raw) {
        /***********************************************************************************************
         * This function is repeated in `src/utils/messages.ts` and the two should be kept in sync.
         * The reason is that this file is marked as having side-effects, and if we import `messages.ts`
         * into it, the whole of `src/utils` will be included in this bundle and none of the functions
         * will be tree shaken.
         ***********************************************************************************************/
        for (var cookedIndex = 1, rawIndex = 1; cookedIndex < cooked.length; cookedIndex++, rawIndex++) {
            if (raw[rawIndex] === '\\') {
                rawIndex++;
            }
            else if (cooked[cookedIndex] === BLOCK_MARKER) {
                return cookedIndex;
            }
        }
        throw new Error("Unterminated $localize metadata block in \"" + raw + "\".");
    }

    /**
     * @license
     * Copyright Google LLC All Rights Reserved.
     *
     * Use of this source code is governed by an MIT-style license that can be
     * found in the LICENSE file at https://angular.io/license
     */

    /**
     * @license
     * Copyright Google LLC All Rights Reserved.
     *
     * Use of this source code is governed by an MIT-style license that can be
     * found in the LICENSE file at https://angular.io/license
     */
    // Attach $localize to the global context, as a side-effect of this module.
    _global.$localize = $localize;

    exports.$localize = $localize;

    Object.defineProperty(exports, '__esModule', { value: true });

})));
//# sourceMappingURL=localize-init.umd.js.map
