import { parseTranslation, translate as _translate } from './utils';
/**
 * Load translations for use by `$localize`, if doing runtime translation.
 *
 * If the `$localize` tagged strings are not going to be replaced at compiled time, it is possible
 * to load a set of translations that will be applied to the `$localize` tagged strings at runtime,
 * in the browser.
 *
 * Loading a new translation will overwrite a previous translation if it has the same `MessageId`.
 *
 * Note that `$localize` messages are only processed once, when the tagged string is first
 * encountered, and does not provide dynamic language changing without refreshing the browser.
 * Loading new translations later in the application life-cycle will not change the translated text
 * of messages that have already been translated.
 *
 * The message IDs and translations are in the same format as that rendered to "simple JSON"
 * translation files when extracting messages. In particular, placeholders in messages are rendered
 * using the `{$PLACEHOLDER_NAME}` syntax. For example the message from the following template:
 *
 * ```html
 * <div i18n>pre<span>inner-pre<b>bold</b>inner-post</span>post</div>
 * ```
 *
 * would have the following form in the `translations` map:
 *
 * ```ts
 * {
 *   "2932901491976224757":
 *      "pre{$START_TAG_SPAN}inner-pre{$START_BOLD_TEXT}bold{$CLOSE_BOLD_TEXT}inner-post{$CLOSE_TAG_SPAN}post"
 * }
 * ```
 *
 * @param translations A map from message ID to translated message.
 *
 * These messages are processed and added to a lookup based on their `MessageId`.
 *
 * @see `clearTranslations()` for removing translations loaded using this function.
 * @see `$localize` for tagging messages as needing to be translated.
 * @publicApi
 */
export function loadTranslations(translations) {
    // Ensure the translate function exists
    if (!$localize.translate) {
        $localize.translate = translate;
    }
    if (!$localize.TRANSLATIONS) {
        $localize.TRANSLATIONS = {};
    }
    Object.keys(translations).forEach(key => {
        $localize.TRANSLATIONS[key] = parseTranslation(translations[key]);
    });
}
/**
 * Remove all translations for `$localize`, if doing runtime translation.
 *
 * All translations that had been loading into memory using `loadTranslations()` will be removed.
 *
 * @see `loadTranslations()` for loading translations at runtime.
 * @see `$localize` for tagging messages as needing to be translated.
 *
 * @publicApi
 */
export function clearTranslations() {
    $localize.translate = undefined;
    $localize.TRANSLATIONS = {};
}
/**
 * Translate the text of the given message, using the loaded translations.
 *
 * This function may reorder (or remove) substitutions as indicated in the matching translation.
 */
export function translate(messageParts, substitutions) {
    try {
        return _translate($localize.TRANSLATIONS, messageParts, substitutions);
    }
    catch (e) {
        console.warn(e.message);
        return [messageParts, substitutions];
    }
}
//# sourceMappingURL=data:application/json;base64,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