/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { Subject } from 'rxjs';
export function mixinErrorState(base) {
    return class extends base {
        constructor(...args) {
            super(...args);
            // This class member exists as an interop with `MatFormFieldControl` which expects
            // a public `stateChanges` observable to emit whenever the form field should be updated.
            // The description is not specifically mentioning the error state, as classes using this
            // mixin can/should emit an event in other cases too.
            /** Emits whenever the component state changes. */
            this.stateChanges = new Subject();
            /** Whether the component is in an error state. */
            this.errorState = false;
        }
        /** Updates the error state based on the provided error state matcher. */
        updateErrorState() {
            const oldState = this.errorState;
            const parent = this._parentFormGroup || this._parentForm;
            const matcher = this.errorStateMatcher || this._defaultErrorStateMatcher;
            const control = this.ngControl ? this.ngControl.control : null;
            const newState = matcher.isErrorState(control, parent);
            if (newState !== oldState) {
                this.errorState = newState;
                this.stateChanges.next();
            }
        }
    };
}
//# sourceMappingURL=data:application/json;base64,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