/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { combineLatest } from 'rxjs';
import { filter, map, scan, startWith, switchMap, take } from 'rxjs/operators';
import { isUrlTree } from '../utils/type_guards';
const INITIAL_VALUE = Symbol('INITIAL_VALUE');
export function prioritizedGuardValue() {
    return switchMap(obs => {
        return combineLatest(obs.map(o => o.pipe(take(1), startWith(INITIAL_VALUE))))
            .pipe(scan((acc, list) => {
            let isPending = false;
            return list.reduce((innerAcc, val, i) => {
                if (innerAcc !== INITIAL_VALUE)
                    return innerAcc;
                // Toggle pending flag if any values haven't been set yet
                if (val === INITIAL_VALUE)
                    isPending = true;
                // Any other return values are only valid if we haven't yet hit a pending
                // call. This guarantees that in the case of a guard at the bottom of the
                // tree that returns a redirect, we will wait for the higher priority
                // guard at the top to finish before performing the redirect.
                if (!isPending) {
                    // Early return when we hit a `false` value as that should always
                    // cancel navigation
                    if (val === false)
                        return val;
                    if (i === list.length - 1 || isUrlTree(val)) {
                        return val;
                    }
                }
                return innerAcc;
            }, acc);
        }, INITIAL_VALUE), filter(item => item !== INITIAL_VALUE), map(item => isUrlTree(item) ? item : item === true), //
        take(1));
    });
}
//# sourceMappingURL=data:application/json;base64,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