"use strict";
var shared_1 = require('./shared');
var url_tree_1 = require('./url_tree');
var collection_1 = require('./utils/collection');
var tree_1 = require('./utils/tree');
function createUrlTree(route, urlTree, commands, queryParams, fragment) {
    if (commands.length === 0) {
        return tree(urlTree._root, urlTree, queryParams, fragment);
    }
    var normalizedCommands = normalizeCommands(commands);
    if (navigateToRoot(normalizedCommands)) {
        return tree(new tree_1.TreeNode(urlTree.root, []), urlTree, queryParams, fragment);
    }
    var startingNode = findStartingNode(normalizedCommands, urlTree, route);
    var updated = normalizedCommands.commands.length > 0 ?
        updateMany(startingNode.children.slice(0), normalizedCommands.commands) :
        [];
    var newRoot = constructNewTree(urlTree._root, startingNode, updated);
    return tree(newRoot, urlTree, queryParams, fragment);
}
exports.createUrlTree = createUrlTree;
function tree(root, urlTree, queryParams, fragment) {
    var q = queryParams ? stringify(queryParams) : urlTree.queryParams;
    var f = fragment ? fragment : urlTree.fragment;
    return new url_tree_1.UrlTree(root, q, f);
}
function navigateToRoot(normalizedChange) {
    return normalizedChange.isAbsolute && normalizedChange.commands.length === 1 &&
        normalizedChange.commands[0] == '/';
}
var NormalizedNavigationCommands = (function () {
    function NormalizedNavigationCommands(isAbsolute, numberOfDoubleDots, commands) {
        this.isAbsolute = isAbsolute;
        this.numberOfDoubleDots = numberOfDoubleDots;
        this.commands = commands;
    }
    return NormalizedNavigationCommands;
}());
function normalizeCommands(commands) {
    if ((typeof commands[0] === 'string') && commands.length === 1 && commands[0] == '/') {
        return new NormalizedNavigationCommands(true, 0, commands);
    }
    var numberOfDoubleDots = 0;
    var isAbsolute = false;
    var res = [];
    for (var i = 0; i < commands.length; ++i) {
        var c = commands[i];
        if (!(typeof c === 'string')) {
            res.push(c);
            continue;
        }
        var parts = c.split('/');
        for (var j = 0; j < parts.length; ++j) {
            var cc = parts[j];
            if (i == 0) {
                if (j == 0 && cc == '.') {
                }
                else if (j == 0 && cc == '') {
                    isAbsolute = true;
                }
                else if (cc == '..') {
                    numberOfDoubleDots++;
                }
                else if (cc != '') {
                    res.push(cc);
                }
            }
            else {
                if (cc != '') {
                    res.push(cc);
                }
            }
        }
    }
    return new NormalizedNavigationCommands(isAbsolute, numberOfDoubleDots, res);
}
function findStartingNode(normalizedChange, urlTree, route) {
    if (normalizedChange.isAbsolute) {
        return urlTree._root;
    }
    else {
        var urlSegment = findUrlSegment(route, urlTree, normalizedChange.numberOfDoubleDots);
        return findMatchingNode(urlSegment, urlTree._root);
    }
}
function findUrlSegment(route, urlTree, numberOfDoubleDots) {
    var urlSegment = route.snapshot._lastUrlSegment;
    var path = urlTree.pathFromRoot(urlSegment);
    if (path.length <= numberOfDoubleDots) {
        throw new Error('Invalid number of \'../\'');
    }
    return path[path.length - 1 - numberOfDoubleDots];
}
function findMatchingNode(segment, node) {
    if (node.value === segment)
        return node;
    for (var _i = 0, _a = node.children; _i < _a.length; _i++) {
        var c = _a[_i];
        var r = findMatchingNode(segment, c);
        if (r)
            return r;
    }
    throw new Error("Cannot find url segment '" + segment + "'");
}
function constructNewTree(node, original, updated) {
    if (node === original) {
        return new tree_1.TreeNode(node.value, updated);
    }
    else {
        return new tree_1.TreeNode(node.value, node.children.map(function (c) { return constructNewTree(c, original, updated); }));
    }
}
function updateMany(nodes, commands) {
    var outlet = getOutlet(commands);
    var nodesInRightOutlet = nodes.filter(function (c) { return c.value.outlet === outlet; });
    if (nodesInRightOutlet.length > 0) {
        var nodeRightOutlet = nodesInRightOutlet[0];
        nodes[nodes.indexOf(nodeRightOutlet)] = update(nodeRightOutlet, commands);
    }
    else {
        nodes.push(update(null, commands));
    }
    return nodes;
}
function getPath(commands) {
    if (!(typeof commands[0] === 'string'))
        return commands[0];
    var parts = commands[0].toString().split(':');
    return parts.length > 1 ? parts[1] : commands[0];
}
function getOutlet(commands) {
    if (!(typeof commands[0] === 'string'))
        return shared_1.PRIMARY_OUTLET;
    var parts = commands[0].toString().split(':');
    return parts.length > 1 ? parts[0] : shared_1.PRIMARY_OUTLET;
}
function update(node, commands) {
    var rest = commands.slice(1);
    var next = rest.length === 0 ? null : rest[0];
    var outlet = getOutlet(commands);
    var path = getPath(commands);
    if (!node && !(typeof next === 'object')) {
        var urlSegment = new url_tree_1.UrlSegment(path, {}, outlet);
        var children = rest.length === 0 ? [] : [update(null, rest)];
        return new tree_1.TreeNode(urlSegment, children);
    }
    else if (!node && typeof next === 'object') {
        var urlSegment = new url_tree_1.UrlSegment(path, stringify(next), outlet);
        return recurse(urlSegment, node, rest.slice(1));
    }
    else if (node && outlet !== node.value.outlet) {
        return node;
    }
    else if (node && typeof path === 'object') {
        var newSegment = new url_tree_1.UrlSegment(node.value.path, stringify(path), node.value.outlet);
        return recurse(newSegment, node, rest);
    }
    else if (node && typeof next === 'object' && compare(path, stringify(next), node.value)) {
        return recurse(node.value, node, rest.slice(1));
    }
    else if (node && typeof next === 'object') {
        var urlSegment = new url_tree_1.UrlSegment(path, stringify(next), outlet);
        return recurse(urlSegment, node, rest.slice(1));
    }
    else if (node && compare(path, {}, node.value)) {
        return recurse(node.value, node, rest);
    }
    else {
        var urlSegment = new url_tree_1.UrlSegment(path, {}, outlet);
        return recurse(urlSegment, node, rest);
    }
}
function stringify(params) {
    var res = {};
    collection_1.forEach(params, function (v, k) { return res[k] = "" + v; });
    return res;
}
function compare(path, params, segment) {
    return path == segment.path && collection_1.shallowEqual(params, segment.parameters);
}
function recurse(urlSegment, node, rest) {
    if (rest.length === 0) {
        return new tree_1.TreeNode(urlSegment, []);
    }
    var children = node ? node.children.slice(0) : [];
    return new tree_1.TreeNode(urlSegment, updateMany(children, rest));
}
//# sourceMappingURL=data:application/json;base64,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