import { Observable } from 'rxjs/Observable';
import { of } from 'rxjs/observable/of';
import { PRIMARY_OUTLET } from './shared';
import { UrlSegment, UrlTree } from './url_tree';
import { first } from './utils/collection';
import { TreeNode } from './utils/tree';
class NoMatch {
}
class GlobalRedirect {
    constructor(segments) {
        this.segments = segments;
    }
}
export function applyRedirects(urlTree, config) {
    try {
        const transformedChildren = urlTree._root.children.map(c => applyNode(config, c));
        return createUrlTree(urlTree, transformedChildren);
    }
    catch (e) {
        if (e instanceof GlobalRedirect) {
            return createUrlTree(urlTree, [constructNodes(e.segments, [], [])]);
        }
        else if (e instanceof NoMatch) {
            return new Observable(obs => obs.error(new Error('Cannot match any routes')));
        }
        else {
            return new Observable(obs => obs.error(e));
        }
    }
}
function createUrlTree(urlTree, children) {
    const transformedRoot = new TreeNode(urlTree.root, children);
    return of(new UrlTree(transformedRoot, urlTree.queryParams, urlTree.fragment));
}
function applyNode(config, url) {
    for (let r of config) {
        try {
            return matchNode(config, r, url);
        }
        catch (e) {
            if (!(e instanceof NoMatch))
                throw e;
        }
    }
    throw new NoMatch();
}
function matchNode(config, route, url) {
    if (!route.path)
        throw new NoMatch();
    if ((route.outlet ? route.outlet : PRIMARY_OUTLET) !== url.value.outlet) {
        throw new NoMatch();
    }
    if (route.path === '**') {
        const newSegments = applyRedirectCommands([], route.redirectTo, {});
        return constructNodes(newSegments, [], []);
    }
    const m = match(route, url);
    if (!m)
        throw new NoMatch();
    const { consumedUrlSegments, lastSegment, lastParent, positionalParamSegments } = m;
    const newSegments = applyRedirectCommands(consumedUrlSegments, route.redirectTo, positionalParamSegments);
    const childConfig = route.children ? route.children : [];
    const transformedChildren = lastSegment.children.map(c => applyNode(childConfig, c));
    const secondarySubtrees = lastParent ? lastParent.children.slice(1) : [];
    const transformedSecondarySubtrees = secondarySubtrees.map(c => applyNode(config, c));
    return constructNodes(newSegments, transformedChildren, transformedSecondarySubtrees);
}
export function match(route, url) {
    const path = route.path.startsWith('/') ? route.path.substring(1) : route.path;
    const parts = path.split('/');
    const positionalParamSegments = {};
    const consumedUrlSegments = [];
    let lastParent = null;
    let lastSegment = null;
    let current = url;
    for (let i = 0; i < parts.length; ++i) {
        if (!current)
            return null;
        const p = parts[i];
        const isLastSegment = i === parts.length - 1;
        const isLastParent = i === parts.length - 2;
        const isPosParam = p.startsWith(':');
        if (!isPosParam && p != current.value.path)
            return null;
        if (isLastSegment) {
            lastSegment = current;
        }
        if (isLastParent) {
            lastParent = current;
        }
        if (isPosParam) {
            positionalParamSegments[p.substring(1)] = current.value;
        }
        consumedUrlSegments.push(current.value);
        current = first(current.children);
    }
    if (!lastSegment)
        throw 'Cannot be reached';
    return { consumedUrlSegments, lastSegment, lastParent, positionalParamSegments };
}
function constructNodes(segments, children, secondary) {
    let prevChildren = children;
    for (let i = segments.length - 1; i >= 0; --i) {
        if (i === segments.length - 2) {
            prevChildren = [new TreeNode(segments[i], prevChildren.concat(secondary))];
        }
        else {
            prevChildren = [new TreeNode(segments[i], prevChildren)];
        }
    }
    return prevChildren[0];
}
function applyRedirectCommands(segments, redirectTo, posParams) {
    if (!redirectTo)
        return segments;
    if (redirectTo.startsWith('/')) {
        const parts = redirectTo.substring(1).split('/');
        throw new GlobalRedirect(createSegments(redirectTo, parts, segments, posParams));
    }
    else {
        return createSegments(redirectTo, redirectTo.split('/'), segments, posParams);
    }
}
function createSegments(redirectTo, parts, segments, posParams) {
    return parts.map(p => p.startsWith(':') ? findPosParamSegment(p, posParams, redirectTo) :
        findOrCreateSegment(p, segments));
}
function findPosParamSegment(part, posParams, redirectTo) {
    const paramName = part.substring(1);
    const pos = posParams[paramName];
    if (!pos)
        throw new Error(`Cannot redirect to '${redirectTo}'. Cannot find '${part}'.`);
    return pos;
}
function findOrCreateSegment(part, segments) {
    const matchingIndex = segments.findIndex(s => s.path === part);
    if (matchingIndex > -1) {
        const r = segments[matchingIndex];
        segments.splice(matchingIndex);
        return r;
    }
    else {
        return new UrlSegment(part, {}, PRIMARY_OUTLET);
    }
}
//# sourceMappingURL=data:application/json;base64,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